package cn.com.duiba.activity.center.api.rank;

import cn.com.duiba.wolf.utils.DateUtils;
import org.springframework.data.redis.core.RedisTemplate;

import java.util.Calendar;
import java.util.Date;

/**
 * 游戏排行榜工厂方法
 * Created by hww on 2018/8/9 下午4:36.
 */
public class NGameRankFactory {

    /**
     * 总排名redis缓存时间
     */
    private static final int EXPIRE_DAYS = 10;
    /**
     * 多游戏总排名redis缓存时间
     */
    private static final int MULTI_NGAME_EXPIRE_DAYS = 30;
    /**
     * 奖项总排名redis缓存时间
     */
    private static final int OPTION_RANK_EXPIRE_DAYS = 30;
    /**
     *  排名限制数
     */
    private static final int RANK_LIMIT_NUM = 5000;

    /**
     * 多游戏排行榜排名限制数
     */
    private static final int MULTI_NGAME_RANK_LIMIT_NUM = 20000;

    /**
     * 每日排名redis缓存时间
     */
    private static final int DAILY_RANK_EXPIRE_DAYS = 3;

    /**
     * 每周排名redis缓存时间
     */
    private static final int WEEKLY_RANK_EXPIRE_DAYS = 14;

    /**
     * 获取总排行榜
     */
    public static NGameRank generateTotalRank(long gameId, boolean isDesc, RedisTemplate<String, String> redisClient) {
        return new NGameRank(
                K024 + gameId,
                K025 + gameId,
                K026 + gameId,
                EXPIRE_DAYS,
                isDesc,
                RANK_LIMIT_NUM,
                redisClient
        );
    }

    /**
     * 获取当日排行榜
     */
    public static NGameRank generateDailyRank(long gameId, boolean isDesc, RedisTemplate<String, String> redisClient) {
        String dailyMark = DateUtils.getDayStr(new Date()) + "-" + gameId;
        return new NGameRank (
                K041 + dailyMark,
                K042 + dailyMark,
                K043 + dailyMark,
                DAILY_RANK_EXPIRE_DAYS,
                isDesc,
                RANK_LIMIT_NUM,
                redisClient
        );
    }

    /**
     * 获取昨日排行榜
     */
    public static NGameRank generateYesterdayRank(long gameId, boolean isDesc, RedisTemplate<String, String> redisClient) {
        String dailyMark = DateUtils.getDayStr(DateUtils.daysAddOrSub(new Date(), -1)) + "-" + gameId;
        return new NGameRank (
                K041 + dailyMark,
                K042 + dailyMark,
                K043 + dailyMark,
                DAILY_RANK_EXPIRE_DAYS,
                isDesc,
                RANK_LIMIT_NUM,
                redisClient
        );
    }

    /**
     * 获取多游戏总排行榜
     */
    public static NGameRank generateMultiNGameTotalRank(Long appId, Long rankId, boolean isDesc, RedisTemplate<String, String> redisClient) {
        String key = rankId + "-" + appId;
        return new NGameRank (
                K047 + key,
                K048 + key,
                K049 + key,
                MULTI_NGAME_EXPIRE_DAYS,
                isDesc,
                MULTI_NGAME_RANK_LIMIT_NUM,
                redisClient
        );
    }

    /**
     * 获取奖项总排行榜
     */
    public static NGameRank generateOptionNumRank(Long appId, Long rankId, boolean distinctByApp, boolean isDesc, RedisTemplate<String, String> redisClient) {
        String key = String.valueOf(rankId);
        if(distinctByApp){
            key = key + "-" + appId;
        }
        return new NGameRank (
                K056 + key,
                K057 + key,
                K058 + key,
                OPTION_RANK_EXPIRE_DAYS,
                isDesc,
                RANK_LIMIT_NUM,
                redisClient
        );
    }

    /**
     * 获取本周排行榜
     */
    public static NGameRank generateWeeklyRank(long gameId, boolean isDesc, RedisTemplate<String, String> redisClient) {
        String weeklyMark = getWeekMark(new Date()) + "-" + gameId;
        return new NGameRank (
                K102 + weeklyMark,
                K103 + weeklyMark,
                K104 + weeklyMark,
                WEEKLY_RANK_EXPIRE_DAYS,
                isDesc,
                RANK_LIMIT_NUM,
                redisClient
        );
    }

    /**
     * 获取上周排行榜
     */
    public static NGameRank generateLastWeekRank(long gameId, boolean isDesc, RedisTemplate<String, String> redisClient) {
        String weeklyMark = getWeekMark(DateUtils.daysAddOrSub(new Date(), -7)) + "-" + gameId;
        return new NGameRank (
                K102 + weeklyMark,
                K103 + weeklyMark,
                K104 + weeklyMark,
                WEEKLY_RANK_EXPIRE_DAYS,
                isDesc,
                RANK_LIMIT_NUM,
                redisClient
        );
    }

    private static int getWeekMark(Date date){
        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.setTime(date);
        return calendar.get(Calendar.WEEK_OF_YEAR);
    }

    //--  redis key 迁移自活动web系统 ---
    //游戏总排行榜相关key
    private static String K024 = "AAW_K024_";
    private static String K025 = "AAW_K025_";
    private static String K026 = "AAW_K026_";

    //新游戏每日排行榜相关key
    private static String K041 = "AAW_K041_";
    private static String K042 = "AAW_K042_";
    private static String K043 = "AAW_K043_";

    //多游戏总排行榜相关key
    private static String K047 = "AAW_K047_";
    private static String K048 = "AAW_K048_";
    private static String K049 = "AAW_K049_";

    //奖项排名中总排行榜相关key
    private static String K056 = "AAW_K056_";
    private static String K057 = "AAW_K057_";
    private static String K058 = "AAW_K058_";

    //新游戏每周排行榜相关key
    private static String K102 = "AAW_K102_";
    private static String K103 = "AAW_K103_";
    private static String K104 = "AAW_K104_";

    //--  redis key 迁移自活动web系统 ---

}
