package cn.com.duiba.apollo.ui;

import cn.com.duiba.apollo.client.dto.*;
import cn.com.duiba.apollo.client.remoteservice.RemoteResourceInstanceService;
import cn.com.duiba.apollo.client.remoteservice.RemoteResourceReferenceService;
import cn.com.duiba.apollo.client.remoteservice.RemoteResourceTemplateService;
import cn.com.duiba.apollo.client.service.ApolloAccessTokenService;
import cn.com.duiba.apollo.client.service.resource.ResourceReferenceClientService;
import cn.com.duiba.apollo.client.utils.BeanUtils;
import cn.com.duiba.apollo.client.utils.PasswordUtils;

import cn.com.duiba.apollo.ui.vo.ResourceReferenceVO;
import cn.com.duiba.apollo.ui.vo.ReferenceBO;
import cn.com.duiba.boot.exception.BizException;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import javax.annotation.Resource;
import java.util.*;

/**
 * @author liuyao
 */
public class ApolloResourceService {

    @Resource
    private RemoteResourceInstanceService remoteResourceInstanceService;
    @Resource
    private ApolloAccessTokenService apolloAccessTokenService;
    @Resource
    private RemoteResourceReferenceService remoteResourceReferenceService;
    @Resource
    private ResourceReferenceClientService resourceReferenceClientService;
    @Resource
    private RemoteResourceTemplateService remoteResourceTemplateService;

    public List<ResourceReferenceVO> findReferenceByBizId(String bizId) throws BizException {
        String accessToken = apolloAccessTokenService.getAccessToken();
        List<ResourceReferenceDTO> references = remoteResourceReferenceService.findResourceReferenceByBizId(accessToken,bizId);
        if(references.isEmpty()){
            return Collections.emptyList();
        }
        List<Long> instanceIds = Lists.newArrayList();
        List<Long> templateIds = Lists.newArrayList();
        List<ResourceReferenceDTO> sources = Lists.newArrayList();

        for(ResourceReferenceDTO reference:references){
            sources.add(reference);
            instanceIds.add(reference.getInstanceId());
            templateIds.add(reference.getTemplateId());
        }

        List<ResourceTypeDTO> resourceTypes = remoteResourceInstanceService.findAllResourceTypes();
        Map<Long,ResourceTypeDTO> resourceTypeMap = BeanUtils.mapByKey("id",resourceTypes);

        List<ResourceInstanceDTO> instances = remoteResourceInstanceService.findInstancesByIds(instanceIds);
        Map<Long,ResourceInstanceDTO> instanceMap = Maps.newHashMap();
        for(ResourceInstanceDTO instance:instances){
            instanceMap.put(instance.getId(),instance);
        }
        Map<Long, ResourceTemplateDTO> templateMap = Maps.newHashMap();
        if(!templateIds.isEmpty()){
            List<ResourceTemplateDTO> templates = remoteResourceTemplateService.findAllTemplateByIds(templateIds);
            for(ResourceTemplateDTO template:templates){
                templateMap.put(template.getId(),template);
            }
        }

        List<ResourceReferenceVO> result = Lists.newArrayList();
        for(ResourceReferenceDTO reference:sources){
            ResourceReferenceVO vo = new ResourceReferenceVO();
            vo.setId(reference.getId());
            vo.setBizId(reference.getBizId());
            vo.setInstanceId(reference.getInstanceId());
            vo.setTemplateId(reference.getTemplateId());
            vo.setActive(reference.getActive());
            vo.setInstanceAlias(reference.getInstanceAlias());
            vo.setOperatorId(reference.getOperator());
            ResourceInstanceDTO instance = instanceMap.get(reference.getInstanceId());
            if(Objects.nonNull(instance)){
                vo.setInstanceKey(instance.getKey());
                vo.setInstanceName(instance.getName());
                vo.setTypeId(instance.getTypeId());
                ResourceTypeDTO resourceType = resourceTypeMap.get(instance.getTypeId());
                vo.setTypeName(resourceType.getName());
            }

            if(Objects.nonNull(reference.getTemplateId())){
                ResourceTemplateDTO template = templateMap.get(reference.getTemplateId());
                vo.setTemplateName(template.getName());
            }
            vo.setAuthorized(reference.getAuthorized());
            result.add(vo);
        }
        return result;
    }

    public List<ResourceInstanceDTO> findApolloInstances(Collection<Long> ids){
        return remoteResourceInstanceService.findInstancesByIds(Lists.newArrayList(ids));
    }

    public ResourceInstanceDTO findApolloInstance(Long instanceId){
        return remoteResourceInstanceService.findOneInstance(instanceId);
    }

    public Properties findApolloInstanceConfig(Long instanceId) throws BizException {
        List<KeyValueDTO> keyValues = remoteResourceInstanceService.findInstanceNativeConfig(instanceId);
        Properties instanceConfigs = new Properties();
        for(KeyValueDTO kv:keyValues){
            instanceConfigs.put(kv.getKey(), kv.getValue());
        }
        return instanceConfigs;
    }

    public ReferenceBO findReferenceItems(String bizId, Long instanceId) throws BizException {

        String accessToken = apolloAccessTokenService.getAccessToken();

        ResourceReferenceDTO reference = remoteResourceReferenceService.findReferenceByBizIdAndInstanceId(accessToken,bizId,instanceId);
        if(reference==null){
            return null;
        }
        ReferenceBO result = new ReferenceBO();
        result.setReferenceId(reference.getId());
        result.setAuthorized(reference.getAuthorized());
        result.setActive(reference.getActive());
        if(!reference.getAuthorized() || !reference.getActive()){
            return result;
        }
        ReferenceViewDTO referenceView = remoteResourceReferenceService.findReferenceView(accessToken,reference.getId());
        result.setReferenceView(referenceView);
        return result;
    }

    public String charkInstanceType(Long instanceId,Set<String> supportTypes) throws BizException {
        ResourceInstanceDTO instance = remoteResourceInstanceService.findOneInstance(instanceId);
        Objects.requireNonNull(instance);
        return charkInstanceType(instance,supportTypes);
    }

    public String charkInstanceType(ResourceInstanceDTO instance,Set<String> supportTypes) throws BizException {
        ResourceTypeDTO resourceType = remoteResourceInstanceService.findTypeById(instance.getTypeId());
        Objects.requireNonNull(resourceType);
        if(!supportTypes.contains(resourceType.getKey())){
            throw new BizException("不支持设置["+resourceType.getName()+"]类型的资源实例");
        }
        return resourceType.getKey();
    }

    /**
     * 构建与apollo实例之间的联系
     */
    public void buildApolloInstanceRelation(ResourceReferenceDTO params) throws BizException {

        String accessToken = apolloAccessTokenService.getAccessToken();
        ResourceReferenceDTO reference = remoteResourceReferenceService.findReferenceByBizIdAndInstanceId(accessToken,params.getBizId(),params.getInstanceId());
        if(Objects.nonNull(reference)){
            return;
        }
        remoteResourceReferenceService.applyResourceInstance(accessToken,params);
    }

    public void deleteReference(String bizId,Long instanceId,Long operator) throws BizException {
        String accessToken = apolloAccessTokenService.getAccessToken();
        ResourceReferenceDTO reference = remoteResourceReferenceService.findReferenceByBizIdAndInstanceId(accessToken,bizId,instanceId);
        if(Objects.isNull(reference)){
            return;
        }
        if(reference.getActive()){
            remoteResourceReferenceService.stopReference(accessToken,reference.getId());
        }
        remoteResourceReferenceService.removeResourceReference(accessToken,operator,reference.getId());
    }

    public ResourceReferenceDTO findResourceReference(String bizId,Long instanceId) throws BizException {
        String accessToken = apolloAccessTokenService.getAccessToken();
        return remoteResourceReferenceService.findReferenceByBizIdAndInstanceId(accessToken,bizId,instanceId);
    }


    /**
     * 获取canal实例完整的配置
     */
    public Properties findCanalInstanceConfig(String bizId,Long instanceId) throws BizException{
        ResourceReferenceDTO reference = findResourceReference(bizId,instanceId);
        if(!reference.getActive()){
            throw new BizException("Apollo实例配置未激活");
        }
        ReferenceReleaseDTO referenceRelease = resourceReferenceClientService.findReferenceProperties(reference.getId());
        Properties properties = referenceRelease.getReleaseProperties();

        Properties resultProperties = new Properties();
        for(String key:properties.stringPropertyNames()){
            String value = properties.getProperty(key);
            resultProperties.put(key, PasswordUtils.decryptValue(value));
        }
        return resultProperties;
    }

}
