define([
    "angular",
    "/apollo/js/apollo-common.js"
], function (angular) {

    const apollo = angular.module("apollo.resource.directive",["apollo.ui.common"]);

    apollo.directive('apolloReference', ApolloReference);
    apollo.directive('apolloReferenceList', ApolloReferenceList);

    apollo.controller("ApolloResourceListController",ApolloResourceListController);
    apollo.controller("ApolloResourcePanelController",ApolloResourcePanelController);
    apollo.controller('ApolloResourceTemplatePreviewController', ApolloResourceTemplatePreviewController);
    apollo.controller("ApolloInstancesModelController",ApolloInstancesModelController);
    apollo.controller("StartReferenceController",StartReferenceController);//启用资源
    apollo.controller("ReleaseReferenceModalController",ReleaseReferenceModalController);//发布资源

    function ApolloReference(){
        return {
            restrict: 'E',
            templateUrl: '/apollo/resource-panel.html',
            transclude: true,
            replace: true,
            controller:"ApolloResourcePanelController",
            scope: {
                resource:'=resource',
                count:'=count'
            }
        }
    }

    function ApolloReferenceList(){
        return {
            restrict: 'E',
            templateUrl: '/apollo/resource-list.html',
            transclude: true,
            replace: true,
            controller:"ApolloResourceListController",
            scope: {
                bizId: '=bizId'
            }
        }
    }

    function ApolloResourceListController($scope,UiModel,$http){

        $scope.resources = [];
        $scope.loaded = false;

        $scope.$on("refresh_resource", function () {
            refreshAllResources();
        });
        refreshAllResources();

        function refreshAllResources(){

            if(!$scope.bizId){
                return;
            }
            const p = UiModel.loading("资源列表加载中...");
            $http.get($rootPath+"/apolloApi/findAllReferenceByBizId",{params:{bizId:$scope.bizId}}).success(function(resources){
                $scope.resources = resources;
                $scope.loaded = true;
                p.close();
            }).error(function(){
                p.close();
            })
        }
    }

    function ApolloResourcePanelController($scope, $http, UiModel, $uibModal, ApolloService,ApolloResourceService,$rootScope) {

        const view_type = {
            TEXT: 'text',
            TABLE: 'table'
        };

        $scope.aceConfig = {
            $blockScrolling: Infinity,
            showPrintMargin: false,
            theme: 'eclipse',
            mode: "properties",
            onLoad: function (_editor) {
                _editor.$blockScrolling = Infinity;
                _editor.setOptions({
                    fontSize: 13,
                    minLines: 28,
                    maxLines: 28
                })
            }
        };
        $scope.initialized = false;
        $scope.resource.viewItems = [];//引用覆写配置
        $scope.resource.searchKey = "";
        $scope.resource.instance = {};//资源实例
        $scope.resource.instance.searchKey = "";
        $scope.resource.instance.viewItems = [];//资源实例配置

        $scope.initResource = initResource;
        $scope.switchView = switchView;
        $scope.publish = publish;
        $scope.editItem = editItem;
        $scope.coverItem = coverItem;
        $scope.preDeleteItem = preDeleteItem;
        $scope.showText = showText;
        $scope.startReference = startReference;
        $scope.removeResourceReference = removeResourceReference;
        $scope.stopReference = stopReference;


        if($scope.count<=2){
            initResource();
        }

        function initResource(viewType) {
            if (!viewType) {
                switchView(view_type.TABLE);
            } else if (viewType === view_type.TABLE) {
                $scope.resource.viewType = view_type.TABLE;
            }
            refreshResource();
        }

        /**
         * 指令参数入口 {bizId,instanceId}
         */
        function refreshResource() {
            $http.get($rootPath+"/apolloApi/findReferenceItems", {params: {bizId: $scope.resource.bizId,instanceId:$scope.resource.instanceId}}).success(function (view) {

                const {active,authorized,adminNames,referenceView} = view;
                $scope.resource.id = view.referenceId;
                $scope.resource.authorized = authorized;
                $scope.resource.active = active;
                if(!active){
                    return;
                }
                $scope.initialized = true;
                $scope.resource.itemModifiedCnt = referenceView.modifiedItemCnt;
                referenceView.items.each(function (item) {
                    item.adminName = adminNames[item.operator];
                });
                $scope.resource.viewItems = referenceView.items;
                $scope.resource.initialized = true;
                $scope.resource.notRelease = referenceView.notRelease;
                $scope.resource.instance.viewItems = referenceView.instanceItems;

                const keys = new Set();
                $scope.resource.viewItems.each(function (item) {
                    keys.add(item.key);
                });
                $scope.resource.instance.viewItems.each(function (item) {
                    item.covered = keys.has(item.key);
                });

            });
        }

        function preDeleteItem(item) {
            UiModel.confirm("确定要删除配置吗?", "删除配置", {
                submitBtn: "确认",
                cancelBtn: "取消",
                size: "sm",//大小可大可小
                submitBtnClass: "danger"
            }).then(function () {

                const params = {
                    referenceId:$scope.resource.id,
                    key: item.key
                };
                $http.post($rootPath+"/apolloApi/removeReferenceItem", params).success(function () {
                    UiModel.message("删除成功");
                    refreshResource();
                })

            });
        }

        function showText(text) {
            ApolloService.showText(text);
        }

        function coverItem(item) {
            const params = {
                key: item.key,
                comment: item.comment,
                password: false
            };
            if (!item.password) {
                params.value = item.newValue;
            }
            ApolloService.create(params).then(function (itemResult) {
                itemResult.referenceId = $scope.resource.id;
                $http.post($rootPath+"/apolloApi/editReferenceItem", itemResult).success(function () {
                    UiModel.message("保存成功");
                    refreshResource();
                })
            });
        }

        function editItem(item) {
            const params = {
                key: item.key,
                comment: item.comment,
                password: item.password
            };
            if (!item.password) {
                params.value = item.newValue;
            }
            ApolloService.edit(params).then(function (itemResult) {
                itemResult.referenceId = $scope.resource.id;
                $http.post($rootPath+"/apolloApi/editReferenceItem", itemResult).success(function () {
                    UiModel.message("保存成功");
                    refreshResource();
                })
            });
        }

        function publish() {
            var uibModalInstance = $uibModal.open({
                size: "lg",
                animation: true,
                templateUrl: '/apollo/model/release-reference.html',
                controller: "ReleaseReferenceModalController",
                backdrop: true,
                resolve: {
                    resource: function () {
                        return angular.copy($scope.resource);
                    }
                }
            });
            uibModalInstance.result.then(function () {
                refreshResource();
            });
        }

        function switchView(viewType) {
            $scope.resource.viewType = viewType;
            if (view_type.TEXT === viewType) {
                $scope.resource.text = parsePropertiesText($scope.resource);
            }
        }

        function startReference() {
            const instance = $uibModal.open({
                size: "mg",
                animation: true,
                templateUrl: '/apollo/model/start-reference.html',
                controller: "StartReferenceController",
                backdrop: 'static',
                resolve: {
                    reference: function () {
                        return angular.copy($scope.resource);
                    },
                }
            });
            instance.result.then(function () {
                UiModel.message("资源启用成功");
                initResource();
            });
        }

        function removeResourceReference(){
            ApolloResourceService.removeResourceReference($scope.resource.bizId,$scope.resource.id).then(function (){
                $rootScope.$broadcast("refresh_resource");
                UiModel.message("成功移除资源");
            })
        }

        function stopReference() {
            UiModel.confirm("确认停用资源[" + $scope.resource.instanceName + "]？", "停用资源", {
                submitBtn: "停用",
                submitBtnClass: 'danger',
            }).then(function () {
                ApolloResourceService.stopReference($scope.resource.bizId,$scope.resource.id).then(function () {
                    UiModel.message("资源已停用");
                    $rootScope.$broadcast("refresh_resource");
                })
            })
        }

        function parsePropertiesText(resource) {
            var result = "";
            const keys = new Set();
            resource.viewItems.forEach(function (item) {
                if (item.key) {
                    //use string \n to display as new line
                    var itemValue = item.newValue.replace(/\n/g, "\\n");
                    result += item.key + " = " + itemValue + "\n";
                }
                keys.add(item.key);
            });
            resource.instance.viewItems.forEach(function (item) {
                if (keys.has(item.key)) {
                    return;
                }
                if (item.key) {
                    //use string \n to display as new line
                    var itemValue = item.newValue.replace(/\n/g, "\\n");
                    result += item.key + " = " + itemValue + "\n";
                }
            });
            return result;
        }
    }

    function ApolloResourceTemplatePreviewController($uibModalInstance,params,UiModel,$scope,$http){

        $scope.items = [];

        const p = UiModel.loading("预览配置加载中...");
        $http.get($rootPath+"/apolloApi/previewTemplateRenderConfigs",{params:params}).success(function(items){
            $scope.items = items;
            p.close();
        },function () {
            p.close();
        });

        $scope.cancel = function () {
            $uibModalInstance.dismiss();
        };
    }

    function ApolloInstancesModelController($scope,$uibModalInstance,$http,modelParams,ApolloService){
        $scope.types = [];
        $scope.title = modelParams.title;
        $scope.typeId = null;
        $scope.hiddenResourceTypeSelect = false;
        $scope.page={
            currentPage:1,
            max:10,
            total:0
        };

        const params = {
            key:"",typeId:null
        };
        $scope.instances = [];
        $scope.loadApolloInstances = loadApolloInstances;
        $scope.search = search;
        $scope.preview = preview;

        function preview(instance){
            ApolloService.preview(instance.id, null);
        }

        $scope.$watch("typeId",function(typeId){
            if(!typeId){
                return;
            }
            search();
        })

        loadTypes();

        function search(){
            params.key = $scope.key;
            params.typeId = $scope.typeId;
            $scope.page.currentPage = 1;
            loadApolloInstances();
        }

        function loadApolloInstances(){
            const request = {
                pageIndex:$scope.page.currentPage-1,
                pageSize:$scope.page.max,
                key:params.key,
                typeId:params.typeId,
            };
            $http.post($rootPath+"/apolloApi/findResourceInstancePage",request).success(function(page){
                $scope.instances = page.list;
                $scope.page.total = page.count;
            })
        }

        function loadTypes(){
            $http.get($rootPath+"/apolloApi/findApolloTypes").success(function(types){
                $scope.types = [];
                types.each(function(type){
                    if(modelParams.types.size ===0 || modelParams.types.has(type.key)){
                        $scope.types.push(type);
                    }
                });
                if($scope.types.length){
                    $scope.typeId = $scope.types[0].id;
                }
                if($scope.types.length===1){
                    $scope.hiddenResourceTypeSelect = true;
                }
            })
        }
        $scope.selectInstance = $uibModalInstance.close;

        $scope.cancel = function () {
            $uibModalInstance.dismiss();
        };
    }


    function StartReferenceController($uibModalInstance, $scope, reference, $http, UiModel, ApolloTemplateService,ApolloService,ApolloResourceService) {

        $scope.reference = reference;
        $scope.instanceAlias = "";

        $scope.selectTemplate = function () {
            ApolloTemplateService.openTemplateSelectModal({
                id: reference.instanceId, name: reference.instanceName, typeId: reference.typeId
            }).then(function (template) {
                $scope.reference.templateId = template.id;
                $scope.reference.templateName = template.name;
            })
        };

        $scope.preview = function () {
            ApolloService.preview(reference.instanceId, reference.templateId,$scope.instanceAlias);
        };

        $scope.submit = function () {
            const params = {
                referenceId:reference.id,
                templateId: $scope.reference.templateId,
                instanceAlias: $scope.instanceAlias
            };
            ApolloResourceService.startReference(reference.bizId,params).then(function(){
                $uibModalInstance.close();
            });
        };

        $scope.cancel = function () {
            $uibModalInstance.dismiss();
        };

    }


    function ReleaseReferenceModalController($uibModalInstance, $scope, resource, $http, UiModel) {
        $scope.resource = resource;

        var date = new Date().Format("yyyyMMddhhmmss");
        $scope.releaseTitle = date + "-release";
        $scope.releaseComment = "";
        $scope.releaseBtnDisabled = false;

        $scope.release = function () {
            $scope.releaseBtnDisabled = true;
            $http.post($rootPath+"/apolloApi/publishReference", {
                referenceId: resource.id,
                name: $scope.releaseTitle,
                comment: $scope.releaseComment
            }).success(function () {
                UiModel.message("发布成功");
                $uibModalInstance.close();
            })
        };

        $scope.cancel = function () {
            $uibModalInstance.dismiss();
        };
    }


})