package cn.com.duiba.apollo.ui.controller;

import cn.com.duiba.apollo.center.api.domain.dto.ResourceTemplateDto;
import cn.com.duiba.apollo.center.api.domain.dto.ResourceTypeDto;
import cn.com.duiba.apollo.center.api.domain.params.KeyValue;
import cn.com.duiba.apollo.center.api.remoteservice.RemoteApolloTemplateService;
import cn.com.duiba.apollo.center.api.remoteservice.RemoteResourceTypeService;
import cn.com.duiba.apollo.client.dto.PagenationDTO;
import cn.com.duiba.apollo.client.dto.ReferenceItemViewDTO;
import cn.com.duiba.apollo.client.dto.ResourceInstanceDTO;
import cn.com.duiba.apollo.client.dto.ResourceReferenceDTO;
import cn.com.duiba.apollo.client.params.*;
import cn.com.duiba.apollo.client.remoteservice.RemoteResourceInstanceService;
import cn.com.duiba.apollo.client.remoteservice.RemoteResourceReferenceService;
import cn.com.duiba.apollo.client.service.ApolloAccessTokenService;
import cn.com.duiba.apollo.ui.ApolloResourceService;
import cn.com.duiba.apollo.ui.vo.ReferenceBO;
import cn.com.duiba.apollo.ui.vo.ResourceReferenceVO;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.sso.api.domain.dto.AdminInfoDto;
import cn.com.duiba.sso.api.remoteservice.RemoteAdminService;
import cn.com.duiba.sso.api.service.logger.SsoLogger;
import cn.com.duiba.sso.api.service.logger.annotation.SsoLoggerMethod;
import cn.com.duiba.sso.api.tool.JsonRender;
import cn.com.duiba.sso.api.tool.RequestTool;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.util.*;

/**
 * @author liuyao
 */
@RestController
@RequestMapping("/${duiba.sso.root-path:'/apollo'}/apolloApi")
public class ApolloInstanceConfigController {

    @Resource
    private RemoteResourceReferenceService remoteResourceReferenceService;
    @Resource
    private ApolloAccessTokenService apolloAccessTokenService;
    @Resource
    private ApolloResourceService apolloResourceService;
    @Resource
    private RemoteResourceInstanceService remoteResourceInstanceService;
    @Resource
    private RemoteAdminService remoteAdminService;
    @Resource
    private RemoteApolloTemplateService remoteApolloTemplateService;
    @Resource
    private RemoteResourceTypeService remoteResourceTypeService;

    @GetMapping("/findApolloTypes")
    public List<ResourceTypeDto> findApolloTypes(){
        return remoteResourceTypeService.findAllTypes();
    }

    @GetMapping("/findAllTemplates")
    public List<ResourceTemplateDto> findAllTemplates(@RequestParam String type){
        return remoteApolloTemplateService.findResourceTemplateByType(type);
    }

    /**
     * 获取物理机实例列表
     */
    @PostMapping("/findResourceInstancePage")
    public PagenationDTO<ResourceInstanceDTO> findResourceInstancePage(@RequestBody @Valid ResourceInstancePageParams params) throws BizException {
        return remoteResourceInstanceService.findInstancePage(params);
    }

    @GetMapping("/findAllReferenceByBizId")
    public List<ResourceReferenceVO> findAllReferenceByBizId(@RequestParam String bizId) throws BizException {
        List<ResourceReferenceVO> vos = apolloResourceService.findReferenceByBizId(bizId);

        List<Long> adminIds = Lists.newArrayList();
        for(ResourceReferenceVO reference:vos){
            adminIds.add(reference.getOperatorId());
        }
        Map<Long, AdminInfoDto> adminInfoDtoMap = Maps.newHashMap();
        List<AdminInfoDto> adminInfoDtos = remoteAdminService.batchFindAdminInfoByIds(adminIds);
        for(AdminInfoDto admin:adminInfoDtos){
            adminInfoDtoMap.put(admin.getId(),admin);
        }
        for(ResourceReferenceVO reference:vos){
            AdminInfoDto admin = adminInfoDtoMap.get(reference.getOperatorId());
            reference.setOperator(admin.getName());
        }
        return Lists.reverse(vos);
    }

    @GetMapping("/findResourceInstance")
    public ResourceInstanceDTO findResourceInstance(@RequestParam Long id){
        return apolloResourceService.findApolloInstance(id);
    }

    @SsoLoggerMethod(value = "bizId:${bizId}发布资源${instanceId}",group = "资源发布")
    @PostMapping("/publishReference")
    public JsonRender publishReference(@RequestBody @Valid PublishReferenceParams params) throws BizException {
        String accessToken = apolloAccessTokenService.getAccessToken();
        params.setOperator(RequestTool.getAdminId());
        remoteResourceReferenceService.publishReference(accessToken,params);
        ResourceReferenceDTO reference = remoteResourceReferenceService.findResourceReference(accessToken,params.getReferenceId());
        SsoLogger.logForPojo(reference);
        return JsonRender.successResult();
    }

    @SsoLoggerMethod(value = "${bizId}资源:${instanceId}的配置发生变更，Key:${key}",group = "资源配置变更")
    @PostMapping("/editReferenceItem")
    public JsonRender editReferenceItem(@RequestBody @Valid ReferenceItemParams params) throws BizException {

        checkItemKeyLength(params.getKey());
        checkItemValueLength(params.getValue());

        String accessToken = apolloAccessTokenService.getAccessToken();
        KeyValue kv = new KeyValue();
        kv.setKey(params.getKey());
        kv.setValue(params.getValue());
        kv.setComment(params.getComment());
        kv.setPassword(params.getPassword());
        remoteResourceReferenceService.overrideReferenceItem(accessToken,RequestTool.getAdminId(),params.getReferenceId(),kv);

        ResourceReferenceDTO reference = remoteResourceReferenceService.findResourceReference(accessToken,params.getReferenceId());

        JSONObject json = new JSONObject();
        json.put("bizId",reference.getBizId());
        json.put("instanceId",reference.getInstanceId());
        json.put("key",params.getKey());
        SsoLogger.logForPojo(json);

        return JsonRender.successResult();
    }

    @SsoLoggerMethod(value = "${bizId}资源:${instanceId}的配置被删除，Key:${key}",group = "删除资源配置")
    @PostMapping("/removeReferenceItem")
    public JsonRender removeReferenceItem(@RequestBody ReferenceItemParams params) throws BizException{

        String accessToken = apolloAccessTokenService.getAccessToken();

        ResourceReferenceDTO reference = remoteResourceReferenceService.findResourceReference(accessToken,params.getReferenceId());
        if(reference==null || !reference.getActive()){
            throw new BizException("资源配置不存在");
        }
        remoteResourceReferenceService.removeReferenceItem(accessToken,RequestTool.getAdminId(),reference.getId(),params.getKey());
        return JsonRender.successResult();
    }

    @GetMapping("/findReferenceItems")
    public ReferenceBO findReferenceItems(@RequestParam String bizId, @RequestParam Long transcriptId) throws BizException {
        ReferenceBO referenceBO = apolloResourceService.findReferenceItemsByTranscriptId(bizId,transcriptId);
        if(!referenceBO.getActive()){
            return referenceBO;
        }
        List<ReferenceItemViewDTO> items = Optional.ofNullable(referenceBO.getReferenceView().getItems()).orElse(Collections.emptyList());
        Set<Long> adminIds = Sets.newHashSet();
        for(ReferenceItemViewDTO item:items){
            adminIds.add(item.getOperator());
        }
        List<AdminInfoDto> adminInfoDtos = remoteAdminService.batchFindAdminInfoByIds(Lists.newArrayList(adminIds));
        Map<Long,String> adminNames = Maps.newHashMap();
        for(AdminInfoDto adminInfoDto:adminInfoDtos){
            adminNames.put(adminInfoDto.getId(),adminInfoDto.getName());
        }
        referenceBO.setAdminNames(adminNames);
        return referenceBO;
    }

    @GetMapping("/previewTranscriptConfigs")
    public List<KeyValue> previewTranscriptConfigs(
            @RequestParam Long transcriptId,
            @RequestParam(required = false) Long templateId,
            @RequestParam(required = false) String instanceAlias) throws BizException {

        TranscriptConfigParams params = new TranscriptConfigParams();
        params.setInstanceAlias(StringUtils.trim(instanceAlias));
        params.setTemplateId(templateId);
        params.setTranscriptId(transcriptId);
        String accessToken = apolloAccessTokenService.getAccessToken();
        return remoteResourceInstanceService.previewTranscriptConfig(accessToken, params);
    }

    private void checkItemKeyLength(String key) throws BizException {
        if (!StringUtils.isEmpty(key) && key.length() > 128) {
            throw new BizException("key too long. length limit:" + 128);
        }
    }

    private void checkItemValueLength(String value) throws BizException {
        if (!StringUtils.isEmpty(value) && value.length() > 20000) {
            throw new BizException("value too long. length limit:" + 20000);
        }
    }


}
