package cn.com.duiba.apollo.ui;

import cn.com.duiba.apollo.center.api.domain.dto.ResourceTemplateDto;
import cn.com.duiba.apollo.center.api.domain.dto.ResourceTypeDto;
import cn.com.duiba.apollo.center.api.domain.params.KeyValue;
import cn.com.duiba.apollo.center.api.remoteservice.RemoteApolloTemplateService;
import cn.com.duiba.apollo.center.api.remoteservice.RemoteResourceTypeService;
import cn.com.duiba.apollo.client.dto.*;
import cn.com.duiba.apollo.client.remoteservice.RemoteResourceInstanceService;
import cn.com.duiba.apollo.client.remoteservice.RemoteResourceReferenceService;
import cn.com.duiba.apollo.client.remoteservice.RemoteResourceTranscriptService;
import cn.com.duiba.apollo.client.service.ApolloAccessTokenService;
import cn.com.duiba.apollo.client.service.resource.ResourceReferenceClientService;
import cn.com.duiba.apollo.client.utils.BeanUtils;
import cn.com.duiba.apollo.client.utils.PasswordUtils;
import cn.com.duiba.apollo.ui.vo.ReferenceBO;
import cn.com.duiba.apollo.ui.vo.ResourceReferenceVO;
import cn.com.duiba.boot.exception.BizException;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import javax.annotation.Resource;
import java.util.*;

/**
 * @author liuyao
 */
public class ApolloResourceService {

    @Resource
    private RemoteResourceInstanceService remoteResourceInstanceService;
    @Resource
    private ApolloAccessTokenService apolloAccessTokenService;
    @Resource
    private RemoteResourceReferenceService remoteResourceReferenceService;
    @Resource
    private ResourceReferenceClientService resourceReferenceClientService;
    @Resource
    private RemoteResourceTypeService remoteResourceTypeService;
    @Resource
    private RemoteApolloTemplateService remoteApolloTemplateService;
    @Resource
    private RemoteResourceTranscriptService remoteResourceTranscriptService;

    public List<ResourceReferenceVO> findReferenceByBizId(String bizId) throws BizException {
        String accessToken = apolloAccessTokenService.getAccessToken();
        List<ResourceReferenceDTO> references = remoteResourceReferenceService.findResourceReferenceByBizId(accessToken,bizId);
        if(references.isEmpty()){
            return Collections.emptyList();
        }
        List<Long> instanceIds = Lists.newArrayList();
        List<Long> transcriptIds = Lists.newArrayList();
        List<Long> templateIds = Lists.newArrayList();
        List<ResourceReferenceDTO> sources = Lists.newArrayList();

        for(ResourceReferenceDTO reference:references){
            sources.add(reference);
            instanceIds.add(reference.getInstanceId());
            transcriptIds.add(reference.getTranscriptId());
            templateIds.add(reference.getTemplateId());
        }

        List<ResourceTypeDto> resourceTypes = remoteResourceTypeService.findAllTypes();
        Map<String,ResourceTypeDto> resourceTypeMap = BeanUtils.mapByKey("type",resourceTypes);

        List<ResourceInstanceDTO> instances = remoteResourceInstanceService.findInstancesByIds(instanceIds);
        Map<Long,ResourceInstanceDTO> instanceMap = Maps.newHashMap();
        for(ResourceInstanceDTO instance:instances){
            instanceMap.put(instance.getId(),instance);
        }
        List<ResourceInstanceTranscriptDTO> transcripts = remoteResourceTranscriptService.findAllByTranscriptIds(transcriptIds);
        Map<Long, ResourceInstanceTranscriptDTO> transcriptMap = Maps.newHashMap();
        for (ResourceInstanceTranscriptDTO transcript : transcripts) {
            transcriptMap.put(transcript.getId(), transcript);
        }
        Map<Long, ResourceTemplateDto> templateMap = Maps.newHashMap();
        if(!templateIds.isEmpty()){
            List<ResourceTemplateDto> templates = remoteApolloTemplateService.findResourceTemplateByIds(templateIds);
            for(ResourceTemplateDto template:templates){
                templateMap.put(template.getId(),template);
            }
        }
        List<ResourceReferenceVO> result = Lists.newArrayList();
        for(ResourceReferenceDTO reference:sources){
            ResourceReferenceVO vo = new ResourceReferenceVO();
            vo.setId(reference.getId());
            vo.setBizId(reference.getBizId());
            vo.setInstanceId(reference.getInstanceId());
            vo.setTemplateId(reference.getTemplateId());
            vo.setActive(reference.getActive());
            vo.setInstanceAlias(reference.getInstanceAlias());
            vo.setOperatorId(reference.getOperator());
            ResourceInstanceDTO instance = instanceMap.get(reference.getInstanceId());
            if(Objects.nonNull(instance)){
                vo.setInstanceKey(instance.getKey());
                vo.setInstanceName(instance.getName());
                vo.setType(instance.getType());
                ResourceTypeDto resourceType = resourceTypeMap.get(instance.getType());
                vo.setTypeName(resourceType.getName());
            }
            ResourceInstanceTranscriptDTO transcript = transcriptMap.get(reference.getTranscriptId());
            if(Objects.nonNull(transcript)){
                vo.setTranscriptName(transcript.getTranscriptName());
            }
            if(Objects.nonNull(reference.getTemplateId())){
                ResourceTemplateDto template = templateMap.get(reference.getTemplateId());
                vo.setTemplateName(template.getName());
            }
            vo.setAuthorized(reference.getAuthorized());
            vo.setTranscriptId(reference.getTranscriptId());
            result.add(vo);
        }
        return result;
    }

    public List<ResourceInstanceDTO> findApolloInstances(Collection<Long> ids){
        return remoteResourceInstanceService.findInstancesByIds(Lists.newArrayList(ids));
    }

    public ResourceInstanceDTO findApolloInstance(Long instanceId){
        return remoteResourceInstanceService.findOneInstance(instanceId);
    }

    public Properties findApolloTranscriptConfig(Long transcriptId) throws BizException {
        List<KeyValue> keyValues = remoteResourceInstanceService.findTranscriptNativeConfig(transcriptId);
        Properties instanceConfigs = new Properties();
        for(KeyValue kv:keyValues){
            instanceConfigs.put(kv.getKey(), kv.getValue());
        }
        return instanceConfigs;
    }

    /**
     * 根据副本查找引用
     *
     * @param bizId
     * @param transcriptId
     * @return
     * @throws BizException
     */
    public ReferenceBO findReferenceItemsByTranscriptId(String bizId, Long transcriptId) throws BizException {
        String accessToken = apolloAccessTokenService.getAccessToken();
        ResourceReferenceDTO reference = remoteResourceReferenceService.findReferenceByBizIdAndTranscriptId(accessToken, bizId, transcriptId);
        if (reference == null) {
            return null;
        }
        ReferenceBO result = new ReferenceBO();
        result.setReferenceId(reference.getId());
        result.setAuthorized(reference.getAuthorized());
        result.setActive(reference.getActive());
        if (!reference.getAuthorized() || !reference.getActive()) {
            return result;
        }
        ReferenceViewDTO referenceView = remoteResourceReferenceService.findReferenceView(accessToken, reference.getId());
        result.setReferenceView(referenceView);
        return result;
    }

    public String charkInstanceType(Long instanceId,Set<String> supportTypes) throws BizException {
        ResourceInstanceDTO instance = remoteResourceInstanceService.findOneInstance(instanceId);
        Objects.requireNonNull(instance);
        return charkInstanceType(instance,supportTypes);
    }

    public String charkInstanceType(ResourceInstanceDTO instance,Set<String> supportTypes) throws BizException {
        ResourceTypeDto resourceType = remoteResourceTypeService.findByType(instance.getType());
        Objects.requireNonNull(resourceType);
        if(!supportTypes.contains(resourceType.getType())){
            throw new BizException("不支持设置["+resourceType.getName()+"]类型的资源实例");
        }
        return resourceType.getType();
    }

    /**
     * 构建与apollo实例副本之间的联系
     */
    public void buildApolloTranscriptRelation(ResourceReferenceDTO params) throws BizException {

        String accessToken = apolloAccessTokenService.getAccessToken();
        ResourceReferenceDTO reference = remoteResourceReferenceService.findReferenceByBizIdAndTranscriptId(accessToken,params.getBizId(),params.getTranscriptId());
        if(Objects.nonNull(reference)){
            return;
        }
        remoteResourceReferenceService.applyResourceInstance(accessToken,params);
    }

    public void deleteReferenceByTranscriptId(String bizId, Long transcriptId, Long operator) throws BizException {
        String accessToken = apolloAccessTokenService.getAccessToken();
        ResourceReferenceDTO reference = remoteResourceReferenceService.findReferenceByBizIdAndTranscriptId(accessToken, bizId, transcriptId);
        if (Objects.isNull(reference)) {
            return;
        }
        if (reference.getActive()) {
            remoteResourceReferenceService.stopReference(accessToken, reference.getId());
        }
        remoteResourceReferenceService.removeResourceReference(accessToken, operator, reference.getId());
    }

    public ResourceReferenceDTO findResourceReferenceByTranscriptId(String bizId, Long transcriptId) throws BizException {
        String accessToken = apolloAccessTokenService.getAccessToken();
        return remoteResourceReferenceService.findReferenceByBizIdAndTranscriptId(accessToken, bizId, transcriptId);
    }

    /**
     * 获取canal实例完整的配置
     */
    public Properties findCanalInstanceConfigByTranscriptId(String bizId, Long transcriptId) throws BizException {
        ResourceReferenceDTO reference = findResourceReferenceByTranscriptId(bizId, transcriptId);
        if (!reference.getActive()) {
            throw new BizException("Apollo实例配置未激活");
        }
        ReferenceReleaseDTO referenceRelease = resourceReferenceClientService.findReferenceProperties(reference.getId());
        Properties properties = referenceRelease.getReleaseProperties();

        Properties resultProperties = new Properties();
        for (String key : properties.stringPropertyNames()) {
            String value = properties.getProperty(key);
            resultProperties.put(key, PasswordUtils.decryptValue(value));
        }
        return resultProperties;
    }

}
