package cn.com.duiba.cloud.duiba.sentinel.service.api.remoteservice.config;

import cn.com.duiba.cloud.duiba.sentinel.service.api.remoteservice.dto.TokenServiceConfigDTO;
import cn.com.duiba.cloud.duiba.sentinel.service.api.remoteservice.remoteservice.RemoteRuleConfigService;
import cn.hutool.extra.spring.SpringUtil;
import com.alibaba.csp.sentinel.cluster.ClusterConstants;
import com.alibaba.csp.sentinel.cluster.ClusterErrorMessages;
import com.alibaba.csp.sentinel.cluster.ClusterTransportClient;
import com.alibaba.csp.sentinel.cluster.TokenResult;
import com.alibaba.csp.sentinel.cluster.TokenResultStatus;
import com.alibaba.csp.sentinel.cluster.TokenServerDescriptor;
import com.alibaba.csp.sentinel.cluster.client.ClientConstants;
import com.alibaba.csp.sentinel.cluster.client.ClusterTokenClient;
import com.alibaba.csp.sentinel.cluster.client.NettyTransportClient;
import com.alibaba.csp.sentinel.cluster.client.config.ClusterClientAssignConfig;
import com.alibaba.csp.sentinel.cluster.client.config.ClusterClientConfigManager;
import com.alibaba.csp.sentinel.cluster.client.config.ServerChangeObserver;
import com.alibaba.csp.sentinel.cluster.log.ClusterClientStatLogUtil;
import com.alibaba.csp.sentinel.cluster.request.ClusterRequest;
import com.alibaba.csp.sentinel.cluster.request.data.FlowRequestData;
import com.alibaba.csp.sentinel.cluster.request.data.ParamFlowRequestData;
import com.alibaba.csp.sentinel.cluster.response.ClusterResponse;
import com.alibaba.csp.sentinel.cluster.response.data.FlowTokenResponseData;
import com.alibaba.csp.sentinel.util.StringUtil;
import lombok.extern.slf4j.Slf4j;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 兑吧集群token-client处理
 *
 * @author huangguosheng@duiba.com.cn
 * @date 2022/3/10 5:18 下午
 **/
@Slf4j
public class DuibaClusterTokenClient implements ClusterTokenClient {
    private HashMap<String, ClusterTransportClient> transportClientMap;
    private HashMap<String, TokenServerDescriptor> serverDescriptorMap;

    private final AtomicBoolean shouldStart = new AtomicBoolean(false);

    public DuibaClusterTokenClient() {
        ClusterClientConfigManager.addServerChangeObserver(new ServerChangeObserver() {
            @Override
            public void onRemoteServerChange(ClusterClientAssignConfig assignConfig) {
                changeServer(assignConfig);
            }
        });
      // initNewConnection();
    }

    private boolean serverEqual(TokenServerDescriptor descriptor, ClusterClientAssignConfig config) {
        if (descriptor == null || config == null) {
            return false;
        }
        return descriptor.getHost().equals(config.getServerHost()) && descriptor.getPort() == config.getServerPort();
    }

    private void initNewConnection() {
        RemoteRuleConfigService configService = SpringUtil.getBean(RemoteRuleConfigService.class);
        if(configService == null){
            return;
        }
        List<TokenServiceConfigDTO> tokenServices = configService.getTokenServices();
        for (TokenServiceConfigDTO item : tokenServices) {
            if (StringUtil.isBlank(item.getHost()) || item.getPort() <= 0) {
                continue;
            }

            // 已存在无需添加
            String server = ServerManage.getServer(item.getHost());
            if (StringUtil.isNotBlank(server)) {
                continue;
            }

            // 添加到serverManage
            ServerManage.addServer(item.getHost());

            try {
                NettyTransportClient transportClient = new NettyTransportClient(item.getHost(), item.getPort());
                transportClientMap.put(item.getHost(), transportClient);
                TokenServerDescriptor serverDescriptor = new TokenServerDescriptor(item.getHost(), item.getPort());
                serverDescriptorMap.put(item.getHost(), serverDescriptor);
                log.info("[DuibaClusterTokenClient] New client created: " + serverDescriptor);
            } catch (Exception ex) {
                log.warn("[DuibaClusterTokenClient] Failed to initialize new token client", ex);
            }
        }
        log.info("[DuibaClusterTokenClient]  client created done:{}", serverDescriptorMap);
    }

    /**
     * 服务变化
     */
    private void changeServer(ClusterClientAssignConfig assignConfig) {
        if(transportClientMap == null){
            initNewConnection();
            return;
        }

        ClusterTransportClient transportClient = transportClientMap.get(assignConfig.getServerHost());
        if (transportClient != null) {
            try {
                // 停止该client
                transportClient.stop();
                log.info("[DuibaClusterTokenClient] 停止 ip:{} token-server连接", assignConfig.getServerHost());
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        transportClientMap.remove(assignConfig.getServerHost());
        serverDescriptorMap.remove(assignConfig.getServerHost());

        // 重新建立连接
        initNewConnection();
    }

    private void startClientIfScheduled() throws Exception {
        if (shouldStart.get()) {
            for (Map.Entry<String, ClusterTransportClient> entry : transportClientMap.entrySet()) {
                if (entry.getValue() != null) {
                    entry.getValue().start();
                } else {
                    log.warn("[DuibaClusterTokenClient] Cannot start transport client: client not created");
                }
            }
        }
    }

    private void stopClientIfStarted() throws Exception {
        if (shouldStart.compareAndSet(true, false)) {
            for (Map.Entry<String, ClusterTransportClient> entry : transportClientMap.entrySet()) {
                if (entry.getValue() != null) {
                    entry.getValue().stop();
                } else {
                    log.warn("[DuibaClusterTokenClient] Cannot stop transport client: client not created");
                }
            }
        }
    }

    @Override
    public void start() throws Exception {
        if (shouldStart.compareAndSet(false, true)) {
            startClientIfScheduled();
        }
    }

    @Override
    public void stop() throws Exception {
        stopClientIfStarted();
    }

    @Override
    public int getState() {
        boolean isOk = true;
        for (Map.Entry<String, ClusterTransportClient> entry : transportClientMap.entrySet()) {
            if (entry.getValue() == null) {
                isOk = false;
            }
            if (!entry.getValue().isReady()) {
                isOk = false;
            }
        }
        return isOk ? ClientConstants.CLIENT_STATUS_STARTED : ClientConstants.CLIENT_STATUS_OFF;
    }

    @Override
    public TokenServerDescriptor currentServer() {
        return null;
    }

    @Override
    public TokenResult requestToken(Long flowId, int acquireCount, boolean prioritized) {
        if (notValidRequest(flowId, acquireCount)) {
            return badRequest();
        }
        FlowRequestData data = new FlowRequestData().setCount(acquireCount)
                .setFlowId(flowId).setPriority(prioritized);
        ClusterRequest<FlowRequestData> request = new ClusterRequest<>(ClusterConstants.MSG_TYPE_FLOW, data);
        try {
            TokenResult result = sendTokenRequest(request, flowId);
            logForResult(result);
            return result;
        } catch (Exception ex) {
            ClusterClientStatLogUtil.log(ex.getMessage());
            return new TokenResult(TokenResultStatus.FAIL);
        }
    }

    @Override
    public TokenResult requestParamToken(Long flowId, int acquireCount, Collection<Object> params) {
        if (notValidRequest(flowId, acquireCount) || params == null || params.isEmpty()) {
            return badRequest();
        }
        ParamFlowRequestData data = new ParamFlowRequestData().setCount(acquireCount)
                .setFlowId(flowId).setParams(params);
        ClusterRequest<ParamFlowRequestData> request = new ClusterRequest<>(ClusterConstants.MSG_TYPE_PARAM_FLOW, data);
        try {
            TokenResult result = sendTokenRequest(request, flowId);
            logForResult(result);
            return result;
        } catch (Exception ex) {
            ClusterClientStatLogUtil.log(ex.getMessage());
            return new TokenResult(TokenResultStatus.FAIL);
        }
    }

    private void logForResult(TokenResult result) {
        switch (result.getStatus()) {
            case TokenResultStatus.NO_RULE_EXISTS:
                ClusterClientStatLogUtil.log(ClusterErrorMessages.NO_RULES_IN_SERVER);
                break;
            case TokenResultStatus.TOO_MANY_REQUEST:
                ClusterClientStatLogUtil.log(ClusterErrorMessages.TOO_MANY_REQUESTS);
                break;
            default:
        }
    }

    private TokenResult sendTokenRequest(ClusterRequest request, Long flowId) throws Exception {
        // 一致性hash获得对应的server
        String server = ServerManage.getRouterServer("" + flowId);
        ClusterTransportClient transportClient = transportClientMap.get(server);
        if (transportClient == null) {
            log.warn(
                    "[DuibaClusterTokenClient] Client not created, please check your config for cluster client");
            return clientFail();
        }
        ClusterResponse response = transportClient.sendRequest(request);
        TokenResult result = new TokenResult(response.getStatus());
        if (response.getData() != null) {
            FlowTokenResponseData responseData = (FlowTokenResponseData) response.getData();
            result.setRemaining(responseData.getRemainingCount())
                    .setWaitInMs(responseData.getWaitInMs());
        }
        return result;
    }

    private boolean notValidRequest(Long id, int count) {
        return id == null || id <= 0 || count <= 0;
    }

    private TokenResult badRequest() {
        return new TokenResult(TokenResultStatus.BAD_REQUEST);
    }

    private TokenResult clientFail() {
        return new TokenResult(TokenResultStatus.FAIL);
    }


}
