package cn.com.duiba.tuia.dsp.engine.api.dsp.alipay;

import cn.com.duiba.spring.boot.starter.dsp.sampler.SamplerLog;
import cn.com.duiba.tuia.dsp.engine.api.dsp.AbstractDspCaller;
import cn.com.duiba.tuia.dsp.engine.api.dsp.alipay.convert.AlipayDspReqConvert;
import cn.com.duiba.tuia.dsp.engine.api.dsp.alipay.convert.AlipayDspRespConvert;
import cn.com.duiba.tuia.dsp.engine.api.dsp.alipay.proto.AlipayDsp;
import cn.com.duiba.tuia.dsp.engine.api.dsp.alipay.proto.AlipayRelDspReq;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.req.AdxCommonBidRequest;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.req.DspInfo;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.resp.AdxCommonBidResponse;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.resp.CommonCreative;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.resp.CommonSeatBid;
import cn.com.duiba.tuia.dsp.engine.api.enums.DspEnum;
import cn.com.duiba.tuia.dsp.engine.api.exception.DspException;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import com.dianping.cat.Cat;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.net.SocketTimeoutException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.TimeUnit;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

@Slf4j
@Component
public class AlipayDspInvoker extends AbstractDspCaller<AlipayDsp.BidResponse, AlipayRelDspReq> {
    @Autowired
    private AlipayDspReqConvert alipayDspReqConvert;

    @Autowired
    private AlipayDspRespConvert alipayDspRespConvert;

    @Autowired
    private AlipayProperties alipayProperties;

    @Resource(name = "dspRestTemplate")
    private RestTemplate restTemplate;

    @Resource(name = "redis03StringRedisTemplate")
    private StringRedisTemplate stringRedisTemplate03;

    @Override
    protected AlipayDsp.BidResponse invokeDsp(AlipayRelDspReq bidRequest) {
        if (bidRequest == null) {
            return null;
        }
        Cat.logMetricForCount("支付宝DSP调用");
        try {
            return CatUtils.executeInCatTransaction(() -> this.doHttpInvoke(alipayProperties.getUrl(), bidRequest), "invokeDSP", "alipay");
        } catch (Throwable t) {
            Cat.logMetricForCount("支付宝_竞价失败");
            if (!(t instanceof RestClientException || t instanceof SocketTimeoutException)) {
                SamplerLog.warn("支付宝调用异常", t);
            }
        }

        return null;
    }

    private AlipayDsp.BidResponse doHttpInvoke(String url, AlipayRelDspReq bidRequest) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/x-protobuf;charset=UTF-8");
        headers.add("Content-Encoding", "gzip");
        headers.add("Accept", "application/x-protobuf;charset=UTF-8");
        headers.add("Accept-Encoding", "gzip");
        //使用gzip压缩
        byte[] requestBytes = gzipCompress(bidRequest.getBidRequest().toByteArray());
        if (requestBytes == null) {
            return null;
        }
        HttpEntity<byte[]> httpEntity = new HttpEntity<>(requestBytes, headers);
        ResponseEntity<byte[]> resEntity = null;
        try {
            resEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, byte[].class);
        } catch (Exception e) {
            if (!(e instanceof RestClientException)) {
                log.warn("支付宝调用异常 response is {}", resEntity, e);
            }
        }
        if (resEntity == null) {
            return null;
        }
        byte[] body = resEntity.getBody();
        AlipayDsp.BidResponse response = null;
        if (body != null) {
            byte[] decompress = gzipDecompress(body);
            if (decompress != null) {
                try {
                    response = AlipayDsp.BidResponse.parseFrom(decompress);
                } catch (Exception e) {
                    //
                    return null;
                }
            }
        }
        if (response != null && CollectionUtils.isNotEmpty(response.getSeatBidList())) {
            SamplerLog.info("支付宝 DSP返回 {}", resEntity);
            return response;
        }
        if (response != null && response.getNbr() == 9) {
            //到这个时间戳过期
            stringRedisTemplate03.opsForValue().set("alipay_nbr9_" + bidRequest.getDeviceId(), "", response.getColdEndTime(), TimeUnit.SECONDS);
        }
        return null;

    }

    @Override
    public String priceEncryption(BigDecimal price) {
        try {
            SecretKeySpec secretKey = new SecretKeySpec("ysDusbwFux32RAfJ".getBytes(StandardCharsets.UTF_8), "AES");
            Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5Padding");
            cipher.init(Cipher.ENCRYPT_MODE, secretKey);
            byte[] bytes = cipher.doFinal(price.toString().getBytes(StandardCharsets.UTF_8));
            //2进制转换成16进制
            StringBuilder sb = new StringBuilder();
            for (byte aByte : bytes) {
                String hex = Integer.toHexString(aByte & 0xFF);
                if (hex.length() == 1) {
                    hex = '0' + hex;
                }
                sb.append(hex);
            }
            return sb.toString();
        } catch (Exception e) {
            log.warn("支付宝加密失败", e);
        }
        return null;
    }

    @Override
    protected Integer getDspId() {
        return DspEnum.DSP_25.getDspId();
    }

    @Override
    protected DspEnum getDsp() {
        return DspEnum.DSP_25;
    }

    @Override
    protected String replaceUrl(String price, String url) {
        if (url.contains("${AUCTION_PRICE}") && price != null) {
            return url.replace("${AUCTION_PRICE}", price);
        }
        return url;
    }

    @Override
    public AlipayRelDspReq convertReq(AdxCommonBidRequest adxCommonBidRequest, DspInfo dspInfo) throws DspException {
        AlipayDsp.BidRequest request = alipayDspReqConvert.convert(adxCommonBidRequest, dspInfo);
        AlipayRelDspReq alipayRelDspReq = new AlipayRelDspReq();
        alipayRelDspReq.setBidRequest(request);
        return alipayRelDspReq;
    }

    @Override
    public AdxCommonBidResponse convertResp(AlipayDsp.BidResponse bidResponse) throws DspException {
        if (bidResponse == null) {
            return null;
        }
        AdxCommonBidResponse convert = alipayDspRespConvert.convert(bidResponse);
        convert.setDspId(getDspId());
        CommonSeatBid commonSeatBid = convert.getCommonSeatBidList().get(0);
        CommonCreative commonCreative = commonSeatBid.getCommonCreative();
        String deepLinkUrl = commonCreative.getDeepLinkUrl();
        String taskId = getTaskId(deepLinkUrl);
        commonCreative.setDspTaskId(taskId);
        return convert;
    }

    private String getTaskId(String deepLinkUrl) {
        if (StringUtils.isBlank(deepLinkUrl)) {
            return null;
        }
        try {
            deepLinkUrl = URLDecoder.decode(deepLinkUrl, StandardCharsets.UTF_8.toString());
            if (deepLinkUrl.contains("taskId")) {
                String[] params = deepLinkUrl.split("&");
                for (String param : params) {
                    if (param.startsWith("taskId")) {
                        String[] keyValue = param.split("=");
                        if (keyValue.length == 2) {
                            return keyValue[1];
                        }
                    }
                }
            }
        } catch (Exception e) {
            //
        }
        return null;
    }

    //gzip压缩
    private byte[] gzipCompress(byte[] data) {
        try (ByteArrayOutputStream bos = new ByteArrayOutputStream(); GZIPOutputStream gzipOS = new GZIPOutputStream(bos)) {
            gzipOS.write(data);
            gzipOS.finish();
            return bos.toByteArray();
        } catch (IOException e) {
            log.warn("gzip压缩失败", e);
        }
        return null;
    }

    //gzip解压
    private byte[] gzipDecompress(byte[] data) {
        try (ByteArrayInputStream bis = new ByteArrayInputStream(data); GZIPInputStream gis = new GZIPInputStream(bis); ByteArrayOutputStream bos = new ByteArrayOutputStream()) {
            byte[] buffer = new byte[4096];
            int len;
            // 4. 循环读取解压后的数据，写入输出流
            while ((len = gis.read(buffer)) != -1) {
                bos.write(buffer, 0, len);
            }
            // 5. 将缓存的解压数据转为字节数组返回
            return bos.toByteArray();
        } catch (IOException e) {
            log.warn("gzip解压失败", e);
        }
        return null;
    }
}
