package cn.com.duiba.duiba.base.service.api.mybatis.plugins;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.duiba.base.service.api.mybatis.plugins.bean.DbEncryptionConstant;
import cn.com.duiba.duiba.base.service.api.mybatis.plugins.bean.PropertyValueBean;
import cn.com.duiba.duiba.base.service.api.mybatis.plugins.config.DbEncryptColumnRule;
import cn.com.duiba.duiba.base.service.api.mybatis.plugins.config.DbEncryptTableRule;
import cn.com.duiba.duiba.base.service.api.mybatis.plugins.handler.command.SqlCommandAdapter;
import cn.com.duiba.duiba.base.service.api.mybatis.plugins.handler.encrypt.EncryptionDecryptionAdapter;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.ibatis.cache.CacheKey;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.mapping.ParameterMapping;
import org.apache.ibatis.mapping.ResultMap;
import org.apache.ibatis.mapping.ResultMapping;
import org.apache.ibatis.mapping.SqlCommandType;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.plugin.Intercepts;
import org.apache.ibatis.plugin.Invocation;
import org.apache.ibatis.plugin.Plugin;
import org.apache.ibatis.plugin.Signature;
import org.apache.ibatis.reflection.MetaObject;
import org.apache.ibatis.session.Configuration;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;

import javax.annotation.Nullable;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author lizhi
 * @date 2024/9/24 13:54
 */
@Slf4j
@Intercepts({
        @Signature(method = "update", type = Executor.class, args = { MappedStatement.class, Object.class }),
        @Signature(method = "query", type = Executor.class, args = { MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class}),
        @Signature(method = "query", type = Executor.class, args = { MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class, CacheKey.class, BoundSql.class})
})
public class DbEncryptionPlugin implements Interceptor {

    private final Map<String, DbEncryptTableRule> tables = new LinkedHashMap<>();

    private final Map<String, String> secretMap = new LinkedHashMap<>();

    private static final String FRCH = "__frch_";

    private static final String LOWER_SELECT = "select";
    private static final String LOWER_FROM = "from";

    @Override
    public Object intercept(Invocation invocation) throws Throwable {
        Object[] args = invocation.getArgs();
        if (MapUtils.isEmpty(tables) || args[1] == null) {
            return invocation.proceed();
        }
        long start = System.currentTimeMillis();
        boolean alreadyEncrypt = false;
        MappedStatement mappedStatement;
        String classAndMethod;
        SqlCommandType sqlCommandType;
        BoundSql boundSql = null;
        DbEncryptTableRule columns;
        boolean needLog = false;
        try {
            try {
                mappedStatement = getMappedStatement(invocation.getTarget(), args[0]);
                // 得到 类名-方法
                classAndMethod = getClassAndMethod(mappedStatement);
                // 获取命令类型
                sqlCommandType = getSqlCommandType(mappedStatement, classAndMethod);
                // 获取sql
                boundSql = getBoundSql(mappedStatement, args, classAndMethod);
                // 获取表对应的加密配置
                columns = getDbEncryptTableRule(sqlCommandType, boundSql, classAndMethod);
            } catch (BizException e) {
                log.info("DbEncryptionPlugin, parse error, boundSql={}, e:", boundSql == null ? null : boundSql.getSql(), e);
                return invocation.proceed();
            } catch (Throwable e) {
                log.error("DbEncryptionPlugin, parse error, boundSql={}, e:", boundSql == null ? null : boundSql.getSql(), e);
                return invocation.proceed();
            }
            if (columns == null || columns.getColumns() == null || columns.getColumns().isEmpty()) {
                // 该表无加密字段
                return invocation.proceed();
            }
            try {
                needLog = true;
                // 加密参数
                Map<String, PropertyValueBean> paramsRuleMap = getNeedEncryptParamsRule(mappedStatement, sqlCommandType, boundSql, columns.getColumns(), classAndMethod);
                Object encrypt = encrypt(paramsRuleMap, args[1], classAndMethod);
                alreadyEncrypt = true;
                args[1] = encrypt;
            } catch (BizException e) {
                log.info("DbEncryptionPlugin, encrypt error, boundSql={}, e:", boundSql.getSql(), e);
                return invocation.proceed();
            } catch (Throwable e) {
                log.error("DbEncryptionPlugin, encrypt error, boundSql={}, e:", boundSql.getSql(), e);
                return invocation.proceed();
            }
        } finally {
            if (needLog) {
                log.info("DbEncryptionPlugin, boundSql={}, args={}, alreadyEncrypt={}, cost={}ms", boundSql.getSql(), JSON.toJSONString(args[1]), alreadyEncrypt, System.currentTimeMillis() - start);
            }
        }
        Object proceed = invocation.proceed();
        if (proceed == null || isPrimitive(proceed.getClass())) {
            return proceed;
        }
        try {
            // 解密结果
            return decrypt(proceed, mappedStatement, boundSql, columns, classAndMethod);
        } catch (BizException e) {
            log.info("DbEncryptionPlugin, decrypt error, boundSql={}, e:", boundSql.getSql(), e);
            return proceed;
        } catch (Throwable e) {
            log.error("DbEncryptionPlugin, decrypt error, boundSql={}, e:", boundSql.getSql(), e);
            return proceed;
        }
    }

    private MappedStatement getMappedStatement(Object target, Object arg) throws BizException {
        if (!(target instanceof Executor)) {
            // 仅处理Executor
            throw new BizException("target非Executor");
        }
        if (!(arg instanceof MappedStatement)) {
            // 理论上第一个参数都是 MappedStatement
            throw new BizException("第一参数非MappedStatement");
        }
        return  (MappedStatement)arg;
    }

    private String getClassAndMethod(MappedStatement mappedStatement) {
        String[] strArr = mappedStatement.getId().split("\\.");
        return strArr[strArr.length-2] + "." + strArr[strArr.length-1];
    }

    private SqlCommandType getSqlCommandType(MappedStatement mappedStatement, String classAndMethod) throws BizException {
        SqlCommandType sqlCommandType = mappedStatement.getSqlCommandType();
        if (sqlCommandType == null) {
            // 获取不到sql命令类型
            throw new BizException(classAndMethod + " sqlCommandType为null");
        }
        return sqlCommandType;
    }

    private BoundSql getBoundSql(MappedStatement mappedStatement, Object[] args, String classAndMethod) throws BizException {
        BoundSql boundSql = mappedStatement.getBoundSql(args[1]);
        if (boundSql == null) {
            throw new BizException(classAndMethod + " boundSql为null");
        }
        return boundSql;
    }

    private DbEncryptTableRule getDbEncryptTableRule(SqlCommandType sqlCommandType, BoundSql boundSql, String classAndMethod) throws BizException {
        // 获取本次sql的表名
        String tableName = SqlCommandAdapter.getTableName(sqlCommandType, boundSql.getSql());
        if (tableName == null || tableName.isEmpty()) {
            throw new BizException(classAndMethod + " 获取tableName失败");
        }
        // 获取表对应的加密配置
        return tables.get(tableName);
    }

    private Map<String, PropertyValueBean> getNeedEncryptParamsRule(MappedStatement mappedStatement, SqlCommandType sqlCommandType, BoundSql boundSql, Map<String, DbEncryptColumnRule> columns, String classAndMethod) throws BizException {
        List<ParameterMapping> parameterMappings = boundSql.getParameterMappings();
        if (parameterMappings == null || parameterMappings.isEmpty()) {
            log.warn("DbEncryptionPlugin, parameterMappings is null or empty, sql={}, classAndMethod={}", boundSql.getSql(), classAndMethod);
            return Collections.emptyMap();
        }
        Map<Integer, DbEncryptColumnRule> paramIndexRuleMap = SqlCommandAdapter.getNeedEncryptParamIndexRule(sqlCommandType, boundSql.getSql(), columns);
        if (paramIndexRuleMap.isEmpty()) {
            return Collections.emptyMap();
        }
        MetaObject metaObject = null;
        Map<String, PropertyValueBean> paramRuleMap = new HashMap<>();
        for (Map.Entry<Integer, DbEncryptColumnRule> entry : paramIndexRuleMap.entrySet()) {
            String property = getProperty(entry.getKey(), parameterMappings, boundSql);
            if (property == null) {
                continue;
            }
            metaObject = put(mappedStatement, classAndMethod, paramRuleMap, property, entry.getValue(), metaObject, boundSql);
        }
        return paramRuleMap;
    }

    private MetaObject put(MappedStatement mappedStatement, String classAndMethod, Map<String, PropertyValueBean> paramRuleMap, String property, DbEncryptColumnRule rule, MetaObject metaObject, BoundSql boundSql) throws BizException {
        Object parameterObject = boundSql.getParameterObject();
        if (!property.startsWith(FRCH)) {
            putNotFrch(paramRuleMap, property, rule);
            return metaObject;
        }
        int separatorIndex = property.indexOf(".");
        if (separatorIndex >= 0) {
            putFrchObj(paramRuleMap, property, rule, separatorIndex);
            return metaObject;
        }
        if (metaObject == null) {
            metaObject = getMetaObject(mappedStatement, classAndMethod, parameterObject);
        }
        putFrchNotObj(paramRuleMap, property, rule, metaObject, boundSql);
        return metaObject;
    }

    private Object encrypt(Map<String, PropertyValueBean> paramsRuleMap, Object parameterObject, String classAndMethod) throws BizException {
        if (parameterObject == null || MapUtils.isEmpty(paramsRuleMap)) {
            return parameterObject;
        }
        if (parameterObject instanceof String) {
            return encryptString((String) parameterObject, paramsRuleMap, classAndMethod);
        }
        Class<?> clazz = parameterObject.getClass();
        if (isPrimitive(clazz)) {
            // 非字符串类型的基本类型
            throw new BizException(classAndMethod + " 需要加密，但入参为非字符串类型的基本类型");
        }
        if (parameterObject instanceof Map) {
            return encryptMap(toMap(parameterObject), paramsRuleMap, classAndMethod);
        }
        if (parameterObject instanceof Collection) {
            // 数组类型
            return encryptCollection(toCollection(parameterObject), paramsRuleMap, classAndMethod);
        }
        // 对象类型
        return encryptObject(parameterObject, paramsRuleMap);
    }

    private String encryptString(String parameterObject, Map<String, PropertyValueBean> paramsRuleMap, String classAndMethod) throws BizException {
        if (paramsRuleMap.size() != 1) {
            throw new BizException(classAndMethod + " 需要加密，入参为String，但是paramsRuleMap.size!=1");
        }
        PropertyValueBean bean = paramsRuleMap.values().stream().filter(Objects::nonNull).findFirst().orElse(null);
        if (bean == null) {
            return parameterObject;
        }
        DbEncryptColumnRule rule = bean.getRule();
        return EncryptionDecryptionAdapter.encryptString(parameterObject, rule, secretMap);
    }

    private Map<Object, Object> encryptMap(Map<Object, Object> parameterObject, Map<String, PropertyValueBean> paramsRuleMap, String classAndMethod) throws BizException {
        if (MapUtils.isEmpty(parameterObject)) {
            return parameterObject;
        }
        Map<Object, Object> newValueMap = new LinkedHashMap<>();
        for (Map.Entry<Object, Object> entry : parameterObject.entrySet()) {
            encryptMap(entry, newValueMap, paramsRuleMap, classAndMethod);
        }
        if (MapUtils.isNotEmpty(newValueMap)) {
            parameterObject.putAll(newValueMap);
        }
        return parameterObject;
    }

    private void encryptMap(Map.Entry<Object, Object> entry, Map<Object, Object> newValueMap, Map<String, PropertyValueBean> paramsRuleMap, String classAndMethod) throws BizException {
        Object key = entry.getKey();
        Object value = entry.getValue();
        if (value == null) {
            return;
        }
        if (value instanceof Collection) {
            // 集合
            newValueMap.put(key, encryptMapCollection(key, toCollection(value), paramsRuleMap, classAndMethod));
            return;
        }
        if (value instanceof Map) {
            // map套map，不支持
            return;
        }
        if (isPrimitive(value.getClass())) {
            // 基础类型，不支持
            return;
        }
        // 字符串或者json对象，不太可能是普通对象
        PropertyValueBean bean = paramsRuleMap.get(key.toString());
        if (bean == null) {
            return;
        }
        // 字符串/字符串json/对象json
        Pair<Boolean, Object> pair = encryptOrDecryptJson(value, bean.getRule(), true);
        if (BooleanUtils.isNotTrue(pair.getLeft())) {
            // 未加密
            return;
        }
        newValueMap.put(key, pair.getValue());
    }

    private Object encryptMapCollection(Object key, Collection<Object> collection, Map<String, PropertyValueBean> paramsRuleMap, String classAndMethod) throws BizException {
        if (CollectionUtils.isEmpty(collection)) {
            return collection;
        }
        PropertyValueBean bean = paramsRuleMap.get(key.toString());
        if (bean != null) {
            // 当前集合对象，可能是：json字符串/普通字符串/json对象
            // 获取到bean了，不应该是普通对象
            return encryptOrDecryptCollectionJson(collection, bean.getRule(), true, true);
        }
        // 未获取到bean
        // 当前集合对象，可能是：json字符串/普通字符串/普通对象
        // 不应该是json对象
        Object value = collection.stream().filter(Objects::nonNull).findFirst().orElse(null);
        if (value == null) {
            return collection;
        }
        if (value instanceof String) {
            // json字符串/普通字符串
            DbEncryptColumnRule rule = getCollectionStringRule(collection, paramsRuleMap, classAndMethod);
            return encryptOrDecryptCollectionString(collection, rule, true, true);
        }
        // 普通对象
        return encryptCollectionObject(collection, paramsRuleMap);
    }

    private Object decrypt(Object proceed, MappedStatement mappedStatement, BoundSql boundSql, DbEncryptTableRule columns, String classAndMethod) throws BizException {
        List<ResultMap> resultMaps = mappedStatement.getResultMaps();
        if (CollectionUtils.isEmpty(resultMaps)) {
            // 无需解密
            return proceed;
        }
        long start = System.currentTimeMillis();
        try {
            // 获取sql中的所有字段
            List<String> columnList = getResultColumns(boundSql.getSql());
            // 获取对象名称里对应的加密规则
            Map<String, DbEncryptColumnRule> propertyRuleMap = getPropertyRuleMap(resultMaps, columns.getColumns(), columnList);
            if (MapUtils.isEmpty(propertyRuleMap)) {
                return proceed;
            }
            // 解密结果
            return decrypt(proceed, propertyRuleMap, classAndMethod);
        } finally {
            log.info("DbEncryptionPlugin, decrypt, boundSql={}, cost={}ms", boundSql.getSql(), System.currentTimeMillis() - start);
        }
    }

    private Object decrypt(Object result, Map<String, DbEncryptColumnRule> propertyRuleMap, String classAndMethod) throws BizException {
        if ((result instanceof String) && propertyRuleMap.size() != 1) {
            throw new BizException(classAndMethod + " 查询一个字段，但是有多个字段要解密");
        }
        if (result instanceof String) {
            // json字符串/普通字符串
            DbEncryptColumnRule firstRule = getFirstRule(propertyRuleMap);
            return encryptOrDecryptJsonString((String) result, firstRule, false);
        }
        if (result instanceof Collection) {
            return decryptCollection(toCollection(result), propertyRuleMap, classAndMethod);
        }
        if (result instanceof Map) {
            // 暂不支持map
            return result;
        }
        // 对象类型
        return decryptObject(result, propertyRuleMap);
    }

    private Collection<Object> decryptCollection(Collection<Object> collection, Map<String, DbEncryptColumnRule> propertyRuleMap, String classAndMethod) throws BizException {
        if (CollectionUtils.isEmpty(collection)) {
            return collection;
        }
        Object value = collection.stream().filter(Objects::nonNull).findFirst().orElse(null);
        if (value == null) {
            return collection;
        }
        if (isPrimitive(value.getClass())) {
            return collection;
        }
        if ((value instanceof String) && propertyRuleMap.size() != 1) {
            throw new BizException(classAndMethod + " 查询一个字段集合，但是有多个字段要解密");
        }
        if (value instanceof String) {
            // 普通字符串/json对象字符串/json数组字符串
            DbEncryptColumnRule firstRule = getFirstRule(propertyRuleMap);
            return encryptOrDecryptCollectionString(collection, firstRule, false, true);
        }
        if (value instanceof List) {
            return collection;
        }
        if (value instanceof Map) {
            return collection;
        }
        // 普通对象
        for (Object v : collection) {
            decryptObject(v, propertyRuleMap);
        }
        return collection;
    }

    private DbEncryptColumnRule getFirstRule(Map<String, DbEncryptColumnRule> propertyRuleMap) {
        return propertyRuleMap.values().stream().filter(Objects::nonNull).findFirst().orElse(null);
    }

    public static List<Field> getNotStaticField(Object obj) {
        List<Field> fieldList = new ArrayList<>();
        Class<?> clazz = obj.getClass();
        while (clazz != null) {
            Field[] fields = clazz.getDeclaredFields();
            for (Field field : fields) {
                if (Modifier.isStatic(field.getModifiers())) {
                    continue;
                }
                fieldList.add(field);
            }
            clazz = clazz.getSuperclass();
        }
        return fieldList;
    }

    private Object decryptObject(Object result, Map<String, DbEncryptColumnRule> propertyRuleMap) {
        try {
            List<Field> collect = getNotStaticField(result);
            for (Field field : collect) {
                String name = field.getName();
                DbEncryptColumnRule rule = propertyRuleMap.get(name);
                if (rule == null) {
                    continue;
                }
                encryptOrDecryptObject(rule, field, result, false);
            }
            return result;
        } catch (IllegalAccessException e) {
            return result;
        }
    }

    private Map<String, DbEncryptColumnRule> getPropertyRuleMap(List<ResultMap> resultMaps, Map<String, DbEncryptColumnRule> columns, List<String> columnList) {
        Map<String, DbEncryptColumnRule> propertyRuleMap = new HashMap<>();
        for (ResultMap resultMap : resultMaps) {
            List<ResultMapping> propertyResultMappings = resultMap.getPropertyResultMappings();
            if (CollectionUtils.isNotEmpty(propertyResultMappings)) {
                putPropertyResultMappings(propertyResultMappings, columns, propertyRuleMap);
                continue;
            }
            // 根据sql中的查询字段构建
            putByColumn(columnList, columns, propertyRuleMap);
        }
        return propertyRuleMap;
    }

    private void putByColumn(List<String> columnList, Map<String, DbEncryptColumnRule> columns, Map<String, DbEncryptColumnRule> propertyRuleMap) {
        if (CollectionUtils.isEmpty(columnList)) {
            return;
        }
        for (String column : columnList) {
            Pair<String, String> pair = parseColumn(column);
            DbEncryptColumnRule rule = columns.get(pair.getLeft());
            if (rule == null) {
                continue;
            }
            propertyRuleMap.put(pair.getRight(), rule);
        }
    }

    private Pair<String, String> parseColumn(String column) {
        int spaceIndex = column.indexOf(DbEncryptionConstant.SPACE);
        String columnName;
        if (spaceIndex < 0) {
            columnName = column;
        } else {
            columnName = column.substring(0, spaceIndex);
        }
        int lastSpaceIndex = column.lastIndexOf(DbEncryptionConstant.SPACE);
        String property;
        if (lastSpaceIndex < 0) {
            // 没有空格，直接下划线转驼峰
            property = toHump(column);
        } else {
            // 有空格，取最后的一个单词
            property = column.substring(lastSpaceIndex + 1);
        }
        return Pair.of(columnName, property);
    }

    private String toHump(String column) {
        String[] split = column.split(DbEncryptionConstant.UNDERLINE);
        if (split.length == 1) {
            return split[0];
        }
        StringBuilder property = new StringBuilder();
        for (int i = 0; i < split.length; i++) {
            String part = split[i];
            if (i == 0) {
                property.append(part);
            } else {
                property.append(part.substring(0, 1).toUpperCase()).append(part.substring(1).toLowerCase());
            }
        }
        return property.toString();
    }

    private List<String> getResultColumns(String sql) {
        int selectIndex = sql.toLowerCase().indexOf(LOWER_SELECT);
        int fromIndex = sql.toLowerCase().indexOf(LOWER_FROM);
        if (selectIndex < 0 || fromIndex < 0 || selectIndex + LOWER_SELECT.length() >= fromIndex) {
            return Collections.emptyList();
        }
        String[] split = sql.substring(selectIndex + LOWER_SELECT.length(), fromIndex).split(DbEncryptionConstant.COMMA);
        return Arrays.stream(split).map(String::trim).filter(StringUtils::isNotBlank).collect(Collectors.toList());
    }

    private void putPropertyResultMappings(List<ResultMapping> propertyResultMappings, Map<String, DbEncryptColumnRule> columns, Map<String, DbEncryptColumnRule> propertyRuleMap) {
        for (ResultMapping resultMapping : propertyResultMappings) {
            String column = resultMapping.getColumn();
            DbEncryptColumnRule rule = columns.get(column);
            if (rule == null) {
                continue;
            }
            propertyRuleMap.put(resultMapping.getProperty(), rule);
        }
    }

    private Collection<Object> encryptOrDecryptCollection(Collection<Object> collection, DbEncryptColumnRule rule, boolean isEncrypt) {
        if (CollectionUtils.isEmpty(collection)) {
            return collection;
        }
        Object value = collection.stream().filter(Objects::nonNull).findFirst().orElse(null);
        if (value == null) {
            return collection;
        }
        if (value instanceof String) {
            return encryptOrDecryptCollectionString(collection, rule, isEncrypt, false);
        }
        if (isPrimitive(value.getClass())) {
            // 基本类型
            return collection;
        }
        if (value instanceof Collection || value instanceof Map) {
            // 集合类型，暂不支持
            return collection;
        }
        if (CollectionUtils.isEmpty(rule.getNames())) {
            return collection;
        }
        return encryptOrDecryptCollectionJsonObj(toCollection(collection), rule, isEncrypt);
    }

    private Collection<Object> encryptCollection(Collection<Object> collection, Map<String, PropertyValueBean> paramsRuleMap, String classAndMethod) throws BizException {
        if (CollectionUtils.isEmpty(collection)) {
            return collection;
        }
        Object value = collection.stream().filter(Objects::nonNull).findFirst().orElse(null);
        if (value == null) {
            return collection;
        }
        if (value instanceof Collection || value instanceof Map) {
            // 集合类型，暂不支持
            return collection;
        }
        if (isPrimitive(value.getClass())) {
            // 基本类型
            return collection;
        }
        if (value instanceof String) {
            DbEncryptColumnRule rule = getCollectionStringRule(collection, paramsRuleMap, classAndMethod);
            return encryptOrDecryptCollectionString(collection, rule, true, true);
        }
        // 对象类型
        return encryptCollectionObject(collection, paramsRuleMap);
    }

    private Collection<Object> encryptCollectionObject(Collection<Object> collection, Map<String, PropertyValueBean> paramsRuleMap) {
        for (Object v: collection) {
            encryptObject(v, paramsRuleMap);
        }
        return collection;
    }

    private Object encryptObject(Object obj, Map<String, PropertyValueBean> paramsRuleMap) {
        if (obj == null) {
            return null;
        }
        try {
            List<Field> fields = getNotStaticField(obj);
            for (Field field : fields) {
                encryptObject(field, obj, paramsRuleMap);
            }
            return obj;
        } catch (IllegalAccessException e) {
            return obj;
        }
    }

    private void encryptObject(Field field, Object obj, Map<String, PropertyValueBean> paramsRuleMap) throws IllegalAccessException {
        String name = field.getName();
        PropertyValueBean bean = paramsRuleMap.get(name);
        if (bean == null) {
            return;
        }
        DbEncryptColumnRule rule = bean.getRule();
        encryptOrDecryptObject(rule, field, obj, true);
    }

    private void encryptOrDecryptObject(DbEncryptColumnRule rule, Field field, Object obj, boolean isEncrypt) throws IllegalAccessException {
        if (rule == null) {
            return;
        }
        fieldSetAccessible(field);
        Object value = field.get(obj);
        Pair<Boolean, Object> pair = encryptOrDecryptJson(value, rule, isEncrypt);
        if (BooleanUtils.isNotTrue(pair.getLeft())) {
            // 没有加解密
            return;
        }
        fieldSet(obj, field, pair.getRight());
    }

    private Pair<Boolean, Object> encryptOrDecryptJson(Object value, DbEncryptColumnRule rule, boolean isEncrypt) {
        if (value == null) {
            return Pair.of(false, null);
        }
        if (value instanceof String) {
            return Pair.of(true, encryptOrDecryptJsonString((String) value, rule, isEncrypt));
        }
        if (isPrimitive(value.getClass()) || value instanceof Map) {
            return Pair.of(false, value);
        }
        if (value instanceof Collection) {
            return Pair.of(true, encryptOrDecryptCollection(toCollection(value), rule, isEncrypt));
        }
        if (CollectionUtils.isNotEmpty(rule.getNames())) {
            // 加密对象是json
            return Pair.of(true, encryptOrDecryptJsonObj(value, rule, isEncrypt));
        }
        return Pair.of(false, value);
    }

    private String encryptOrDecryptJsonString(String value, DbEncryptColumnRule rule, boolean isEncrypt) {
        if (rule == null) {
            return value;
        }
        if (CollectionUtils.isEmpty(rule.getNames())) {
            // 非json
            return encryptOrDecryptString(isEncrypt, value, rule);
        }
        if (value.startsWith(DbEncryptionConstant.LEFT_MIDDLE_BRACKET)) {
            // 值是json数组
            return encryptOrDecryptJsonArray(value, rule, isEncrypt);
        }
        if (value.startsWith(DbEncryptionConstant.LEFT_CURLY_BRACKET)) {
            // 值是json对象
            return encryptOrDecryptJsonObject(value, rule, isEncrypt);
        }
        return encryptOrDecryptString(isEncrypt, value, rule);
    }

    private String encryptOrDecryptJsonArray(String value, DbEncryptColumnRule rule, boolean isEncrypt) {
        JSONArray array = JSON.parseArray(value);
        Object o = encryptOrDecryptCollectionJson(array, rule, isEncrypt, false);
        return JSON.toJSONString(o);
    }

    private String encryptOrDecryptJsonObject(String value, DbEncryptColumnRule rule, boolean isEncrypt) {
        JSONObject jsonObject = JSON.parseObject(value);
        encryptOrDecryptJsonMap(jsonObject, rule, isEncrypt);
        return jsonObject.toJSONString();
    }

    private @Nullable Collection<Object> encryptOrDecryptCollectionJson(Collection<Object> value, DbEncryptColumnRule rule, boolean isEncrypt, boolean maybeJson) {
        if (value == null) {
            return null;
        }
        // json是数组
        Object first = value.stream().filter(Objects::nonNull).findFirst().orElse(null);
        if (first == null) {
            return value;
        }
        if (first instanceof String) {
            // 字符串数组
            return encryptOrDecryptCollectionString(value, rule, isEncrypt, maybeJson);
        }
        if (isPrimitive(first.getClass())) {
            // 基础类型不支持
            return value;
        }
        // json对象数组
        return encryptOrDecryptCollectionJsonObj(value, rule, isEncrypt);
    }

    private void encryptOrDecryptJsonMap(JSONObject jsonObject, DbEncryptColumnRule rule, boolean isEncrypt) {
        if (CollectionUtils.isEmpty(rule.getNames()) || MapUtils.isEmpty(jsonObject)) {
            return;
        }
        for (Map.Entry<String, Object> entry : jsonObject.entrySet()) {
            String name = entry.getKey();
            if (!rule.getNames().contains(name) || entry.getValue() == null) {
                continue;
            }
            jsonObject.put(name, encryptOrDecryptString(isEncrypt, entry.getValue().toString(), rule));
        }
    }

    private Object encryptOrDecryptJsonObj(Object value, DbEncryptColumnRule rule, boolean isEncrypt) {
        if (CollectionUtils.isEmpty(rule.getNames()) || value == null) {
            return value;
        }
        try {
            List<Field> collect = getNotStaticField(value);
            for (Field field : collect) {
                encryptJsonObjField(value, field, rule, isEncrypt);
            }
            return value;
        } catch (IllegalAccessException e) {
            return value;
        }
    }

    private void encryptJsonObjField(Object value, Field field, DbEncryptColumnRule rule, boolean isEncrypt) throws IllegalAccessException {
        String name = field.getName();
        if (!rule.getNames().contains(name)) {
            return;
        }
        fieldSetAccessible(field);
        Object v = field.get(value);
        if (v == null) {
            return;
        }
        if (!(v instanceof String)) {
            return;
        }
        fieldSet(value, field, encryptOrDecryptString(isEncrypt, (String) v, rule));
    }

    private String encryptOrDecryptString(boolean isEncrypt, String v, DbEncryptColumnRule rule) {
        if (isEncrypt) {
            return EncryptionDecryptionAdapter.encryptString(v, rule, secretMap);
        }
        return EncryptionDecryptionAdapter.decryptString(v, secretMap);
    }

    private DbEncryptColumnRule getCollectionStringRule(Collection<Object> collection, Map<String, PropertyValueBean> paramsRuleMap, String classAndMethod) throws BizException {
        List<PropertyValueBean> beanList = paramsRuleMap.values().stream().filter(bean -> {
            if (!bean.isFrch() || bean.isFrchObj()) {
                return false;
            }
            return isSameValueCollection(collection, bean.getValueMap());
        }).collect(Collectors.toList());
        if (beanList.isEmpty()) {
            return null;
        }
        if (beanList.size() != 1) {
            throw new BizException(classAndMethod + " 对象入参，有多个集合值相同");
        }
        PropertyValueBean bean = beanList.get(0);
        return bean.getRule();
    }

    private Collection<Object> encryptOrDecryptCollectionString(Collection<Object> collection, DbEncryptColumnRule rule, boolean isEncrypt, boolean maybeJson) {
        if (rule == null) {
            return collection;
        }
        Stream<String> stream = collection.stream().map(v -> maybeJson ? encryptOrDecryptJsonString((String) v, rule, isEncrypt) : encryptOrDecryptString(isEncrypt, (String) v, rule));
        if (collection instanceof Set) {
            return stream.collect(Collectors.toSet());
        } else {
            return stream.collect(Collectors.toList());
        }
    }

    private Collection<Object> encryptOrDecryptCollectionJsonObj(Collection<Object> collection, DbEncryptColumnRule rule, boolean isEncrypt) {
        for (Object v : collection) {
            if (v instanceof JSONObject) {
                encryptOrDecryptJsonMap((JSONObject) v, rule, isEncrypt);
            } else {
                encryptOrDecryptJsonObj(v, rule, isEncrypt);
            }
        }
        return collection;
    }

    private boolean isSameValueCollection(Collection<Object> collection, Map<Integer, Object> valueMap) {
        if (CollectionUtils.isEmpty(collection) || valueMap == null || valueMap.isEmpty() || collection.size() != valueMap.size()) {
            return false;
        }
        int index = 0;
        for (Object v : collection) {
            Object o = valueMap.get(index++);
            if (!Objects.equals(v, o)) {
                return false;
            }
        }
        return true;
    }

    protected boolean isPrimitive(Class<?> parameterType) {
        return parameterType.isPrimitive() || Number.class.isAssignableFrom(parameterType);
    }

    private MetaObject getMetaObject(MappedStatement mappedStatement, String classAndMethod, Object parameterObject) throws BizException {
        Configuration configuration = mappedStatement.getConfiguration();
        if (configuration == null) {
            throw new BizException(classAndMethod + " configuration为null");
        }
        return configuration.newMetaObject(parameterObject);
    }

    private String getProperty(Integer index, List<ParameterMapping> parameterMappings, BoundSql boundSql) {
        if (index == null || index < 0 || index >= parameterMappings.size()) {
            log.warn("DbEncryptionPlugin, index error, index={}, sql={}, parameterMappings={}", index, boundSql.getSql(), JSON.toJSONString(parameterMappings));
            return null;
        }
        ParameterMapping parameterMapping = parameterMappings.get(index);
        if (parameterMapping == null) {
            log.warn("DbEncryptionPlugin, parameterMapping is null, index={}, sql={}, parameterMappings={}", index, boundSql.getSql(), JSON.toJSONString(parameterMappings));
            return null;
        }
        return parameterMapping.getProperty();
    }

    private void putNotFrch(Map<String, PropertyValueBean> paramRuleMap, String property, DbEncryptColumnRule rule) {
        PropertyValueBean bean = new PropertyValueBean();
        bean.setRule(rule);
        paramRuleMap.put(property, bean);
    }

    private void putFrchObj(Map<String, PropertyValueBean> paramRuleMap, String property, DbEncryptColumnRule rule, int separatorIndex) {
        String name = property.substring(separatorIndex + 1);
        PropertyValueBean bean = paramRuleMap.computeIfAbsent(name, k -> buildPropertyValueBean(rule, null));
        bean.setFrch(true);
        bean.setFrchObj(true);
    }

    private void putFrchNotObj(Map<String, PropertyValueBean> paramRuleMap, String property, DbEncryptColumnRule rule, MetaObject metaObject, BoundSql boundSql) {
        String[] split = property.replace(FRCH, "").split("_");
        String name = split[0];
        int index = Integer.parseInt(split[1]);
        PropertyValueBean bean = paramRuleMap.computeIfAbsent(name, k -> buildPropertyValueBean(rule, new HashMap<>()));
        bean.setFrch(true);
        bean.setFrchObj(false);
        Object value = getValue(property, metaObject, boundSql);
        if (value != null) {
            bean.getValueMap().put(index, value);
        }
    }

    private Object getValue(String property, MetaObject metaObject, BoundSql boundSql) {
        if (metaObject.hasGetter(property)) {
            return metaObject.getValue(property);
        } else if (boundSql.hasAdditionalParameter(property)) {
            return boundSql.getAdditionalParameter(property);
        } else {
            return null;
        }
    }

    private PropertyValueBean buildPropertyValueBean(DbEncryptColumnRule rule, Map<Integer, Object> valueMap) {
        PropertyValueBean bean = new PropertyValueBean();
        bean.setRule(rule);
        bean.setValueMap(valueMap);
        return bean;
    }

    private Map<Object, Object> toMap(Object value) {
        return (Map<Object, Object>) value;
    }

    private Collection<Object> toCollection(Object value) {
        return (Collection<Object>)value;
    }

    private void fieldSetAccessible(Field field) {
        field.setAccessible(true);
    }

    private void fieldSet(Object obj, Field field, Object value) throws IllegalAccessException {
        field.set(obj, value);
    }

    @Override
    public Object plugin(Object target) {
        return Plugin.wrap(target, this);
    }

    @Override
    public void setProperties(Properties properties) {
        putTables(properties);
        putSecretMap(properties);
    }

    private void putSecretMap(Properties properties) {
        String secretMapJson = properties.getProperty("secretMap");
        if (secretMapJson == null || secretMapJson.isEmpty()) {
            return;
        }
        try {
            Map<String, String> propertiesSecretMap = JSON.parseObject(secretMapJson, new TypeReference<Map<String, String>>() {});
            if (propertiesSecretMap == null || propertiesSecretMap.isEmpty()){
                return;
            }
            secretMap.putAll(propertiesSecretMap);
        } catch (Exception e) {
            log.error("DbEncryptionPlugin, add secretMap error, e:", e);
        }
    }

    private void putTables(Properties properties) {
        String tablesJson = properties.getProperty("tables");
        if (tablesJson == null || tablesJson.isEmpty()) {
            return;
        }
        try {
            Map<String, DbEncryptTableRule> propertiesTables = JSON.parseObject(tablesJson, new TypeReference<Map<String, DbEncryptTableRule>>() {});
            if (propertiesTables == null || propertiesTables.isEmpty()){
                return;
            }
            tables.putAll(propertiesTables);
        } catch (Exception e) {
            log.error("DbEncryptionPlugin, add tables error, e:", e);
        }
    }
}
