package cn.com.duiba.duiba.base.service.api.mybatis.plugins.handler.encrypt;

import cn.com.duiba.duiba.base.service.api.mybatis.plugins.bean.DbEncryptionConstant;
import cn.com.duiba.duiba.base.service.api.mybatis.plugins.config.DbEncryptColumnRule;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import java.util.HashMap;
import java.util.Map;

/**
 * @author lizhi
 * @date 2024/9/26 10:32
 */
@Slf4j
public class EncryptionDecryptionAdapter {
    
    private static final Map<String, EncryptionDecryptionHandler> handlerMap = new HashMap<>();
    
    private EncryptionDecryptionAdapter() {}

    public static void register(EncryptionDecryptionHandler handler) {
        handlerMap.put(handler.encryptorEnum().name(), handler);
    }

    private static EncryptionDecryptionHandler get(String encryptor) {
        EncryptionDecryptionHandler handler = handlerMap.get(encryptor);
        if (handler == null) {
            log.info("EncryptionDecryption, handler is null, encryptor={}", encryptor);
            return null;
        }
        return handler;
    }

    /**
     * 加密字符串
     * @param value 字符串
     * @param rule 加密规则
     * @param secretMap key：加密方式及版本，value密钥
     * @return 加密后的字符串
     */
    public static String encryptString(String value, DbEncryptColumnRule rule, Map<String, String> secretMap) {
        if (StringUtils.isBlank(value) || rule == null || MapUtils.isEmpty(secretMap)) {
            return value;
        }
        if (BooleanUtils.isNotTrue(rule.getStart())) {
            return value;
        }
        EncryptionDecryptionHandler handler = get(rule.getEncryptor());
        if (handler == null) {
            return value;
        }
        try {
            Pair<String, Integer> pair = parseEncryptorVersion(value);
            if (pair.getLeft() != null && pair.getRight() != null && get(pair.getLeft()) != null) {
                // 已经是密文
                return value;
            }
            String secretKey = getSecretKey(rule.getEncryptor(), rule.getVersion());
            String secret = secretMap.get(secretKey);
            if (StringUtils.isBlank(secret)) {
                log.warn("EncryptionDecryption, encrypt secret is blank, encryptor={}, version={}", rule.getEncryptor(), rule.getVersion());
                return value;
            }
            String encrypt = handler.encrypt(value, secret);
            if (StringUtils.isBlank(encrypt)) {
                return value;
            }
            return secretKey + DbEncryptionConstant.UNDERLINE + encrypt;
        } catch (Exception e) {
            log.error("EncryptionDecryption, encrypt error, encryptor={}, version={}", rule.getEncryptor(), rule.getVersion());
            return value;
        }
    }
    
    private static Pair<String, Integer> parseEncryptorVersion(String value) {
        try {
            int index = value.indexOf(DbEncryptionConstant.UNDERLINE);
            if (index <= 0) {
                return Pair.of(null, null);
            }
            String encryptor = value.substring(0, 1);
            Integer version = Integer.parseInt(value.substring(1, index));
            return Pair.of(encryptor, version);
        } catch (Exception e) {
            return Pair.of(null, null);
        }
    }

    /**
     * 加密字符串
     * @param value 字符串
     * @param secretMap key：加密方式及版本，value密钥
     * @return 加密后的字符串
     */
    public static String decryptString(String value, Map<String, String> secretMap) {
        if (StringUtils.isBlank(value) || MapUtils.isEmpty(secretMap)) {
            return value;
        }
        Pair<String, Integer> pair = parseEncryptorVersion(value);
        if (StringUtils.isBlank(pair.getLeft()) || pair.getRight() == null) {
            return value;
        }
        EncryptionDecryptionHandler handler = get(pair.getLeft());
        if (handler == null) {
            return value;
        }
        try {
            String secretKey = getSecretKey(pair.getLeft(), pair.getRight());
            String secret = secretMap.get(secretKey);
            if (StringUtils.isBlank(secret)) {
                log.warn("EncryptionDecryption, decrypt secret is blank, encryptor={}, version={}", pair.getLeft(), pair.getRight());
                return value;
            }
            String decrypt = handler.decrypt(value.substring((secretKey + DbEncryptionConstant.UNDERLINE).length()), secret);
            if (StringUtils.isBlank(decrypt)) {
                return value;
            }
            return decrypt;
        } catch (Exception e) {
            log.error("EncryptionDecryption, decrypt error, encryptor={}, version={}", pair.getLeft(), pair.getRight());
            return value;
        }
    }


    private static String getSecretKey(String encryptor, Integer version) {
        return encryptor + version;
    }
}
