package cn.com.duiba.duiba.qutui.center.api.dto.wxmessage.wrapper;

import cn.com.duiba.duiba.qutui.center.api.dto.wxmessage.BaseMessageDto;
import cn.com.duiba.duiba.qutui.center.api.dto.wxmessage.MessageUserDto;
import cn.com.duiba.duiba.qutui.center.api.dto.wxmessage.innerlog.WxSendEmbedDto;
import cn.com.duiba.duiba.qutui.center.api.dto.wxmessage.pushtask.PushTaskDto;
import cn.com.duiba.duiba.qutui.center.api.dto.wxmessage.template.TemplateMessageDto;
import cn.com.duiba.duiba.qutui.center.api.enums.DegradedTypeEnum;
import cn.com.duiba.duiba.qutui.center.api.enums.MessageTypeEnum;
import lombok.extern.slf4j.Slf4j;

import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * Created by dugq on 2020-03-01.
 */
@Slf4j
public class WxMessageWrapperBuilder {

    /**
     * 消息类型 {@link cn.com.duiba.duiba.qutui.center.api.enums.MessageTypeEnum}
     */
    private Integer messageType;

    /**
     * 消息体
     */
    @NotNull(message = "消息体不能为空")
    private BaseMessageDto messageDto;

    /**
     * 优先选择user对象的oaID
     * 接受消息的目标公众号
     */
    @NotNull(message = "接受消息的公众号不能为空")
    private Long oaId;

    /**
     * 推送事件
     */
    private WxSendEmbedDto wxSendEmbedBean;

    /**
     * 降级策略
     * @see cn.com.duiba.duiba.qutui.center.api.enums.DegradedTypeEnum
     */
    @NotNull(message = "降级策略不能为空")
    private Integer degradedType = DegradedTypeEnum.DISCARD.getCode();

    /**
     * 是否需要推送结果
     * 如果该参数是true  会通过mq 发送推送结果到livecenter
     * 统计推送成功与失败
     */
    private Boolean needPushResult;

    /**
     * 推送任务对象
     */
    private PushTaskDto pushTaskDto;

    /**
     * 降级消息
     * 可以选择其他任意类型的消息
     * 如切换成模版消息，发送其他公众号或者小程序等等
     */
    private WxMessageWrapperDto degradedMessageWrapperDto;

    public WxMessageWrapperBuilder setOaId(Long oaId){
        if (Objects.isNull(oaId) || oaId<1){
            throw new RuntimeException("oaId 非法");
        }
        this.oaId = oaId;
        return this;
    }

    public WxMessageWrapperBuilder setNeedPushResult(Boolean needPushResult){
        this.needPushResult = needPushResult;
        return this;
    }

    public WxMessageWrapperBuilder setPushTaskDto(PushTaskDto pushTaskDto){
        if(Objects.isNull(pushTaskDto) || Objects.isNull(pushTaskDto.getTaskId()) || Objects.isNull(pushTaskDto.getPushTaskType())){
            throw new RuntimeException("推送任务对象错误");
        }
        this.pushTaskDto = pushTaskDto;
        return this;
    }

    /**
     * 设置消息体。
     * @param messageType 消息类型
     * @param messageDto 消息体
     * @return
     */
    public WxMessageWrapperBuilder setMessageDto(Integer messageType,BaseMessageDto messageDto){
        MessageTypeEnum.getByCode(messageType);
        this.messageType = messageType;
        this.messageDto = messageDto;
        return this;
    }

    /**
     * 设置推送日志打印参数
     * @param pushEventType push_id，与BI沟通获取
     * @param customMap 其他需要记录的参数，没有填null
     * @return
     */
    public WxMessageWrapperBuilder setEmbedParam(String pushEventType, Map<String, Object> customMap){
        WxSendEmbedDto wxSendEmbedDto = new WxSendEmbedDto(pushEventType,customMap);
        this.wxSendEmbedBean = wxSendEmbedDto;
        return this;
    }

    /**
     * 设置降级消息，灵活配置，发给别人，别的公众号都可以。强制不可以再次降级。
     * 降级如果时模版消息，推荐使用 {@link #setDegradedTemplateMessage} 方法。
     * @param degradedMessageWrapperDto 降级消息还是使用此 builder 重新构建一个完整的消息配置
     * @return
     */
    public WxMessageWrapperBuilder setDegradedMessage(WxMessageWrapperDto degradedMessageWrapperDto){
        degradedMessageWrapperDto.setDegradedType(DegradedTypeEnum.DISCARD.getCode());
        degradedMessageWrapperDto.setDegradedMessageWrapperDto(null);
        this.degradedType = DegradedTypeEnum.OTHER_MESSAGE.getCode();
        this.degradedMessageWrapperDto = degradedMessageWrapperDto;
        return this;
    }

    /**
     * 设置降级模版消息
     * @param templateMessageDto 模版消息
     * @param messageUserDto 降级如果还是发给这个人，为null这还是发送给这个人的这个公众号，切换公众号推送，或者直接发送给别人时必填。
     * @return
     */
    public WxMessageWrapperBuilder setDegradedTemplateMessage(TemplateMessageDto templateMessageDto, MessageUserDto messageUserDto){
        SingleWxMessageWrapperDto degradedMessageWrapperDto = new SingleWxMessageWrapperDto();
        degradedMessageWrapperDto.setMessageType(MessageTypeEnum.TEMPLATE.getCode());
        degradedMessageWrapperDto.setMessageDto(templateMessageDto);
        degradedMessageWrapperDto.setDegradedType(DegradedTypeEnum.DISCARD.getCode());
        degradedMessageWrapperDto.setDegradedMessageWrapperDto(null);
        degradedMessageWrapperDto.setUser(messageUserDto);
        this.degradedType = DegradedTypeEnum.OTHER_MESSAGE.getCode();
        this.degradedMessageWrapperDto = degradedMessageWrapperDto;
        return this;
    }

    /**
     * 构建批量消息体
     * @param userList 消息发送的批量用户列表
     * @return
     */
    public BatchWxMessageWrapperDto createBatchMessage(List<MessageUserDto> userList){
        BatchWxMessageWrapperDto batchWxMessageWrapperDto = new BatchWxMessageWrapperDto();
        batchWxMessageWrapperDto.setWxSendEmbedBean(wxSendEmbedBean);
        batchWxMessageWrapperDto.setMessageDto(messageDto);
        batchWxMessageWrapperDto.setOaId(oaId);
        batchWxMessageWrapperDto.setMessageType(messageType);
        batchWxMessageWrapperDto.setDegradedType(degradedType);
        batchWxMessageWrapperDto.setDegradedMessageWrapperDto(degradedMessageWrapperDto);
        batchWxMessageWrapperDto.setNeedPushResult(needPushResult);
        batchWxMessageWrapperDto.setPushTaskDto(pushTaskDto);
        batchWxMessageWrapperDto.setUserList(userList);
        return batchWxMessageWrapperDto;
    }

    /**
     * 构建批量消息体
     * @param user 消息发送的用户
     * @return
     */
    public SingleWxMessageWrapperDto createSingleMessage(MessageUserDto user){
        SingleWxMessageWrapperDto singleWxMessageWrapperDto = new SingleWxMessageWrapperDto();
        singleWxMessageWrapperDto.setWxSendEmbedBean(wxSendEmbedBean);
        singleWxMessageWrapperDto.setMessageDto(messageDto);
        singleWxMessageWrapperDto.setOaId(oaId);
        singleWxMessageWrapperDto.setMessageType(messageType);
        singleWxMessageWrapperDto.setDegradedType(degradedType);
        singleWxMessageWrapperDto.setDegradedMessageWrapperDto(degradedMessageWrapperDto);
        singleWxMessageWrapperDto.setNeedPushResult(needPushResult);
        singleWxMessageWrapperDto.setPushTaskDto(pushTaskDto);
        singleWxMessageWrapperDto.setUser(user);
        return singleWxMessageWrapperDto;
    }

}
