package cn.com.duiba.goods.center.api.remoteservice.dto.item;

import java.io.Serializable;

/**
 * 商品查询参数
 * <p>使用方法：</p>
 * <p><b>ItemQueries.empty().with***().with***()...</b></p>
 * 或者：
 * <p><b>ItemQueries.all().without***().without***()...</b></p>
 * 或者混合使用
 * @Author xuwei
 * @Date 2018/11/23
 */
public class ItemQueries implements Serializable {
    private static final long serialVersionUID = -47787140896534918L;

    //图文详情描述
    private static final int OFFSET_DESC = 1;
    //限购信息
    private static final int OFFSET_LIMIT = 1 << 1;
    //sku信息
    private static final int OFFSET_SKU = 1 << 2;
    //积分换购信息
    private static final int OFFSET_CREDITS = 1 << 3;
    //sku积分换购信息
    private static final int OFFSET_CREDITS_SKU = 1 << 4;
    //供货价
    private static final int OFFSET_SUPPLY_PRICE = 1 << 5;

    //商品查询信息bitmap
    private int value;
    //是否查询缓存
    private boolean queryCache;

    /**
     * @see ItemQueries#empty()
     */
    public ItemQueries() {
    }

    /**
     * @param all        查询所有
     * @param queryCache 查询缓存
     */
    private ItemQueries(boolean all, boolean queryCache) {
        if (all) {
            value = Integer.MAX_VALUE;
        }
        this.queryCache = queryCache;
    }

    /**
     * 获取一个空查询条件，只查询商品主信息
     *
     * @return
     */
    public static ItemQueries empty() {
        return new ItemQueries();
    }

    /**
     * 获取一个完整查询条件，查询商品所有信息
     * @return
     */
    public static ItemQueries all() {
        return new ItemQueries(true, false);
    }

    /**
     * 是否只查询商品主信息
     *
     * @return
     */
    public boolean isEmpty() {
        return value == 0;
    }

    /**
     * 从缓存中获取数据（缓存存在的情况下）
     *
     * @return
     */
    public ItemQueries queryFromCache() {
        queryCache = true;
        return this;
    }

    /**
     * 不从缓存中获取数据
     *
     * @return
     */
    public ItemQueries queryWithoutCace() {
        queryCache = false;
        return this;
    }

    /**
     * 包含图文描述信息
     * @return
     */
    public ItemQueries withDescConfig() {
        return with(OFFSET_DESC);
    }

    /**
     * 包含限购信息
     * @return
     */
    public ItemQueries withLimitConfig() {
        return with(OFFSET_LIMIT);
    }

    /**
     * 包含SKU销售信息
     * @return
     */
    public ItemQueries withSku() {
        return with(OFFSET_SKU);
    }

    /**
     * 包含积分换购信息
     * @return
     */
    public ItemQueries withCredits() {
        return with(OFFSET_CREDITS);
    }

    /**
     * 包含SKU积分换购信息
     * @return
     */
    public ItemQueries withCreditsSku() {
        return with(OFFSET_CREDITS_SKU);
    }

    /**
     * 包含供货价
     *
     * @return
     */
    public ItemQueries withSupplyPrice() {
        return with(OFFSET_SUPPLY_PRICE);
    }

    private ItemQueries with(int offset) {
        value |= offset;
        return this;
    }

    /**
     * 不包含图文描述信息
     * @return
     */
    public ItemQueries withoutDescConfig() {
        return without(OFFSET_DESC);
    }

    /**
     * 不包含限购信息
     * @return
     */
    public ItemQueries withoutLimitConfig() {
        return without(OFFSET_LIMIT);
    }

    /**
     * 不包含SKU销售信息
     * @return
     */
    public ItemQueries withoutSku() {
        return without(OFFSET_SKU);
    }

    /**
     * 不包含积分换购信息
     * @return
     */
    public ItemQueries withoutCredits() {
        return without(OFFSET_CREDITS);
    }

    /**
     * 不包含SKU积分换购信息
     * @return
     */
    public ItemQueries withoutCreditsSku() {
        return without(OFFSET_CREDITS_SKU);
    }

    /**
     * 不包含供货价
     *
     * @return
     */
    public ItemQueries withoutSupplyPrice() {
        return without(OFFSET_SUPPLY_PRICE);
    }

    private ItemQueries without(int offset) {
        value &= ~offset;
        return this;
    }

    /**
     * 是否包含图文描述信息
     * @return
     */
    public boolean hasDescConfig() {
        return has(OFFSET_DESC);
    }

    /**
     * 是否包含限购信息
     * @return
     */
    public boolean hasLimitConfig() {
        return has(OFFSET_LIMIT);
    }

    /**
     * 是否包含SKU销售信息
     * @return
     */
    public boolean hasSku() {
        return has(OFFSET_SKU);
    }

    /**
     * 是否包含积分换购信息
     * @return
     */
    public boolean hasCredits() {
        return has(OFFSET_CREDITS);
    }

    /**
     * 是否包含SKU积分换购信息
     * @return
     */
    public boolean hasCreditsSku() {
        return has(OFFSET_CREDITS_SKU);
    }

    /**
     * 是否包含供货价
     *
     * @return
     */
    public boolean hasSupplyPrice() {
        return has(OFFSET_SUPPLY_PRICE);
    }

    private boolean has(int offset) {
        return (value & offset) > 0;
    }

    /**
     * 是否从缓存中获取数据（缓存存在的情况下）
     *
     * @return
     */
    public boolean isQueryFromCache() {
        return queryCache;
    }
}
