package cn.com.duiba.kjy.base.api.request.iddecode;

import cn.com.duiba.kjy.base.api.enums.IdDecode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.ConversionNotSupportedException;
import org.springframework.beans.TypeMismatchException;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.core.annotation.Order;
import org.springframework.validation.BindException;
import org.springframework.validation.Errors;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.support.WebDataBinderFactory;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.method.annotation.MethodArgumentConversionNotSupportedException;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.ModelAndViewContainer;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Created by dugq on 2019-05-13.
 * 解析GET方式提交参数中的加密ID
 */
@Order(Integer.MIN_VALUE)
@Slf4j
public class IdArgumentResolver extends IdDecodeBean implements HandlerMethodArgumentResolver {

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        return (parameter.hasParameterAnnotation(IdDecode.class)  //基本类型
                || parameter.getParameterType().isAnnotationPresent(IdDecode.class))  //对象接受
                && !parameter.getParameterType().isAnnotationPresent(RequestBody.class)  //非post
                ;
    }

    @Override
    public Object resolveArgument(MethodParameter parameter, ModelAndViewContainer mavContainer, NativeWebRequest webRequest, WebDataBinderFactory binderFactory) throws Exception {
        return resolverParameterInRequestUrl(parameter, webRequest,binderFactory);
    }

    private Object resolverParameterInRequestUrl(MethodParameter parameter, NativeWebRequest webRequest, WebDataBinderFactory binderFactory) throws Exception {
        Class<?> parameterType = parameter.getParameterType();
        if(isPrimitiveOrString(parameterType)){    //参数 是 基本类型或者string
            return resolverPrimitiveAndStringValue(parameter, webRequest);
        }else if(isArray(parameterType) || isCollection(parameterType)){
            String parameterName = parameter.getParameterName();
            String[] parameterValues = webRequest.getParameterValues(parameterName);
            if(Objects.isNull(parameterValues)){
                return null;
            }
            IdDecode parameterAnnotation = parameter.getParameterAnnotation(IdDecode.class);
            List<Long> ids = Arrays.stream(parameterValues).map(value -> decodeId( value, Long.class,parameterName)).collect(Collectors.toList());
            if(isArray(parameterType)){
                return ids.toArray(new Long[0]);
            }
            if(Set.class.isAssignableFrom(parameterType)){
                return new HashSet<>(ids);
            }
            return ids;
        }
        else{
            List<Field> declaredFields = getAllFields(parameterType);
            Object instance = parameterType.newInstance();
            for(Field field : declaredFields){  //遍历属性赋值
                Object fieldValue = webRequest.getParameter(field.getName()); //获取参数值
                if(Objects.nonNull(fieldValue)){
                    resolverObjectFiled(parameter, webRequest, binderFactory, instance, field);//填充值
                }
            }
            validatorIfNecessary(parameter, webRequest, binderFactory, instance); //spring validator
            return instance;
        }
    }

    //spring validator
    private void validatorIfNecessary(MethodParameter parameter, NativeWebRequest webRequest, WebDataBinderFactory binderFactory, Object instance) throws Exception {
        WebDataBinder binder = binderFactory.createBinder(webRequest, instance, parameter.getParameterName());
        if (binder.getTarget() != null) {
            validateIfApplicable(binder, parameter);
            if (binder.getBindingResult().hasErrors() && isBindExceptionRequired(parameter)) {
                throw new BindException(binder.getBindingResult());
            }
        }
    }

    //解析 object 的field
    private void resolverObjectFiled(MethodParameter parameter, NativeWebRequest webRequest, WebDataBinderFactory binderFactory, Object instance, Field field) throws Exception {
        IdDecode annotation = field.getAnnotation(IdDecode.class);
        if(isPrimitiveOrString(field.getType()) && Objects.nonNull(annotation)) {
            String fieldValue = webRequest.getParameter(field.getName());
            Long id = decodeId( fieldValue, parameter.getParameterType(),field.getName());
            WebDataBinder binder = binderFactory.createBinder(webRequest, null, field.getName());
            Object valueObj = binder.convertIfNecessary(id, field.getType(), field);
            field.setAccessible(true);
            field.set(instance, valueObj);
        }else{
            Object fieldValue = webRequest.getParameter(field.getName());
            field.setAccessible(true);
            if (binderFactory != null) {
                WebDataBinder binder = binderFactory.createBinder(webRequest, null, field.getName());
                try {
                    fieldValue = binder.convertIfNecessary(fieldValue, field.getType(), field);
                    field.set(instance,fieldValue);
                }
                catch (ConversionNotSupportedException ex) {
                    throw new MethodArgumentConversionNotSupportedException(fieldValue, ex.getRequiredType(),
                            field.getName(), parameter, ex.getCause());
                }
                catch (TypeMismatchException ex) {
                    throw new MethodArgumentTypeMismatchException(fieldValue, ex.getRequiredType(),
                            field.getName(), parameter, ex.getCause());
                }
            }else{
                log.error("binderFactory is null ");
            }

        }
    }

    private Object resolverPrimitiveAndStringValue(MethodParameter parameter, NativeWebRequest webRequest) {
        String parameterName = parameter.getParameterName();
        String value = webRequest.getParameter(parameterName);
        return decodeId( value,parameter.getParameterType(),parameterName);
    }


    private boolean isBindExceptionRequired( MethodParameter parameter) {
        int i = parameter.getParameterIndex();
        Class<?>[] paramTypes = parameter.getMethod().getParameterTypes();
        boolean hasBindingResult = (paramTypes.length > (i + 1) && Errors.class.isAssignableFrom(paramTypes[i + 1]));
        return !hasBindingResult;
    }

    private void validateIfApplicable(WebDataBinder binder, MethodParameter parameter) {
        Annotation[] annotations = parameter.getParameterAnnotations();
        for (Annotation ann : annotations) {
            Validated validatedAnn = AnnotationUtils.getAnnotation(ann, Validated.class);
            if (validatedAnn != null || ann.annotationType().getSimpleName().startsWith("Valid")) {
                Object hints = (validatedAnn != null ? validatedAnn.value() : AnnotationUtils.getValue(ann));
                Object[] validationHints = (hints instanceof Object[] ? (Object[]) hints : new Object[] {hints});
                binder.validate(validationHints);
                break;
            }
        }
    }
}
