package cn.com.duiba.nezha.alg.feature.parse;

import cn.com.duiba.nezha.alg.common.enums.DateStyle;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.DateUtil;
import cn.com.duiba.nezha.alg.common.util.LocalDateUtil;
import cn.com.duiba.nezha.alg.feature.vo.AdxFeatureDo;
import com.alibaba.fastjson.JSON;
import org.slf4j.LoggerFactory;


import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by pc on 2019/2/21.
 */
public class AdxFeatureParse {

    public static int F_MAX_SIZE = 64;
    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(AdxFeatureParse.class);

    public static Integer[] HUABEI_REGION = {80, 11, 12, 13, 14, 15};
    public static Integer[] DONGBEI_REGION = {81, 21, 22, 23};
    public static Integer[] HUADONG_REGION = {82, 31, 32, 33, 34, 35, 36, 37};
    public static Integer[] HUAZHONG_REGION = {83, 41, 42, 43};
    public static Integer[] HUANAN_REGION = {84, 44, 45, 46};
    public static Integer[] XINAN_REGION = {85, 50, 51, 52, 53, 54};
    public static Integer[] XIBEI_REGION = {86, 61, 62, 63, 64, 65};
    public static Integer[] HMT_REGION = {87, 71, 8100, 8200};


    public static Integer[] FIRST_TIER_CITY = {11, 12, 31, 50, 2101, 3201, 3202, 3205, 3301, 3302, 3702, 4101, 4201, 4301, 4401, 4403, 4419, 5101, 6101};
    public static Integer[] SECOND_TIER_CITY = {1301, 1306, 1401, 2102, 2201, 2301, 3203, 3204, 3206, 3303, 3304, 3306, 3307, 3310, 3401, 3501, 3502, 3505, 3601, 3701, 3706, 3707, 4406, 4413, 4420, 4501, 5201, 5301, 6201, 6501};
    public static Integer[] THIRD_TIER_CITY = {1309,1303,1502,2202,3207,3209,3305,3412,3509,3611,3708,3703,4114,4110,4210,4307,4306,4407,4408,4601,5107,6104,1304,1302,1501,2306,3212,3210,3403,3407,3503,3607,3609,3713,4103,4107,4108,4206,4304,4302,4452,4412,4503,4602,5203,6301,1310,1305,2103,3208,3213,3211,3411,3402,3506,3604,3717,3710,4113,4115,4211,4205,4303,4451,4405,4404,4502,5106,5303,6401};



    public static Map<String, String> generateFeatureMapStatic(AdxFeatureDo cf) {
        Map<String, String> retMap = new HashMap<>(F_MAX_SIZE);
        try {
            if (cf != null) {

                cf.setCurrentCreatTime(getStringToTimeStamp());

                //推广组
                retMap.put("f1010010", DataUtil.Long2String(cf.getGroupId())); // 推广组ID
                retMap.put("f1010020", DataUtil.Integer2String(cf.getAdxType())); // 平台类型
                retMap.put("f1010030", cf.getCat()); // 推广组对应类别
                retMap.put("f1010031", cf.getTuiaCat()); // 推啊平台的推广组对应类别(标准)
                retMap.put("f1010040", cf.getSectionCat()); // 推广组当前部分的类别
                retMap.put("f1010041", cf.getTuiaSectionCat()); // 推啊平台的推广组当前部分的类别(标准)
                retMap.put("f1010050", cf.getKeywords()); // 推广组关键词

                //资源位
                retMap.put("f2010010", DataUtil.Long2String(cf.getResourceId())); // 资源位ID
                retMap.put("f2010020", DataUtil.Integer2String(cf.getAdvertType())); // 广告类型
                retMap.put("f2010030", DataUtil.Integer2String(cf.getPosType())); // 广告位类型
                retMap.put("f2010040", DataUtil.Integer2String(cf.getPosStyle())); // 广告位样式

                //上下文
                retMap.put("f4010010", cf.getUa()); // ua
                retMap.put("f4010020", cf.getIp()); // ip
                retMap.put("f4010030", DataUtil.Long2String(cf.getAreaCode())); // 地域

                Long regionLevel = getRegionLevel(cf.getAreaCode());
                retMap.put("f4010031", DataUtil.Long2String(regionLevel)); // 地区级别（华北/东北/华东/华中/华南/西南/西北/港澳）
                Long tierCityLevel = getTierCityLevel(cf.getAreaCode());
                retMap.put("f4010032", DataUtil.Long2String(tierCityLevel)); // 城市级别（一线/二线/三线）

                retMap.put("f4010040", DataUtil.Integer2String(cf.getDeviceType())); // 设备类型
                retMap.put("f4010041", DataUtil.Integer2String(cf.getTuiaDeviceType())); // 推啊平台的设备类型(标准)
                retMap.put("f4010050", cf.getMake()); // 设备厂商
                retMap.put("f4010060", cf.getModel()); // 设备型号
                retMap.put("f4010070", cf.getOs()); // 操作系统
                retMap.put("f4010080", cf.getCarrier()); // 运营商
                retMap.put("f4010090", DataUtil.Integer2String(cf.getConnectionType())); // 联网方式
                retMap.put("f4010091", DataUtil.Integer2String(cf.getTuiaConnectionType())); // 推啊平台的联网方式(标准)
                retMap.put("f4010100", getFloat2String(cf.getLat())); // 纬度
                retMap.put("f4010110", getFloat2String(cf.getLon())); // 经度

                Integer hour = getTimeStampToHour(cf.getCurrentCreatTime());
                retMap.put("f4010121", DataUtil.Integer2String(hour)); // 竞价请求创建时间-小时

                Integer weekDay = getTimeStampToWeekDay(cf.getCurrentCreatTime());
                retMap.put("f4010122", DataUtil.Integer2String(weekDay)); // 竞价请求创建时间-星期几

                //用户
                retMap.put("f5010010", cf.getGender()); // 性别
                retMap.put("f5010020", cf.getInterestKeyWords()); // 用户兴趣
                retMap.put("f5010030", cf.getYob()); // 出生年月

                //上下文-新增
                retMap.put("f6010010", cf.getAppName()); //应用名称
                retMap.put("f6010020", cf.getAppBundle()); // 包名
                retMap.put("f6010030", cf.getAppPageCat()); // 应用当前页类型
                retMap.put("f6010040", DataUtil.Integer2String(cf.getAppPaid())); // 0-免费 1-付费
                retMap.put("f6010050", cf.getGeoCountry()); // 设备位置信息-国家
                retMap.put("f6010060", cf.getGeoRegion()); // 设备位置信息-地区
                retMap.put("f6010070", cf.getGeoCity()); //设备位置信息-城市
                retMap.put("f6010080", DataUtil.Integer2String(cf.getGeoType())); //设备位置信息-定位类型


            }

        } catch (Exception e) {

            logger.error("AdxFeatureParse.generateFeatureMapStatic error:" + e);
        }

        return retMap;
    }


    public static Map<String, String> generateFeatureMapDynamic(AdxFeatureDo cf, AdxFeatureDo staticCf) {
        Map<String, String> retMap = new HashMap<>(F_MAX_SIZE);
        try {
            if (cf != null) {

                //创意（素材）
                retMap.put("f3010010", DataUtil.Long2String(cf.getIdeaId()));
                retMap.put("f3010020", cf.getButtonText());
                retMap.put("f3010030", DataUtil.Integer2String(cf.getIconH()));
                retMap.put("f3010040", DataUtil.Integer2String(cf.getIconW()));
                retMap.put("f3010050", DataUtil.Integer2String(cf.getMjPicType()));
                retMap.put("f3010060", DataUtil.Integer2String(cf.getIdeaPicH1()));
                retMap.put("f3010070", DataUtil.Integer2String(cf.getIdeaPicW1()));
                retMap.put("f3010080", DataUtil.Integer2String(cf.getIdeaPicH2()));
                retMap.put("f3010090", DataUtil.Integer2String(cf.getIdeaPicW2()));
                retMap.put("f3010100", DataUtil.Integer2String(cf.getIdeaPicH3()));
                retMap.put("f3010110", DataUtil.Integer2String(cf.getIdeaPicW3()));
                retMap.put("f3010120", DataUtil.Integer2String(cf.getIdeaPicNum()));

                // 该创意+设备号的当天累计曝光次数
                Long ideaExpTimes = getIdeaExpTime(cf.getIdeaId(), staticCf.getExpTimeMap());
                retMap.put("f5010040", DataUtil.Long2String(ideaExpTimes));

            }
        }catch (Exception e) {
            logger.error("AdxFeatureParse.generateFeatureMapDynamic error:", e);

        }
        return retMap;
    }




    public static Integer getTimeStampToHour(Long timeStamp) {
        Integer ret = null;
        if (timeStamp != null) {
            Date date = new Date(timeStamp);

            if (date != null){
                LocalDateTime localDate = LocalDateUtil.dateToLocal(date);

                if (localDate != null) {
                    ret = LocalDateUtil.getHour(localDate);

                }
            }
        }
        return ret;
    }


    public static Integer getTimeStampToWeekDay(Long timeStamp) {
        Integer ret = null;
        if (timeStamp != null) {
            Date date = new Date(timeStamp);

            if (date != null){
                LocalDateTime localDate = LocalDateUtil.dateToLocal(date);

                if (localDate != null) {
                    ret =  LocalDateUtil.getDayOfWeek(localDate);

                }
            }
        }
        return ret;
    }


    //
    public static Long getStringToTimeStamp() {
        Long ret = null;
        String currentTime = LocalDateUtil.getCurrentLocalDateTime(DateStyle.YYYY_MM_DD_HH_MM_SS.getValue());

        if (currentTime != null) {
            ret = LocalDateUtil.getTimeMillis(currentTime, DateStyle.YYYY_MM_DD_HH_MM_SS.getValue());

        }
        return ret;
    }


    public static String getFloat2String(Float src) {
        String ret = null;
        if (src != null) {
            ret = Float.toString(src);
        }
        return ret;

    }


    public static Long getObject2Long(Object src) {
        Long ret = null;
        if (AssertUtil.isNotEmpty(src)) {
            ret = DataUtil.string2Long(String.valueOf(src));
        }
        return ret;

    }


    //该创意+设备号的当天累计曝光次数 f5010040
    public static Long getIdeaExpTime(Long ideaId, Map<Long, Object> expTimeMap) {

        Long ret = null;

        if (AssertUtil.isAllNotEmpty(ideaId, expTimeMap)) {

            Long cnt = getObject2Long(expTimeMap.get(ideaId));
            if (cnt == null || cnt < 1) {
                ret = 0L;
            } else if (cnt <= 7) {
                ret = cnt;
            } else {
                ret = -1L;
            }
        }
        return ret;
    }



    //根据areaCode解析地区级别（华北/东北/华东/华中/华南/西南/西北/港澳） f4010031
    public static Long getRegionLevel(Long areaCode) {

        Long ret = null;

        if (AssertUtil.isNotEmpty(areaCode)) {

            int area = DataUtil.toInt(areaCode);
            if ( areaCode >= 1000L && areaCode != 8100L && areaCode != 8200L) {
               area =  DataUtil.toInt(Long.valueOf(areaCode / 100));
            }

            if (Arrays.asList(HUABEI_REGION).contains(area)) { ret = 80L; }
            else if (Arrays.asList(DONGBEI_REGION).contains(area)) { ret = 81L; }
            else if (Arrays.asList(HUADONG_REGION).contains(area)) { ret = 82L; }
            else if (Arrays.asList(HUAZHONG_REGION).contains(area)) { ret = 83L; }
            else if (Arrays.asList(HUANAN_REGION).contains(area)) { ret = 84L; }
            else if (Arrays.asList(XINAN_REGION).contains(area)) { ret = 85L; }
            else if (Arrays.asList(XIBEI_REGION).contains(area)) { ret = 86L; }
            else if (Arrays.asList(HMT_REGION).contains(area)) { ret = 87L; }

        }

        return ret;
    }



    //根据areaCode解析城市级别（一线/二线/三线） f4010032
    public static Long getTierCityLevel(Long areaCode) {

        Long ret = null;

        if (AssertUtil.isNotEmpty(areaCode)) {

            int area = DataUtil.toInt(areaCode);

            if (Arrays.asList(FIRST_TIER_CITY).contains(area)) { ret = 1L; }
            else if (Arrays.asList(SECOND_TIER_CITY).contains(area)) { ret = 2L; }
            else if (Arrays.asList(THIRD_TIER_CITY).contains(area)) { ret = 3L; }

        }

        return ret;
    }





    //测试
    public static void main(String[] args) {

        Long currentGmtCreateTime1 = new Date().getTime();
        System.out.println("time:"+currentGmtCreateTime1);

        Integer hour = getTimeStampToHour(currentGmtCreateTime1);
        System.out.println("hour:"+DataUtil.Integer2String(hour));

        Integer weekDay = getTimeStampToWeekDay(currentGmtCreateTime1);
        System.out.println("weekday:"+DataUtil.Integer2String(weekDay));

        Long currentTime1 = getStringToTimeStamp();
        System.out.println("time1:"+currentTime1);

        System.out.println("ret1:"+getFloat2String(null));

        System.out.println("ret2:"+getObject2Long(null));

        System.out.println("getRegionLevel:"+getRegionLevel(4690L));
        System.out.println("getTierCityLevel:"+getTierCityLevel(null));

        AdxFeatureDo adxFeatureDo1 = new AdxFeatureDo();
        adxFeatureDo1.setCurrentCreatTime( new Date().getTime());
        Map<String, String> staticMap1 = new HashMap();
        staticMap1 = AdxFeatureParse.generateFeatureMapStatic(adxFeatureDo1);
        System.out.println("staticMap1=" + JSON.toJSONString(staticMap1));

    }

}