package cn.com.duiba.nezha.alg.model;

import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.feature.vo.FeatureMapDo;
import cn.com.duiba.nezha.alg.model.enums.HyperParams;
import cn.com.duiba.nezha.alg.model.enums.MutModelType;
import cn.com.duiba.nezha.alg.model.enums.PredictResultType;
import cn.com.duiba.nezha.alg.model.tf.LocalTFModelV2;
import cn.com.duiba.nezha.alg.model.tf.TFServingClient;
import com.alibaba.fastjson.JSON;
import lombok.Data;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

@Data
public class ESMMV3 implements Serializable {
    //for 大盘深度模型（用新版特征）；这个是复制的ESSMV2
    // 删掉了ctr/cvr FusingModel（因为兜底模型不再维护，且只在预热环节），以及ctr/cvr TFServingClient（旧版的远程访问模型文件方式）
    private static final long serialVersionUID = -316102112618444131L;

    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(ESMMV3.class);

    private int PB_MAX_SIZE = 128;

    /**
     * 算法策略类型
     */
    private Integer algType;

    /**
     * 点击率预估模型   //在线fm
     */
    private IModel ctrModel;

    /**
     * 转化率预估模型  //在线fm
     */
    private IModel cvrModel;


    /**
     * 深度-本地模型 ctr
     */
    private LocalTFModelV2 ctrLocalTFModel;

    /**
     * 深度-本地模型 cvr
     */
    private LocalTFModelV2 cvrLocalTFModel;  //dapan-deep-model(with-new-feature)

    /**
     * 模型类型 ESMM  UN_ESMM
     */
    private MutModelType mutModelType;

    /**
     * 模型是否预热标志
     */
    private Boolean isTmp;



    /**
     * 23.4.23 hangyi
     *
     * @param ctrModel
     * @param cvrModel
     * @param ctrLocalTFModel;USE V2
     * @param cvrLocalTFModel;调用V2
     * @param mutModelType
     */
    public ESMMV3(IModel ctrModel,
                  IModel cvrModel,
                  LocalTFModelV2 ctrLocalTFModel,
                  LocalTFModelV2 cvrLocalTFModel,
                  MutModelType mutModelType) {

        this.ctrModel = ctrModel;
        this.cvrModel = cvrModel;

        this.ctrLocalTFModel = ctrLocalTFModel;
        this.cvrLocalTFModel = cvrLocalTFModel;

        this.mutModelType = mutModelType;
    }


    /**
     * 预估主接口 2023.04.23
     *
     * @auther hangyi
     * @param featureMap 预估样本集
     */
    public <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRs(Map<T, FeatureMapDo> featureMap, Map<HyperParams, Double> params) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = null;

        if (MutModelType.DEBIAS == mutModelType) {   // 深度，负样本采样修正？
            ret = predictCTRsAndCVRsLocalTFNew(featureMap);

            Map<T, Double> cvrPrd = ret.get(PredictResultType.CVR);
            Map<T, Double> newCvrPrd = ret.get(PredictResultType.CVR);

            Double sampleRate = params.getOrDefault(HyperParams.NCW, 1.0);// ==NCW // 负采样修正系数

            for (Map.Entry<T, Double> entry: cvrPrd.entrySet()) {
                Double prob = entry.getValue();
                newCvrPrd.put(entry.getKey(), prob / (prob + (1-prob)/sampleRate));
            }

            ret.put(PredictResultType.CVR, newCvrPrd); // 覆盖修正前
        }
        else if (MutModelType.DEEP_E2E_LOCAL == mutModelType) {

            ret = predictCTRsAndCVRsLocalTFNew(featureMap);
        }

        return ret;
    }


    /**
     * CTR && CVR预估
     *
     * @param featureMap
     * @param <T>
     * @return   //localTFV
     * @throws Exception
     */
    public <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsLocalTFNew(Map<T, FeatureMapDo> featureMap) throws Exception {

        if (Objects.equals(isTmp, Boolean.TRUE)) {
            logger.info("模型预热, algType: {}", algType);
        }

        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);  // PB_MAX_SIZE==128

        Map<T, Double> ctrMap = new HashMap<>();
        if (ctrLocalTFModel != null) {
            ctrMap = ctrModel.predictWithLocalTFV2(featureMap, ctrLocalTFModel);    //wyj,替换成IModelV2
        }
        Map<T, Double> cvrMap = new HashMap<>();
        if (cvrLocalTFModel != null) {
            cvrMap = cvrModel.predictWithLocalTFV2(featureMap, cvrLocalTFModel);   //wyj,替换成IModelV2
        }


        /**
         * 记录预估值
         */

        String cvrMapStr= null;

        if(featureMap.size()<40){
            cvrMapStr= JSON.toJSONString(cvrMap);
        }

        for (Map.Entry<T, FeatureMapDo> entry : featureMap.entrySet()) {
            FeatureMapDo featureMapDo = entry.getValue();
            T key = entry.getKey();
            Double preValue= cvrMap.get(key);
            featureMapDo.setCodeDoStr(cvrMapStr+";"+cvrLocalTFModel.getVersion()+";"+preValue+";"+featureMapDo.getCodeDoStr());
        }

        if (AssertUtil.isAllNotEmpty(ctrMap, cvrMap)) {   //对象非空
            ret.put(PredictResultType.CTR, ctrMap);
            ret.put(PredictResultType.CVR, cvrMap);
        } else {
            ret = null;   //为空则返回值直接为空
        }

        return ret;

    }

    public static void main(String[] args) {
        HashMap<String, Double> ctrMap = new HashMap<>(10);
        ctrMap.put("1", 0.01);

        HashMap<String, Double> cvrMap = new HashMap<>(10);
        cvrMap.put("2", 0.22);

        System.out.println(AssertUtil.isAllNotEmpty(ctrMap, cvrMap));
    }



    /**
     * @param featureMap
     * @return
     * @throws Exception
     */
    private <T> Map<T, List<Float>> getParams(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<T, List<Float>> ret = new HashMap<>(PB_MAX_SIZE);

        if (AssertUtil.isNotEmpty(featureMap)) {
            for (Map.Entry<T, Map<String, String>> entry : featureMap.entrySet()) {
                List<Float> ctrP = ctrModel.getParam(entry.getValue());
                List<Float> cvrP = cvrModel.getParam(entry.getValue());
                ret.put(entry.getKey(), paramsSplicing(ctrP, cvrP));
            }
        }
        return ret;

    }

    /**
     * @param featureMap
     * @return
     * @throws Exception
     */
    private <T> Map<T, List<Float>> getParamsNew(Map<T, FeatureMapDo> featureMap) throws Exception {
        Map<T, List<Float>> ret = new HashMap<>(PB_MAX_SIZE);

        if (AssertUtil.isNotEmpty(featureMap)) {
            for (Map.Entry<T, FeatureMapDo> entry : featureMap.entrySet()) {
                List<Float> ctrP = ctrModel.getParam(entry.getValue());
                List<Float> cvrP = cvrModel.getParam(entry.getValue());
                ret.put(entry.getKey(), paramsSplicing(ctrP, cvrP));
            }
        }
        return ret;

    }


    private List<Float> paramsSplicing(List<Float> ctrP, List<Float> cvrP) {
        ctrP.addAll(cvrP);
        return ctrP;
    }


}