package cn.com.duiba.nezha.alg.model;

import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.feature.vo.FeatureMapDo;
import cn.com.duiba.nezha.alg.model.enums.HyperParams;
import cn.com.duiba.nezha.alg.model.enums.MutModelType;
import cn.com.duiba.nezha.alg.model.enums.PredictResultType;
import cn.com.duiba.nezha.alg.model.tf.LocalTFModel;
import cn.com.duiba.nezha.alg.model.tf.TFServingClient;
import lombok.Data;
import net.sf.cglib.core.Local;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Data
public class MOE implements Serializable {
    private static final long serialVersionUID = -316102112618444131L;

    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(MOE.class);

    private int PB_MAX_SIZE = 128;

    /**
     * 点击率预估模型
     */
    private IModel ctrModel;

    /**
     * 转化率预估模型
     */
    private IModel cvrModel;
    private Map<String, IModel> cvrModels;


    /**
     * 降级点击率预估模型
     */
    private IModel ctrFusingModel;

    /**
     * 降级转化率预估模型
     */
    private IModel cvrFusingModel;


    /**
     * 深度-本地模型 ctr
     */
    private LocalTFModel ctrLocalTFModel;

    /**
     * 深度-本地模型 cvr
     */
    private LocalTFModel cvrLocalTFModel;

    /**
     * 深度-本地模型 cvr
     */
    private Map<String, LocalTFModel> cvrLocalTFModels;


    /**
     * git
     * 点击率预估深度学习TF-S-Client
     */
    private TFServingClient ctrTFServingClient;

    /**
     * 转化率预估深度学习TF-S-Client
     */
    private Map<String, TFServingClient> cvrTFServingClients;

    /**
     * 模型类型
     */
    private MutModelType mutModelType;


    public MOE(IModel ctrModel,
               IModel cvrModel,
               IModel ctrFusingModel,
               IModel cvrFusingModel,
               LocalTFModel ctrLocalTFModel,
               LocalTFModel cvrLocalTFModel,
               Map<String, LocalTFModel> cvrLocalTFModels,
               TFServingClient ctrTFServingClient,
               Map<String, TFServingClient> cvrTFServingClients,
               MutModelType mutModelType) {
        this.ctrModel = ctrModel;
        this.cvrModel = cvrModel;
        this.ctrFusingModel = ctrFusingModel;
        this.cvrFusingModel = cvrFusingModel;
        this.ctrLocalTFModel = ctrLocalTFModel;
        this.cvrLocalTFModel = cvrLocalTFModel;
        this.cvrLocalTFModels = cvrLocalTFModels;
        this.ctrTFServingClient = ctrTFServingClient;
        this.cvrTFServingClients = cvrTFServingClients;
        this.mutModelType = mutModelType;
    }


    public MOE(IModel ctrModel,
               IModel cvrModel,
               TFServingClient ctrTFServingClient,
               Map<String, TFServingClient> cvrTFServingClients,
               MutModelType mutModelType) {
        this.ctrModel = ctrModel;
        this.cvrModel = cvrModel;
        this.ctrTFServingClient = ctrTFServingClient;
        this.cvrTFServingClients = cvrTFServingClients;
        this.mutModelType = mutModelType;
    }


    public MOE(IModel ctrModel,
               Map<String, IModel> cvrModels,
               MutModelType mutModelType) {
        this.ctrModel = ctrModel;
        this.cvrModels = cvrModels;
        this.mutModelType = mutModelType;
    }


    public MOE(IModel ctrModel,
               LocalTFModel ctrLocalTFModel,
               IModel cvrModel,
               LocalTFModel cvrLocalTFModel,
               MutModelType mutModelType) {
        this.ctrModel = ctrModel;
        this.ctrLocalTFModel = ctrLocalTFModel;
        this.cvrModel = cvrModel;
        this.cvrLocalTFModel = cvrLocalTFModel;
        this.mutModelType = mutModelType;
    }


    /**
     * 预估主接口
     *
     * 点击率预估 && 转化率预估  &&  深度学习
     */
    public <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsWithTF(
            Map<T, Map<String, String>> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret;

        if (MutModelType.CTR == mutModelType) {
            ret = predictCTRs(featureMap);
        } else {
            ret = predictCTRsAndCVRsWithTFDEEP(featureMap);
        }
        return ret;
    }

    /**
     * 预估主接口
     * 点击率预估 && 转化率预估  &&  深度学习
     */
    public <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsWithTFNew(Map<T, FeatureMapDo> featureMap, Map<HyperParams, Map<String, Double>> params) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret;

        if (MutModelType.CTR == mutModelType) {
            ret = predictCTRsNew(featureMap);
        } else {
            ret = predictCTRsAndCVRsWithTFDEEPNew(featureMap, params);
        }
        return ret;
    }


    /**
     * 预估主接口
     * 点击率预估 && 转化率预估  &&  深度学习
     */
    public <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsWithTFNew(Map<T, FeatureMapDo> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret;

        if (MutModelType.CTR == mutModelType) {
            ret = predictCTRsNew(featureMap);
        } else {
            Map<HyperParams, Map<String, Double>> params = new HashMap<>();
            ret = predictCTRsAndCVRsWithTFDEEPNew(featureMap, params);
        }
        return ret;
    }





    /**
     * CTR预估
     */
    public <T> Map<PredictResultType, Map<T, Double>> predictCTRsNew(Map<T, FeatureMapDo> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);

        Map<T, Double> ctrMap = ctrModel.predictWithTFNew(featureMap, ctrTFServingClient);

        if (AssertUtil.isAllNotEmpty(ctrMap)) {
            ret.put(PredictResultType.CTR, ctrMap);
        } else {
            ret = null;
        }
        return ret;
    }



    /**
     * CTR预估
     */
    public <T> Map<PredictResultType, Map<T, Double>> predictCTRs(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);

        Map<T, Double> ctrMap = ctrModel.predictWithTF(featureMap, ctrTFServingClient);

        if (AssertUtil.isAllNotEmpty(ctrMap)) {
            ret.put(PredictResultType.CTR, ctrMap);
        } else {
            ret = null;
        }
        return ret;
    }

    public <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsWithTFDEEP(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret;

        if (MutModelType.DEEP_ESMM == mutModelType) { // ESMM 深度

            ret = ctrTFServingClient.predictMut(getParams(featureMap));

        } else { // 普通 深度

            ret = predictCTRsAndCVRs(featureMap);
        }

        if (ret == null) { // 降级
            ret = predictFusingCTRsAndCVRs(featureMap);

            for (Map.Entry<PredictResultType, Map<T, Double>> entry : ret.entrySet()) {
                PredictResultType key = entry.getKey();
                if (key == PredictResultType.CTR) {
                    for (Map.Entry<T, Double> subEntry : entry.getValue().entrySet()) {
                        Double predictCtr = subEntry.getValue();

                        if (predictCtr == 0) {
                            String logInfo = ctrTFServingClient.modelName + " predictFusing warn preCtr=0";
                            logger.warn(logInfo);
                        }
                    }
                }
            }

        }

        return ret;

    }



    public <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsWithTFDEEPNew(Map<T, FeatureMapDo> featureMap, Map<HyperParams, Map<String, Double>> params) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret;

        if (MutModelType.DEEP_ESMM == mutModelType) { // ESMM 深度

            ret = ctrTFServingClient.predictMut(getParamsNew(featureMap));

        } else if (MutModelType.DEEP_E2E_LOCAL == mutModelType) { // 本地模型

            ret = predictCTRsAndCVRsLocalTFNew(featureMap);

        } else if (MutModelType.MOE_DEB == mutModelType){ // 分目标在线预测+采样修正
            Map<String, Double> sampleRate = params.getOrDefault(HyperParams.NCW, new HashMap<>());
            ret = predictCTRsAndCVRsMutObjDebSam(featureMap, sampleRate);
        } else if (MutModelType.MOE_DEB_SAME == mutModelType) {
            ret = predictCTRsAndCVRsMutObjDebSamSame(featureMap);
        } else if (MutModelType.MMOE_LOCAL == mutModelType) {

            /**
             * 本地模型
             */
            ret = predictCTRsAndCVRsLocalMMOE(featureMap);

        } else if (MutModelType.MOE_DIFF == mutModelType) { // 分目标在线预测 + 不同FM模型

            ret = predictCTRsAndCVRsDiff(featureMap);

        } else { // 普通 深度
            ret = predictCTRsAndCVRsNew(featureMap);
        }

        if (ret == null) {
            ret = predictFusingCTRsAndCVRsNew(featureMap);

            for (Map.Entry<PredictResultType, Map<T, Double>> entry : ret.entrySet()) {
                PredictResultType key = entry.getKey();
                if (key == PredictResultType.CTR) {
                    for (Map.Entry<T, Double> subEntry : entry.getValue().entrySet()) {
                        Double predictCtr = subEntry.getValue();

                        if (predictCtr == 0) {
                            String logInfo = ctrTFServingClient.modelName + " predictFusing warn preCtr=0";
                            logger.warn(logInfo);
                        }
                    }
                }
            }

        }


        return ret;

    }


    /**
     * CTR && CVR预估
     *
     * @param featureMap
     * @param <T>
     * @return
     * @throws Exception
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsLocalMMOE(Map<T, FeatureMapDo> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);

        Map<T, Double> ctrMap = ctrModel.predictWithLocalTFNew(featureMap, ctrLocalTFModel);

        CODER coder = (CODER) cvrModel;
        Map<PredictResultType, Map<T, Double>> cvrMap = cvrLocalTFModel.predictStrMMOE(coder.getCodesNew(featureMap));
        if (AssertUtil.isAllNotEmpty(ctrMap, cvrMap)) {
            ret.put(PredictResultType.CTR, ctrMap);
            ret.putAll(cvrMap);
        } else {
            ret = null;
        }

        return ret;

    }


    /**
     * CTR && CVR预估
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRs(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);

        Map<T, Double> ctrMap = ctrModel.predictWithTF(featureMap, ctrTFServingClient);
        if (AssertUtil.isNotEmpty(ctrMap)) {
            Map<T, Double> cvrMap = new HashMap<>();
            for(Map.Entry<String, TFServingClient>  cvrTFServingClient: cvrTFServingClients.entrySet()) {
                Map<T, Map<String, String>> subFeatures = new HashMap<>();
                if (cvrTFServingClient.getKey().equals("2") || cvrTFServingClient.getKey().equals("3")) {
                    for (Map.Entry<T, Map<String, String>> instance : featureMap.entrySet()) {
                        if (instance.getValue().get("f115001").equals(cvrTFServingClient.getKey())) {
                            subFeatures.put(instance.getKey(), instance.getValue());
                        }
                    }
                } else {
                    for (Map.Entry<T, Map<String, String>> instance : featureMap.entrySet()) {
                        if (!instance.getValue().getOrDefault("f115001", "0").equals("2")
                                && !instance.getValue().getOrDefault("f115001", "0").equals("3")) {
                            subFeatures.put(instance.getKey(), instance.getValue());
                        }
                    }
                }

                Map<T, Double> subCvrMap = cvrModel.predictWithTF(subFeatures, cvrTFServingClient.getValue());
                if (subCvrMap != null) {
                    subCvrMap.forEach((key, value) -> cvrMap.merge(key, value, (v1, v2) -> v1));
                }
            }
            if (AssertUtil.isNotEmpty(cvrMap)) {
                ret.put(PredictResultType.CTR, ctrMap);
                ret.put(PredictResultType.CVR, cvrMap);
            }
        } else {
            ret = null;
        }

        return ret;
    }


    /**
     * CTR && CVR预估
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsNew(Map<T, FeatureMapDo> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);
        Map<T, Double> ctrMap = ctrModel.predictWithTFNew(featureMap, ctrTFServingClient);

        if (AssertUtil.isNotEmpty(ctrMap)) {
            Map<T, Double> cvrMap = new HashMap<>();
            Map<T, FeatureMapDo> featureMap0 = new HashMap<>();
            Map<T, FeatureMapDo> featureMap2 = new HashMap<>();
            Map<T, FeatureMapDo> featureMap3 = new HashMap<>();
            for (Map.Entry<T, FeatureMapDo> instance : featureMap.entrySet()) {
                String objType = instance.getValue().getDynamicFeatureMap().getOrDefault("f115001", "0");
                if (objType.equals("2")) {
                    featureMap2.put(instance.getKey(), instance.getValue());
                } else if (objType.equals("3")) {
                    featureMap3.put(instance.getKey(), instance.getValue());
                } else {
                    featureMap0.put(instance.getKey(), instance.getValue());
                }
            }
            Map<T, Double> subCvrMap = cvrModel.predictWithTFNew(featureMap0, cvrTFServingClients.get("0"));
            if (subCvrMap != null) {
                subCvrMap.forEach((key, value) -> cvrMap.merge(key, value, (v1, v2) -> v1));
            }

            subCvrMap = cvrModel.predictWithTFNew(featureMap2, cvrTFServingClients.get("2"));
            if (subCvrMap != null) {
                subCvrMap.forEach((key, value) -> cvrMap.merge(key, value, (v1, v2) -> v1));
            }

            subCvrMap = cvrModel.predictWithTFNew(featureMap3, cvrTFServingClients.get("3"));
            if (subCvrMap != null) {
                subCvrMap.forEach((key, value) -> cvrMap.merge(key, value, (v1, v2) -> v1));
            }
            if (AssertUtil.isNotEmpty(cvrMap)) {
                ret.put(PredictResultType.CTR, ctrMap);
                ret.put(PredictResultType.CVR, cvrMap);
            }
        } else {
            ret = null;
        }

        return ret;
    }


    /**
     * 分目标预测+预估修正：当修正Map为空时，则预估修正不生效
     *
     * CTR && CVR预估
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsMutObjDeb(Map<T, FeatureMapDo> featureMap, Map<String, Double> sampleRates) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);
        Map<T, Double> ctrMap = ctrModel.predictWithTFNew(featureMap, ctrTFServingClient);

        if (AssertUtil.isNotEmpty(ctrMap)) {
            Map<T, Double> cvrMap = new HashMap<>();
            Map<String, Map<T, FeatureMapDo>> modelFeatureMap = new HashMap<>();
            for (Map.Entry<T, FeatureMapDo> instance : featureMap.entrySet()) {
                String objType = instance.getValue().getDynamicFeatureMap().getOrDefault("f115001", "0");
                if (!modelFeatureMap.containsKey(objType)) {
                    modelFeatureMap.put(objType, new HashMap<>());
                }
                modelFeatureMap.get(objType).put(instance.getKey(), instance.getValue());
            }
            for (Map.Entry<String, Map<T, FeatureMapDo>> entry: modelFeatureMap.entrySet()) {
                if (cvrModels != null && cvrModels.containsKey(entry.getKey())) {
                    Double sampleRate = sampleRates.getOrDefault(entry.getKey(),1.0);
                    Map<T, Double> subCvrMap = cvrModels.get(entry.getKey()).predictsNew(entry.getValue());
                    Map<T, Double> newCvrPrd = new HashMap<>();
                    for (Map.Entry<T, Double> subEntry: subCvrMap.entrySet()) {
                        Double prob = subEntry.getValue();
                        newCvrPrd.put(subEntry.getKey(), prob / (prob + (1-prob)/sampleRate));
                    }
                    newCvrPrd.forEach((key, value) -> cvrMap.merge(key, value, (v1, v2) -> v1));
                }
            }

            if (AssertUtil.isNotEmpty(cvrMap)) {
                ret.put(PredictResultType.CTR, ctrMap);
                ret.put(PredictResultType.CVR, cvrMap);
            }
        } else {
            ret = null;
        }

        return ret;
    }


    /**
     * 分目标预测+预估修正：当修正Map为空时，则预估修正不生效
     *
     * CTR && CVR预估
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsMutObjDebSam(Map<T, FeatureMapDo> featureMap, Map<String, Double> sampleRates) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);
        Map<T, Double> ctrMap = ctrModel.predictWithTFNew(featureMap, ctrTFServingClient);

        if (AssertUtil.isNotEmpty(ctrMap)) {
            Map<T, Double> cvrMap = new HashMap<>();
            Map<String, Map<T, FeatureMapDo>> modelFeatureMap = new HashMap<>();
            for (Map.Entry<T, FeatureMapDo> instance : featureMap.entrySet()) {
                Map<String, String>  dynamicFeatureMap = instance.getValue().getDynamicFeatureMap();
                String objType = dynamicFeatureMap.getOrDefault("f115001", "0");
                String trade = dynamicFeatureMap.getOrDefault("f660001", "0");
                // 样本 按 "行业 + 目标类型" 粒度进行分组
                String tmpKey = trade + "_" + objType;
                String defaultKey =  "0_0";
                if (!sampleRates.containsKey(tmpKey)) {
                    tmpKey = defaultKey;
                }
                if (!modelFeatureMap.containsKey(tmpKey)) {
                    modelFeatureMap.put(tmpKey, new HashMap<>());
                }
                modelFeatureMap.get(tmpKey).put(instance.getKey(), instance.getValue());
            }

            String firstKey = "0";
            String secondKey = "1";
            for (Map.Entry<String, Map<T, FeatureMapDo>> entry: modelFeatureMap.entrySet()) {
                String tmpKey = entry.getKey();
                String objType = tmpKey.substring(tmpKey.indexOf("_") + 1);
                String modelIndex = firstKey;
                if (!objType.equalsIgnoreCase(firstKey)) {
                    modelIndex = secondKey;
                }
//                System.out.println("tmpKey : " + tmpKey);
                if (cvrModels != null && cvrModels.containsKey(modelIndex)) {
                    Double sampleRate = sampleRates.getOrDefault(tmpKey,1.0);
                    Map<T, Double> subCvrMap = cvrModels.get(modelIndex).predictsNew(entry.getValue());
                    Map<T, Double> newCvrPrd = new HashMap<>();
                    // 对正样本做过采样，修正公式做调整
                    for (Map.Entry<T, Double> subEntry: subCvrMap.entrySet()) {
                        Double prob = subEntry.getValue();
                        newCvrPrd.put(subEntry.getKey(), prob / (prob + (1-prob) * sampleRate));
                    }
                    newCvrPrd.forEach((key, value) -> cvrMap.merge(key, value, (v1, v2) -> v1));
                }
            }

            if (AssertUtil.isNotEmpty(cvrMap)) {
                ret.put(PredictResultType.CTR, ctrMap);
                ret.put(PredictResultType.CVR, cvrMap);
            }
        } else {
            ret = null;
        }

        return ret;
    }

    /**
     * 分目标预测+预估修正：
     *
     * CTR && CVR预估
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsMutObjDebSamSame(Map<T, FeatureMapDo> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);
        Map<T, Double> ctrMap = ctrModel.predictWithTFNew(featureMap, ctrTFServingClient);
        String firstKey = "0";
        String secondKey = "1";
        if (AssertUtil.isNotEmpty(ctrMap)) {
            Map<T, Double> cvrMap = new HashMap<>();
            Map<String, Map<T, FeatureMapDo>> modelFeatureMap = new HashMap<>();
            for (Map.Entry<T, FeatureMapDo> instance : featureMap.entrySet()) {
                Map<String, String>  dynamicFeatureMap = instance.getValue().getDynamicFeatureMap();
                String objType = dynamicFeatureMap.getOrDefault("f115001", "0");
                // 样本 按 目标类型 进行 二分 分组
                String tmpKey = firstKey;
                if (!objType.equalsIgnoreCase(firstKey)) {
                    tmpKey = secondKey;
                }
                if (!modelFeatureMap.containsKey(tmpKey)) {
                    modelFeatureMap.put(tmpKey, new HashMap<>());
                }
                modelFeatureMap.get(tmpKey).put(instance.getKey(), instance.getValue());
            }

            for (Map.Entry<String, Map<T, FeatureMapDo>> entry: modelFeatureMap.entrySet()) {
                String tmpKey = entry.getKey();

                if (cvrModels != null && cvrModels.containsKey(tmpKey)) {
                    Map<T, Double> subCvrMap = cvrModels.get(tmpKey).predictsNew(entry.getValue());
                    Map<T, Double> newCvrPrd = new HashMap<>();
                    double sampleRate = 1.0;
                    if (!tmpKey.equalsIgnoreCase(firstKey)) {
                        sampleRate = 7.0;
                    }
                    // 对正样本做过采样，修正公式做调整
                    for (Map.Entry<T, Double> subEntry: subCvrMap.entrySet()) {
                        Double prob = subEntry.getValue();
                        newCvrPrd.put(subEntry.getKey(), prob / (prob + (1-prob) * sampleRate));
                    }
                    newCvrPrd.forEach((key, value) -> cvrMap.merge(key, value, (v1, v2) -> v1));
                }
            }

            if (AssertUtil.isNotEmpty(cvrMap)) {
                ret.put(PredictResultType.CTR, ctrMap);
                ret.put(PredictResultType.CVR, cvrMap);
            }
        } else {
            ret = null;
        }

        return ret;
    }

    /**
     * 分目标预测+ 前后端使用不同FM模型：
     *
     * CTR && CVR预估
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsDiff(Map<T, FeatureMapDo> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);
        Map<T, Double> ctrMap = ctrModel.predictWithTFNew(featureMap, ctrTFServingClient);
        String firstKey = "0";
        String secondKey = "1";
        if (AssertUtil.isNotEmpty(ctrMap)) {
            Map<T, Double> cvrMap = new HashMap<>();
            Map<String, Map<T, FeatureMapDo>> modelFeatureMap = new HashMap<>();
            for (Map.Entry<T, FeatureMapDo> instance : featureMap.entrySet()) {
                Map<String, String>  dynamicFeatureMap = instance.getValue().getDynamicFeatureMap();
                String objType = dynamicFeatureMap.getOrDefault("f115001", "0");
                // 样本 按 目标类型 进行 二分 分组
                String tmpKey = firstKey;
                if (!objType.equalsIgnoreCase(firstKey)) {
                    tmpKey = secondKey;
                }
                if (!modelFeatureMap.containsKey(tmpKey)) {
                    modelFeatureMap.put(tmpKey, new HashMap<>());
                }
                modelFeatureMap.get(tmpKey).put(instance.getKey(), instance.getValue());
            }

            for (Map.Entry<String, Map<T, FeatureMapDo>> entry: modelFeatureMap.entrySet()) {
                String tmpKey = entry.getKey();
                if (cvrModels != null && cvrModels.containsKey(tmpKey)) {
                    Map<T, Double> subCvrMap = cvrModels.get(tmpKey).predictsNew(entry.getValue());
                    subCvrMap.forEach((key, value) -> cvrMap.merge(key, value, (v1, v2) -> v1));
                }
            }

            if (AssertUtil.isNotEmpty(cvrMap)) {
                ret.put(PredictResultType.CTR, ctrMap);
                ret.put(PredictResultType.CVR, cvrMap);
            }
        } else {
            ret = null;
        }

        return ret;
    }

    /**
     * CTR && CVR预估
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsLocalTFNew(Map<T, FeatureMapDo> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);

        Map<T, Double> ctrMap = ctrModel.predictWithLocalTFNew(featureMap, ctrLocalTFModel);

        if (AssertUtil.isNotEmpty(ctrMap)) {
            Map<T, Double> cvrMap = new HashMap<>();
            for(Map.Entry<String, LocalTFModel>  cvrLocalTFModel: cvrLocalTFModels.entrySet()) {
                Map<T, FeatureMapDo> subFeatures = new HashMap<>();
                if (cvrLocalTFModel.getKey().equals("2") || cvrLocalTFModel.getKey().equals("3")) {
                    for (Map.Entry<T, FeatureMapDo> instance : featureMap.entrySet()) {
                        if (AssertUtil.isEmpty(instance.getValue()) || AssertUtil.isEmpty(instance.getValue().getDynamicFeatureMap())) {
                            continue;
                        }
                        if (instance.getValue().getDynamicFeatureMap().getOrDefault("f115001", "0").equals(cvrLocalTFModel.getKey())) {
                            subFeatures.put(instance.getKey(), instance.getValue());
                        }
                    }
                } else {
                    for (Map.Entry<T, FeatureMapDo> instance : featureMap.entrySet()) {
                        if (AssertUtil.isEmpty(instance.getValue()) || AssertUtil.isEmpty(instance.getValue().getDynamicFeatureMap())) {
                            continue;
                        }
                        if (!instance.getValue().getDynamicFeatureMap().getOrDefault("f115001", "0").equals("2")
                                && !instance.getValue().getDynamicFeatureMap().getOrDefault("f115001", "0").equals("3")) {
                            subFeatures.put(instance.getKey(), instance.getValue());
                        }
                    }
                }

                Map<T, Double> subCvrMap = cvrModel.predictWithLocalTFNew(subFeatures, cvrLocalTFModel.getValue());
                if (subCvrMap != null) {
                    subCvrMap.forEach((key, value) -> cvrMap.merge(key, value, (v1, v2) -> v1));
                }
            }
            if (AssertUtil.isNotEmpty(cvrMap)) {
                ret.put(PredictResultType.CTR, ctrMap);
                ret.put(PredictResultType.CVR, cvrMap);
            }
        } else {
            ret = null;
        }

        return ret;

    }


    /**
     * 降级 CTR && CVR预估
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictFusingCTRsAndCVRs(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);

        ret.put(PredictResultType.CTR, ctrFusingModel.predictWithTF(featureMap, null));
        ret.put(PredictResultType.CVR, cvrFusingModel.predictWithTF(featureMap, null));
        return ret;

    }

    /**
     * 降级 CTR && CVR预估
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictFusingCTRsAndCVRsNew(Map<T, FeatureMapDo> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);

        ret.put(PredictResultType.CTR, ctrFusingModel.predictWithTFNew(featureMap, null));
        ret.put(PredictResultType.CVR, cvrFusingModel.predictWithTFNew(featureMap, null));
        return ret;

    }


    /**
     * 获取模型参数
     */
    private <T> Map<T, List<Float>> getParams(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<T, List<Float>> ret = new HashMap<>(PB_MAX_SIZE);

        if (AssertUtil.isNotEmpty(featureMap)) {
            for (Map.Entry<T, Map<String, String>> entry : featureMap.entrySet()) {
                List<Float> ctrP = ctrModel.getParam(entry.getValue());
                List<Float> cvrP = cvrModel.getParam(entry.getValue());
                ret.put(entry.getKey(), paramsSplicing(ctrP, cvrP));
            }
        }
        return ret;

    }

    /**
     * 获取模型参数
     */
    private <T> Map<T, List<Float>> getParamsNew(Map<T, FeatureMapDo> featureMap) throws Exception {
        Map<T, List<Float>> ret = new HashMap<>(PB_MAX_SIZE);

        if (AssertUtil.isNotEmpty(featureMap)) {
            for (Map.Entry<T, FeatureMapDo> entry : featureMap.entrySet()) {
                List<Float> ctrP = ctrModel.getParam(entry.getValue());
                List<Float> cvrP = cvrModel.getParam(entry.getValue());
                ret.put(entry.getKey(), paramsSplicing(ctrP, cvrP));
            }
        }
        return ret;

    }


    private List<Float> paramsSplicing(List<Float> ctrP, List<Float> cvrP) {
        ctrP.addAll(cvrP);
        return ctrP;
    }


}