package cn.com.duiba.odps.center.api.param.common;

import cn.com.duiba.api.bo.page.PageQuery;

import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 通用查询参数
 * 支持动态表名和动态查询条件
 * 
 * @author zhangdaqing
 * @date 2025/10/13
 */
public class GenericQueryParam extends PageQuery {
    
    private static final long serialVersionUID = 1L;
    
    /**
     * 表名（必填）
     */
    private String tableName;
    
    /**
     * 查询字段列表（可选，默认查询所有字段 *）
     */
    private List<String> selectFields;
    
    /**
     * 查询条件Map
     * key: 字段名
     * value: 查询值，支持以下类型：
     *   - String: 精确匹配 (=)
     *   - List<String>: IN 查询
     *   - DateRange: 时间范围查询 (>= and <=)
     *   - LikeValue: LIKE 查询
     *   - BetweenValue: BETWEEN 查询
     */
    private Map<String, Object> conditions;
    
    /**
     * 排序字段和排序方式
     * key: 字段名
     * value: ASC 或 DESC
     */
    private Map<String, String> orderBy;
    
    /**
     * 是否去重
     */
    private Boolean distinct;
    
    /**
     * 分组字段
     */
    private List<String> groupBy;
    
    /**
     * 最后查询的ID（用于分页优化）
     */
    private Long lastId;
    
    /**
     * ID字段名（默认：id）
     */
    private String idFieldName = "id";
    
    /**
     * 返回类型类名（用于Remote Service泛型支持）
     * 例如：cn.com.duiba.odps.center.api.dto.wanda.WandaOrderBillDTO
     */
    private String resultType;
    
    /**
     * 构造函数
     */
    public GenericQueryParam() {
        this.conditions = new HashMap<>();
        this.orderBy = new HashMap<>();
    }
    
    /**
     * 构造函数
     * 
     * @param tableName 表名
     */
    public GenericQueryParam(String tableName) {
        this();
        this.tableName = tableName;
    }
    
    /**
     * 添加精确匹配条件
     * 
     * @param field 字段名
     * @param value 字段值
     * @return 当前对象
     */
    public GenericQueryParam addEqualCondition(String field, Object value) {
        if (value != null) {
            this.conditions.put(field, value);
        }
        return this;
    }
    
    /**
     * 添加IN条件
     * 
     * @param field 字段名
     * @param values 值列表
     * @return 当前对象
     */
    public GenericQueryParam addInCondition(String field, List<String> values) {
        if (values != null && !values.isEmpty()) {
            this.conditions.put(field, new InCondition(values));
        }
        return this;
    }
    
    /**
     * 添加时间范围条件
     * 
     * @param field 字段名
     * @param startTime 开始时间
     * @param endTime 结束时间
     * @return 当前对象
     */
    public GenericQueryParam addDateRangeCondition(String field, Date startTime, Date endTime) {
        if (startTime != null || endTime != null) {
            this.conditions.put(field, new DateRangeCondition(startTime, endTime));
        }
        return this;
    }
    
    /**
     * 添加LIKE条件
     * 
     * @param field 字段名
     * @param value 值
     * @return 当前对象
     */
    public GenericQueryParam addLikeCondition(String field, String value) {
        if (value != null && !value.isEmpty()) {
            this.conditions.put(field, new LikeCondition(value));
        }
        return this;
    }
    
    /**
     * 添加排序
     * 
     * @param field 字段名
     * @param direction 排序方向（ASC/DESC）
     * @return 当前对象
     */
    public GenericQueryParam addOrderBy(String field, String direction) {
        if (field != null && direction != null) {
            this.orderBy.put(field, direction.toUpperCase());
        }
        return this;
    }
    
    // Getters and Setters
    public String getTableName() {
        return tableName;
    }
    
    public void setTableName(String tableName) {
        this.tableName = tableName;
    }
    
    public List<String> getSelectFields() {
        return selectFields;
    }
    
    public void setSelectFields(List<String> selectFields) {
        this.selectFields = selectFields;
    }
    
    public Map<String, Object> getConditions() {
        return conditions;
    }
    
    public void setConditions(Map<String, Object> conditions) {
        this.conditions = conditions;
    }
    
    public Map<String, String> getOrderBy() {
        return orderBy;
    }
    
    public void setOrderBy(Map<String, String> orderBy) {
        this.orderBy = orderBy;
    }
    
    public Boolean getDistinct() {
        return distinct;
    }
    
    public void setDistinct(Boolean distinct) {
        this.distinct = distinct;
    }
    
    public List<String> getGroupBy() {
        return groupBy;
    }
    
    public void setGroupBy(List<String> groupBy) {
        this.groupBy = groupBy;
    }
    
    public Long getLastId() {
        return lastId;
    }
    
    public void setLastId(Long lastId) {
        this.lastId = lastId;
    }
    
    public String getIdFieldName() {
        return idFieldName;
    }
    
    public void setIdFieldName(String idFieldName) {
        this.idFieldName = idFieldName;
    }
    
    public String getResultType() {
        return resultType;
    }
    
    public void setResultType(String resultType) {
        this.resultType = resultType;
    }
    
    /**
     * 设置返回类型（泛型支持）
     * 
     * @param clazz 返回类型Class
     * @return 当前对象
     */
    public GenericQueryParam setResultType(Class<?> clazz) {
        if (clazz != null) {
            this.resultType = clazz.getName();
        }
        return this;
    }
    
    /**
     * IN条件内部类
     */
    public static class InCondition implements Serializable {
        private static final long serialVersionUID = 1L;
        private List<String> values;
        
        public InCondition(List<String> values) {
            this.values = values;
        }
        
        public List<String> getValues() {
            return values;
        }
    }
    
    /**
     * 时间范围条件内部类
     */
    public static class DateRangeCondition implements Serializable {
        private static final long serialVersionUID = 1L;
        private Date startTime;
        private Date endTime;
        
        public DateRangeCondition(Date startTime, Date endTime) {
            this.startTime = startTime;
            this.endTime = endTime;
        }
        
        public Date getStartTime() {
            return startTime;
        }
        
        public Date getEndTime() {
            return endTime;
        }
    }
    
    /**
     * LIKE条件内部类
     */
    public static class LikeCondition implements Serializable {
        private static final long serialVersionUID = 1L;
        private String value;
        
        public LikeCondition(String value) {
            this.value = value;
        }
        
        public String getValue() {
            return value;
        }
    }
}

