package cn.com.duiba.quanyi.center.api.enums.taibao.cq;

import lombok.AllArgsConstructor;
import lombok.Getter;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author lizhi
 * @date 2025/1/23 14:37
 */
@Getter
@AllArgsConstructor
public enum CqtbOrderStatusEnum {
    /**
     * 不需要审核或者审核通过的名单，未到生效时间，次日8:00生效
     * 可操作：删除
     * 备注：设置为次日（自然日）8:00生效
     */
    PENDING_EFFECTIVENESS(1, "待生效：待推送", true),
    /**
     * 已生效，但是用户还未在服务通知领取红包
     */
    GRANT_PROCESSING(2, "发放中：已推送待领取", false),
    /**
     * 名单中所有所有订单状态都是已领取
     */
    ALL_SUCCESS(3, "发放成功：全部领取", true),
    /**
     * 名单中的订单，有一个状态是已领取，并且不是全部已领取。
     */
    PART_SUCCESS(4, "部分成功：部分领取", true),
    /**
     * 名单中其中有一个订单状态是：发放异常（待补发），且没有发放异常（其他）状态。即有过期未领取的红包
     * 可操作：停止补发、补发、删除
     * 备注：1、如果一个名单中的订单既有发放异常（待补发），又有发放异常（其他），则发放状态为发放异常（其他）。
     * 2、如果一个名单里面所有订单状态都为发放异常（待补发）/发放异常（其他），支持删除名单。名单一旦领取部分或者发放部分不再支持删除名单。
     * 3、停止补发：操作后将不能对名单的所有订单操作补发或者自动补发。
     */
    EXPIRED(5, "发放异常：已过期待补发", true),
    /**
     * 名单中其中有一个订单状态是：发放异常（其他），或者有自动发放超过2次的订单
     * 可操作：停止补发、补发、删除
     * 备注：1、需要坐席介入处理的名单
     * 2、如果一个名单里面所有订单状态都为发放异常（待补发）/发放异常（其他），支持删除名单。名单一旦领取部分或者发放部分不再支持删除名单。
     * 3、停止补发：操作后将不能对名单的所有订单操作补发或者自动补发。
     */
    EXCEPTION(6, "发放异常：异常待人工处理", true),
    /**
     * 点击停止补发功能按钮的名单状态
     */
    CLOSE(7, "发放异常（不补发）：手动停止补发", true),
    ;

    /**
     * 数据库存储值
     */
    private final Integer code;

    /**
     * 返回给太保的msg(状态+中文冒号+msg反馈内容)
     */
    private final String msg;

    /**
     * 接口返回的状态
     */
    private final boolean status;
    
    private static final Map<Integer, CqtbOrderStatusEnum> ENUM_MAP = Collections.unmodifiableMap(Arrays.stream(values()).collect(Collectors.toMap(CqtbOrderStatusEnum::getCode, Function.identity(), (v1, v2) -> v2)));
    
    /**
     * 根据枚举值获取枚举
     * @param code 枚举值
     * @return 枚举
     */
    public static CqtbOrderStatusEnum getByCode(Integer code) {
        if (code == null) {
            return null;
        }
        return ENUM_MAP.get(code);
    }
}
