package cn.com.duiba.quanyi.center.api.enums.deposit;

import lombok.AllArgsConstructor;
import lombok.Getter;

import java.util.Arrays;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author xuzhigang
 * @date 2025/9/2 14:16
 **/
@Getter
@AllArgsConstructor
public enum DepositStatusEnum {
    /**
     * 发起中
     */
    INITIATING(1, "发起中"),

    /**
     * 发起失败
     */
    INITIATE_FAILED(2, "发起失败"),

    /**
     * 审批中
     */
    APPROVING(3, "审批中"),

    /**
     * 审核拒绝
     */
    APPROVAL_REJECTED(4, "审核拒绝"),

    /**
     * 审批成功（待回款）
     */
    PENDING_REFUND(5, "待回款"),

    /**
     * 部分回款
     */
    PARTIAL_REFUND(6, "部分回款"),

    /**
     * 已回款
     */
    REFUNDED(7, "已回款");

    private final Integer status;
    private final String desc;

    private static final Map<Integer, DepositStatusEnum> STATUS_MAP =
            Arrays.stream(values()).collect(Collectors.toMap(DepositStatusEnum::getStatus, Function.identity()));

    /**
     * 根据状态值获取枚举
     *
     * @param status 状态值
     * @return 保证金状态枚举
     */
    public static DepositStatusEnum getByStatus(Integer status) {
        return STATUS_MAP.get(status);
    }

    /**
     * 根据状态值获取描述
     *
     * @param status 状态值
     * @return 状态描述
     */
    public static String getDescByStatus(Integer status) {
        DepositStatusEnum statusEnum = getByStatus(status);
        return statusEnum != null ? statusEnum.getDesc() : "未知状态";
    }

    /**
     * 判断指定状态是否可删除
     * 只有发起失败和审核拒绝状态可以删除
     *
     * @param status 状态值
     * @return true-可删除，false-不可删除
     */
    public static boolean canDelete(Integer status) {
        return INITIATE_FAILED.getStatus().equals(status) || APPROVAL_REJECTED.getStatus().equals(status);
    }
}
