package cn.com.duiba.thirdparty.util.huaxizi;

import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class UrlUtil {

    public static final String EN_US = "en_US";
    public static final String ZH_CN = "zh_CN";

    public static class UrlEntity {

        public String getBaseUrl() {
            return baseUrl;
        }

        public Map<String, String> getParams() {
            return Collections.unmodifiableMap(params);
        }

        private String baseUrl;
        private Map<String, String> params;
    }

    public static UrlEntity parse(String url) {
        UrlEntity entity = new UrlEntity();
        if (url == null) {
            return entity;
        }
        url = url.trim();
        if (url.equals("")) {
            return entity;
        }
        String[] urlParts = url.split("\\?");
        entity.baseUrl = urlParts[0];
        // 没有参数
        if (urlParts.length == 1) {
            return entity;
        }
        // 有参数
        String[] params = urlParts[1].split("&");
        entity.params = new HashMap<>();
        for (String param : params) {
            String[] keyValue = param.split("=");
            entity.params.put(keyValue[0], keyValue.length>1?keyValue[1]:"");
        }
        return entity;
    }



    public static String getParam(String url, String name) {
        String params = url.substring(url.indexOf("?") + 1, url.length());
        Map<String, String> split = Splitter.on("&").withKeyValueSeparator("=").split(params);
        return split.get(name);
    }

    public static String getParamByQueryString(String queryString, String name) {
        return getParamByQueryString(queryString).get(name);
    }

    /**
     * 解析Url
     * @param queryString
     * @return
     */
    public static Map<String, String> getParamByQueryString(String queryString) {
        Map<String, String> split;
        if (!Strings.isNullOrEmpty(queryString)) {
            split = Splitter.on("&").withKeyValueSeparator("=").split(queryString);
        } else {
            split = Maps.newHashMap();
        }
        return split;
    }


    /**
     * 解析出url参数中的键值对，过滤掉链接地址，只会返回参数key和value
     * 如 "index.jsp?Action=del&id=123"，解析出Action:del,id:123存入map中
     *
     * @param URL url地址
     * @return url请求参数部分
     */
    public static Map<String, String> getParamByUrl(String URL) {
        Map<String, String> mapRequest = new HashMap<String, String>();

        String[] arrSplit = null;

        String strUrlParam = TruncateUrlPage(URL);
        if (strUrlParam == null) {
            return mapRequest;
        }
        //每个键值为一组 www.2cto.com
        arrSplit = strUrlParam.split("[&]");
        for (String strSplit : arrSplit) {
            String[] arrSplitEqual = null;
            arrSplitEqual = strSplit.split("[=]");

            //解析出键值
            if (arrSplitEqual.length > 1) {
                //正确解析
                mapRequest.put(arrSplitEqual[0], arrSplitEqual[1]);

            } else {
                if (arrSplitEqual[0] != "") {
                    //只有参数没有值，不加入
                    mapRequest.put(arrSplitEqual[0], "");
                }
            }
        }
        return mapRequest;
    }

    /**
     * 去掉url中的路径，留下请求参数部分
     *
     * @param strURL url地址
     * @return url请求参数部分
     */
    private static String TruncateUrlPage(String strURL) {
        String strAllParam = null;
        String[] arrSplit = null;

        strURL = strURL.trim();

        arrSplit = strURL.split("[?]");
        if (strURL.length() > 1) {
            if (arrSplit.length > 1) {
                if (arrSplit[1] != null) {
                    strAllParam = arrSplit[1];
                }
            }
        }

        return strAllParam;
    }

    /**
     * 去掉url中的参数
     *
     * @param strURL url地址
     * @return url请求参数部分
     */
    public static String TruncateUrl(String strURL) {
        String strAllParam = null;
        String[] arrSplit = null;

        strURL = strURL.trim();

        arrSplit = strURL.split("[?]");
        if (strURL.length() > 1) {
            if (arrSplit.length > 1) {
                if (arrSplit[0] != null) {
                    strAllParam = arrSplit[0];
                }
            }
        }

        return strAllParam;
    }

    /**
     * 判断指定的Url中，是否包含指定的param参数
     * @param urlStr
     * @param param
     * @return 返回 0:表示只有链接没有参数，1:有指定的key也有值，2:有指定的key但是没有值，3：没有指定的key,但是有其他key
     */
    public static int isUrlIncludeParam(String urlStr, String param){
        Map<String, String> map = getParamByUrl(urlStr);
        if (null == map || map.size()==0){
            return 0;
        }
        if (map.containsKey(param)){
            if (null != map.get(param) && !"".equals(map.get(param))){
                return 1;
            }
            return 2;
        }
        return 3;
    }

    /**
     * 从指定的url中，获取param参数的值
     * @param urlStr
     * @param param
     * @return
     */
    public static String getUrlIncludeParam(String urlStr, String param){
        Map<String, String> map = getParamByUrl(urlStr);
        if (null == map || map.size()==0){
            return "";
        }
        if (map.containsKey(param)){
            return map.get(param);
        }
        return "";
    }

    /**
     * 拼接url,若url中已带上指定的param参数且没有值，则将用value替换之
     * @param urlStr
     * @param param 参数名字
     * @param value 参数值
     * @return
     */
    public static String jointUrl(String urlStr, String param, String value){
        if (Strings.isNullOrEmpty(urlStr) || Strings.isNullOrEmpty(param) || Strings.isNullOrEmpty(value)){
            return urlStr;
        }
        int type = isUrlIncludeParam(urlStr, param);
        StringBuilder url = new StringBuilder();

        switch (type){
            case 0:
                url.append(urlStr);
                url.append("?");
                url.append(param);
                url.append("=");
                url.append(value);
                break;
            case 1:
                url.append(urlStr);
                break;
            case 2:
                url.append(TruncateUrl(urlStr));
                Map<String, String> map = getParamByUrl(urlStr);
                int i = 0;
                for (Map.Entry<String, String> m:map.entrySet()) {
                    if (i==0){
                        url.append("?");
                    }else{
                        url.append("&");
                    }
                    if (!m.getKey().equals(param)){
                        url.append(m.getKey());
                        url.append("=");
                        url.append(m.getValue());
                    }else{
                        url.append(m.getKey());
                        url.append("=");
                        url.append(value);
                    }
                    i++;
                }
                break;
            case 3:
                url.append(urlStr);
                url.append("&");
                url.append(param);
                url.append("=");
                url.append(value);
                break;
        }

        return url.toString();
    }

    /**
     * url追加参数
     *
     * @param url
     * @param paramMap
     * @return
     */
    public static String appendUrl(String url, Map<String, Object> paramMap) {
        String newUrl = url;
        StringBuffer param = new StringBuffer();
        for (String key : paramMap.keySet()) {
            param.append(key + "=" + paramMap.get(key).toString() + "&");
        }

        String paramStr = param.toString();
        paramStr = paramStr.substring(0, paramStr.length() - 1);

        if (newUrl.contains("?")) {
            newUrl += "&" + paramStr;
        } else {
            newUrl += "?" + paramStr;
        }
        return newUrl;

    }

    /**
     * 获取指定url中的某个参数
     *
     * @param url
     * @param name
     * @return
     */
    public static String getParamByUrl(String url, String name) {
        url += "&";
        String pattern = "(\\?|&){1}#{0,1}" + name + "=[^&]*(&{1})";
        Pattern r = Pattern.compile(pattern);
        Matcher m = r.matcher(url);
        if (m.find()) {
            return m.group(0).split("=")[1].replace("&", "");
        } else {
            return null;
        }
    }

}
