package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.constant.CebBankConfig;
import cn.com.duiba.credits.sdk.CreditConsumeParams;
import cn.com.duiba.credits.sdk.SignTool;
import cn.com.duiba.dao.AppDAO;
import cn.com.duiba.domain.AppDO;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.enums.cebbank.CebBankTransCodeEnum;
import cn.com.duiba.enums.cebbank.CebBankTypeEnum;
import cn.com.duiba.notifycenter.domain.NotifyQueueDO;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.thirdparty.dto.cebbank.CebBankObjectNotifyDto;
import cn.com.duiba.tool.AssembleTool;
import org.apache.commons.collections.CollectionUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;

/**
 * @program: thirdparty-all
 * @description: 光大银行定制api
 * @author: Simba
 * @create: 2020-02-18 15:49
 **/
@Service
public class CebBankApi {

    private static final Logger LOGGER = LoggerFactory.getLogger(CebBankApi.class);

    @Autowired
    private CebBankConfig cebBankConfig;
    @Autowired
    private AppDAO appDAO;

    public boolean isCebBank(Long appId) {
        return appId != null
                && CollectionUtils.isNotEmpty(cebBankConfig.getAppIdSet())
                && cebBankConfig.getAppIdSet().contains(appId);
    }

	/**
	 * 构造扣积分请求-入口是remote接口
	 */
	public CreditsMessage getSubCreditsMessage(CreditsMessage request) {
		String url = request.getHttpUrl();
		String newUrl = url.substring(0, url.indexOf('?'));
		String params = url.substring(url.indexOf('?') + 1);
		Map<String, String> requestParams = buildRequestParams(Long.valueOf(request.getAppId()), params, request.getRelationId(), request.getParams());
		request.setHttpUrl(AssembleTool.assembleUrl(newUrl, requestParams));

		LOGGER.info("光大银行-构造扣积分请求-入口是remote接口，httpUrl:{}", request.getHttpUrl());
		return request;
	}

	/**
	 * 构造扣积分请求参数-入口是remote接口
	 */
	private Map<String, String> buildRequestParams(Long appId, String authParams, String relationId, Map<String, String> paramMap) {
		Map<String, String> authParamMap = AssembleTool.getUrlParams(authParams);
		AppDO appDO = appDAO.getAppByCache(appId);

		Map<String, String> requestParams = new HashMap<>();
		requestParams.put("uid", authParamMap.get("uid"));
		requestParams.put("credits", authParamMap.get("credits"));
		requestParams.put("appKey", appDO.getAppKey());
		requestParams.put("timestamp", String.valueOf(System.currentTimeMillis()));
		requestParams.put("description", authParamMap.get("description"));
		requestParams.put("orderNum", relationId);
		requestParams.put("type", CebBankTypeEnum.GAME.getCode());
		requestParams.put("transType", paramMap.get("opId"));
		requestParams.put("transCode", CebBankTransCodeEnum.SUB_CREDITS.getCode());
		requestParams.put("appSecret", appDAO.getAppSecret(appDO));
		requestParams.put("sign", SignTool.sign(requestParams));

		requestParams.remove("appSecret");
		requestParams.put("appId", String.valueOf(appId));

		return requestParams;
	}

    /**
     * 构造扣积分请求-入口是rocketmq
     */
    public SubCreditsMsgWrapper getSubCreditsMessage(SubCreditsMsgWrapper subCreditsMsgWrapper) {
        String url = subCreditsMsgWrapper.getHttpUrl();
        String newUrl = url.substring(0, url.indexOf('?'));
        Map<String, String> requestParams = buildRequestParams(subCreditsMsgWrapper);
        subCreditsMsgWrapper.setHttpUrl(AssembleTool.assembleUrl(newUrl, requestParams));

	    LOGGER.info("光大银行-构造扣积分请求-入口是rocketmq，httpUrl:{}", subCreditsMsgWrapper.getHttpUrl());
        return subCreditsMsgWrapper;
    }

    /**
     * 构造扣积分请求参数-入口是rocketmq
     */
    private Map<String, String> buildRequestParams(SubCreditsMsgWrapper subCreditsMsgWrapper) {
        SubCreditsMsgDto subCreditsMsgDto = subCreditsMsgWrapper.getSubCreditsMsg();
        CreditConsumeParams creditConsumeParams = subCreditsMsgDto.getCreditConsumeParams();
	    Map<String, String> paramMap = subCreditsMsgDto.getParams();
        AppDO appDO = appDAO.getAppByCache(subCreditsMsgDto.getAppId());

        Map<String, String> requestParams = new HashMap<>();
        requestParams.put("uid", creditConsumeParams.getUid());
	    requestParams.put("credits", String.valueOf(creditConsumeParams.getCredits()));
	    requestParams.put("appKey", appDO.getAppKey());
	    requestParams.put("timestamp", String.valueOf(System.currentTimeMillis()));
	    requestParams.put("description", creditConsumeParams.getDescription());
	    requestParams.put("orderNum", subCreditsMsgDto.getRelationId());
	    requestParams.put("type", CebBankTypeEnum.GAME.getCode());
	    requestParams.put("transType", paramMap.get("opId"));
	    requestParams.put("transCode", CebBankTransCodeEnum.SUB_CREDITS.getCode());
	    requestParams.put("appSecret", appDAO.getAppSecret(appDO));
	    requestParams.put("sign", SignTool.sign(requestParams));

	    requestParams.remove("appSecret");
	    requestParams.put("appId", String.valueOf(subCreditsMsgDto.getAppId()));

        return requestParams;
    }

    /**
     * 构造加积分请求
     */
    public CreditsMessageDto getAddCreditsMessage(CreditsMessageDto request){
        String url = request.getHttpUrl();
        String newUrl = url.substring(0, url.indexOf('?'));
        Map<String, String> requestParams = buildAddCreditsRequestParams(request);
        request.setHttpUrl(AssembleTool.assembleUrl(newUrl, requestParams));

	    LOGGER.info("光大银行-构造加积分请求，httpUrl:{}", request.getHttpUrl());
        return request;
    }

    /**
     * 构造加积分请求参数
     */
    private Map<String, String> buildAddCreditsRequestParams(CreditsMessageDto request){

        Map<String, String> authParams = request.getAuthParams();
	    Map<String, String> paramMap = request.getParams();
        AppDO appDO = appDAO.getAppByCache(Long.valueOf(request.getAppId()));

        Map<String, String> requestParams = new HashMap<>();
        requestParams.put("appKey", appDO.getAppKey());
	    requestParams.put("uid", authParams.get("uid"));
	    requestParams.put("credits", authParams.get("credits"));
	    requestParams.put("type", CebBankTypeEnum.GAME.getCode());
	    requestParams.put("orderNum", request.getRelationId());
	    requestParams.put("timestamp", String.valueOf(System.currentTimeMillis()));
	    requestParams.put("description", authParams.get("description"));
	    requestParams.put("transType", paramMap.get("opId"));
	    requestParams.put("transCode", CebBankTransCodeEnum.ADD_CREDITS.getCode());
	    requestParams.put("appSecret", appDAO.getAppSecret(appDO));
	    requestParams.put("sign", SignTool.sign(requestParams));

	    requestParams.remove("appSecret");
	    requestParams.put("appId", String.valueOf(request.getAppId()));
	    return requestParams;
    }

    /**
     * 构造通知请求
     */
    public HttpRequestBase getRequestNotify(String notifyUrl, NotifyQueueDO record, AppDO appDO){
        Map<String, String> requestParams = buildNotifyRequestParams(record, appDO);
        HttpRequestBase request = new HttpGet(AssembleTool.assembleUrl(notifyUrl, requestParams));
        request.setConfig(getDefaultConfig());

	    LOGGER.info("光大银行-构造通知请求，httpUrl:{}", request.getURI());
        return request;
    }

    /**
     * 通知超时时间
     */
    private RequestConfig getDefaultConfig() {
        return RequestConfig.custom().setConnectTimeout(30000).setConnectionRequestTimeout(30000).setSocketTimeout(30000).build();
    }

    /**
     * 构造通知请求参数
     */
    private Map<String, String> buildNotifyRequestParams(NotifyQueueDO record, AppDO appDO) {

	    Map<String, String> requestParams = new HashMap<>();
	    requestParams.put("appKey", appDO.getAppKey());
	    requestParams.put("uid", record.getPartnerUserId());
	    requestParams.put("orderNum", String.valueOf(record.getRelationId()));
	    requestParams.put("timestamp", String.valueOf(System.currentTimeMillis()));
	    requestParams.put("success", record.getResult().toString());
	    requestParams.put("bizId", record.getDeveloperBizId());
	    requestParams.put("errorMessage", record.getError4developer());
	    requestParams.put("transType", String.valueOf(record.getRelationId()));
	    requestParams.put("transCode", CebBankTransCodeEnum.NOTIFY.getCode());
	    requestParams.put("appSecret", appDAO.getAppSecret(appDO));
	    requestParams.put("sign", SignTool.sign(requestParams));

	    requestParams.remove("appSecret");
	    requestParams.put("appId", String.valueOf(record.getAppId()));
	    return requestParams;
    }

    /**
     * 构造虚拟商品请求，光大银行虚拟商品走对方的加积分接口
     */
    public SupplierRequest getVirtualRequest(SupplierRequest request){
        String url = request.getHttpUrl();
        String newUrl = url.substring(0, url.indexOf('?'));
        String authParams = url.substring(url.indexOf('?') + 1);
        Map<String, String> requestParams = buildVirtualRequestParams(Long.valueOf(request.getAppId()), authParams, request.getParams());
        request.setHttpUrl(AssembleTool.assembleUrl(newUrl, requestParams));

	    LOGGER.info("光大银行-构造虚拟商品请求，httpUrl:{}", request.getHttpUrl());
        return request;
    }

    /**
     * 构造虚拟商品请求参数
     */
    private Map<String, String> buildVirtualRequestParams(Long appId, String authParams, Map<String, String> paramMap) {

        Map<String, String> authParamMap = AssembleTool.getUrlParams(authParams);

        AppDO appDO = appDAO.getAppByCache(appId);

        Map<String, String> requestParams = new HashMap<>();
        requestParams.put("appKey", appDO.getAppKey());
	    requestParams.put("uid", authParamMap.get("uid"));
	    requestParams.put("credits", authParamMap.get("params"));
	    requestParams.put("type", CebBankTypeEnum.GAME.getCode());
	    requestParams.put("orderNum", authParamMap.get("orderNum"));
	    requestParams.put("timestamp", String.valueOf(System.currentTimeMillis()));
	    requestParams.put("description", authParamMap.get("description"));
	    requestParams.put("transType", paramMap.get("opId"));
	    requestParams.put("transCode", CebBankTransCodeEnum.ADD_CREDITS.getCode());
	    requestParams.put("appSecret", appDAO.getAppSecret(appDO));
	    requestParams.put("sign", SignTool.sign(requestParams));

	    requestParams.remove("appSecret");
	    requestParams.put("appId", String.valueOf(appId));
	    return requestParams;
    }

	/**
	 * 构造实物兑换通知请求参数
	 */
	public Map<String, String> buildRequestPrams(CebBankObjectNotifyDto cebBankObjectNotifyDto) {
		AppDO appDO = appDAO.getAppByCache(cebBankObjectNotifyDto.getAppId());

		Map<String, String> requestParams = new HashMap<>();
		requestParams.put("uid", cebBankObjectNotifyDto.getUid());
		requestParams.put("itemCode", cebBankObjectNotifyDto.getItemCode());
		requestParams.put("appKey", appDO.getAppKey());
		requestParams.put("timestamp", String.valueOf(System.currentTimeMillis()));
		requestParams.put("orderNum", cebBankObjectNotifyDto.getOrderNum());
		requestParams.put("transType", cebBankObjectNotifyDto.getTransType());
		requestParams.put("type", CebBankTypeEnum.OBJECT.getCode());
		requestParams.put("userName", cebBankObjectNotifyDto.getUserName());
		requestParams.put("userMobile", cebBankObjectNotifyDto.getUserMobile());
		requestParams.put("address", cebBankObjectNotifyDto.getAddress());
		requestParams.put("province", cebBankObjectNotifyDto.getProvince());
		requestParams.put("city", cebBankObjectNotifyDto.getCity());
		requestParams.put("county", cebBankObjectNotifyDto.getCounty());
		requestParams.put("description", cebBankObjectNotifyDto.getDescription());
		requestParams.put("transCode", CebBankTransCodeEnum.OBJECT_EXCHANGE.getCode());
		requestParams.put("appSecret", appDAO.getAppSecret(appDO));
		requestParams.put("sign", SignTool.sign(requestParams));

		requestParams.remove("appSecret");
		requestParams.put("appId", String.valueOf(cebBankObjectNotifyDto.getAppId()));
		return requestParams;
	}
}

