package cn.com.duiba.biz.credits.impl;

import cn.com.duiba.api.bo.custom.funheadlines.FunheadLinesAppMapping;
import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.biz.credits.FunHeadlinesApi;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.FunHeadlinesConfig;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.thirdparty.enums.funheadlines.FunHeadlinesErrorCode;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.FunHeadlinesSignTool;
import cn.com.duiba.tool.JsonTool;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpRequestBase;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @Description:
 * @Author: fxr
 * @Date: 2019/4/3
 */
@Service
public class FunHeadlinesApiImpl implements FunHeadlinesApi {
    private static final Logger LOGGER = LoggerFactory.getLogger(FunHeadlinesApiImpl.class);
    /**
     * 开发者积分操作，不会同步用户的积分剩余
     * 因此在每次操作积分后，兑吧本地积分设置固定数额；
     */
    public static final String CREDITS = "5000";

    /**
     * 分隔符
     */
    private static final String DELIMITER = "?";

    @Autowired
    private FunHeadlinesConfig funHeadlinesConfig;

    @Override
    public Boolean isFunHeadlines(Long appId) {
        return funHeadlinesConfig.getAppIds().contains(appId);
    }

    @Override
    public SubCreditsMsgWrapper getRequestCredits(SubCreditsMsgWrapper subCreditsMsgWrapper) {
        Map<String, String> originParams = subCreditsMsgWrapper.getSubCreditsMsg().getAuthParams();
        Map<String, String> newParams;
        try {
            newParams = rebuildCreditsParams(originParams);
        } catch (BizException e) {
            LOGGER.info("[FunHeadlinesApiImpl-getRequestCredits]info:", e);
            return subCreditsMsgWrapper;
        }
        subCreditsMsgWrapper.getSubCreditsMsg().setAuthParams(newParams);
        subCreditsMsgWrapper.getSubCreditsMsg().setHttpType(SubCreditsMsgDto.HTTP_POST);

        String url = subCreditsMsgWrapper.getHttpUrl();
        String host = url.indexOf('?') == -1 ? url : url.substring(0, url.indexOf('?'));
        subCreditsMsgWrapper.setHttpUrl(host);
        return subCreditsMsgWrapper;
    }

    @Override
    public CreditsMessageDto getAddCreditsMessage(CreditsMessageDto request) {
        Map<String, String> originParams = request.getAuthParams();
        Map<String, String> newParams;
        try {
            newParams = rebuildCreditsParams(originParams);
        } catch (BizException e) {
            LOGGER.info("[FunHeadlinesApiImpl-getAddCreditsMessage]info:", e);
            return request;
        }
        request.setAuthParams(newParams);
        request.setHttpType(CreditsMessageDto.HTTP_POST);

        String url = request.getHttpUrl();
        String host = url.indexOf('?') == -1 ? url : url.substring(0, url.indexOf('?'));
        request.setHttpUrl(host);
        return request;
    }

    @Override
    public SupplierRequest getVirturalRequest(SupplierRequest request) {
        String url = request.getHttpUrl();
        //  解析URL，获取网关地址，请求参数
        List<String> analysisList = analysisUrl(url);
        if (CollectionUtils.isEmpty(analysisList)) {
            return request;
        }
        String host = analysisList.get(0);
        Map<String, String> params = AssembleTool.getUrlParams(analysisList.get(1));

        Map<String, String> authParams = Maps.newHashMap();
        authParams.put("uid", params.get("uid"));
        authParams.put("orderNum", params.get("orderNum"));
        authParams.put("credits", params.get("params"));
        authParams.put("transfer", params.get("transfer"));

        Map<String, String> newParams;
        try {
            newParams = rebuildCreditsParams(authParams);
        } catch (BizException e) {
            LOGGER.info("[FunHeadlinesApiImpl-getVirturalRequest]info:", e);
            return request;
        }
        request.setAuthParams(newParams);
        request.setHttpUrl(host);
        return request;
    }


    @Override
    public String getVirturalResponse(SupplierRequest message, String body) {
        Map<String, String> duibaDoc = new HashMap<>();
        try {
            String data = checkResponse(body);
            duibaDoc.put("status", "success");
            duibaDoc.put("data", data);
        } catch (Exception e) {
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", e.getMessage());
            LOGGER.info("[FunHeadlinesApiImpl-getVirturalResponse]趣头条-虚拟商品请求响应解析异常:", e);
        }
        return JsonTool.objectToJson(duibaDoc);
    }

    @Override
    public String parseCreditsRsp(String body) {
        Map<String, String> duibaDoc = Maps.newHashMap();
        try {
            checkResponse(body);
            duibaDoc.put("status", "ok");
        } catch (Exception e) {
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", e.getMessage());
            LOGGER.info("[FunHeadlinesApiImpl-parseCreditsRsp]趣头条-加/减积分请求响应解析异常:", e);
        }
        duibaDoc.put("credits", CREDITS);
        return JsonTool.objectToJson(duibaDoc);
    }

    @Override
    public HttpRequestBase resetFunHeadlinesVirtualHttpType(SupplierRequest request, HttpRequestBase http) {
        return AssembleTool.assembleRequest(request.getHttpUrl(), request.getAuthParams());
    }

    /**
     * 重新封装积分操作请求参数
     *
     * @param originParams
     * @return
     */
    private Map<String, String> rebuildCreditsParams(Map<String, String> originParams) throws BizException {
        String transfer = originParams.get("transfer");
        FunheadLinesAppMapping mapping = getMapping(transfer);
        Map<String, String> authParams = new HashMap<>();
        authParams.put("app_id", mapping.getAppId());
        authParams.put("open_id", originParams.get("uid"));
        authParams.put("coin_num", originParams.get("credits"));
        authParams.put("trade_no", originParams.get("orderNum"));
        String sign = FunHeadlinesSignTool.sign(authParams, mapping.getAppkey());
        authParams.put("sign", sign);
        return authParams;
    }


    /**
     * 解析response
     *
     * @param result
     * @return
     * @throws BizException
     */
    private String checkResponse(String result) throws BizException {
        JSONObject resultJson = JSONObject.parseObject(result);
        if (null == resultJson) {
            throw new BizException(FunHeadlinesErrorCode.E20001.getDesc());
        }

        String code = resultJson.getString("code");
        FunHeadlinesErrorCode funHeadlinesErrorCode = FunHeadlinesErrorCode.valueOfCode(code);
        if (!Objects.equals(FunHeadlinesErrorCode.SUCCESS, funHeadlinesErrorCode)) {
            throw new BizException(funHeadlinesErrorCode.getDesc());
        }

        String dataString = resultJson.getString("data");
        if (org.apache.commons.lang.StringUtils.isBlank(dataString)) {
            throw new BizException(FunHeadlinesErrorCode.E20002.getDesc());
        }
        return dataString;
    }

    private FunheadLinesAppMapping getMapping(String appId) throws BizException {
        FunheadLinesAppMapping mapping = funHeadlinesConfig.getFunheadLinesAppMappingByAppId(appId);
        if (mapping == null) {
            throw new BizException(FunHeadlinesErrorCode.E20005.getDesc()).withCode(FunHeadlinesErrorCode.E20005.getCode());
        }
        return mapping;
    }

    /**
     * 解析URL，获取网关地址和参数串
     *
     * @param url
     * @return java.lang.String
     * @throw
     */
    private static List<String> analysisUrl(String url) {
        List<String> analysis = Lists.newArrayList();
        if (StringUtils.isNotBlank(url)) {
            int index = url.indexOf(DELIMITER);
            if (index != -1) {
                analysis.add(url.substring(0, index));
                analysis.add(url.substring(index + 1, url.length()));
            } else {
                analysis.add(url);
                analysis.add("");
            }
        }
        return analysis;
    }
}

