package cn.com.duiba.biz.credits.strategy.Impl;

import cn.com.duiba.api.enums.phapp.PhAppSerialNoEnum;
import cn.com.duiba.biz.credits.strategy.ApiStrategy;
import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.constant.PuhuiConfig;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.puhui.AESEncodeUtil;
import cn.com.duiba.tool.puhui.PuhuiUtils;
import cn.com.duiba.wolf.redis.RedisAtomicClient;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.duiba.wolf.utils.UrlUtils2;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.apache.http.Header;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;


/**
 * 浦惠到家定制
 *
 * @author fja
 */
@Service
public class PuhuiApiStrategy implements ApiStrategy {

    private static final long SUCCESS_CODE = 20000;

    private static final long TEN_MILLION = 10000000L;

    private final Logger LOGGER = LoggerFactory.getLogger(this.getClass());

    private final String DEFAULT_CHARSET = "UTF-8";

    @Resource
    private PuhuiConfig puhuiConfig;

    @Resource(name = "redisTemplate")
    private RedisAtomicClient redisAtomicClient;

    private final RequestConfig config;

    {
        //http请求超时配置
        config = RequestConfig.custom().setConnectTimeout(5000).setSocketTimeout(5000).setConnectionRequestTimeout(1000).build();
    }


    /**
     * 生成虚拟商品request
     *
     * @param request -
     * @return -
     */
    @Override
    public HttpRequestBase getVirtualRequest(SupplierRequest request) {
        try {
            String url = UrlUtils2.extractUrl(request.getHttpUrl());
            Map<String, String> originData = UrlUtils2.extractUrlParamsFromUrl(request.getHttpUrl());
            LOGGER.info("[浦惠到家]虚拟商品兑换originData = [{}]", JSON.toJSONString(originData));
            HttpPost httpPost = new HttpPost(url);
            JSONObject params = new JSONObject();
            //设置请求头
            Header[] headers = createHeaders(PhAppSerialNoEnum.VIRTUAL);
            httpPost.setHeaders(headers);
            //请求参数
            //couponId：商家编码
            params.put("couponId", originData.get("params"));
            //手机号码：partnerUserId
            params.put("phone", originData.get("uid"));
            Map<String, String> extraMap = request.getParams();
            LOGGER.info("[浦惠到家]虚拟商品兑换extraMap = [{}]", JSON.toJSONString(extraMap));
            if (Objects.nonNull(extraMap)) {
                //ip
                params.put("ip", extraMap.get("ip"));
                //风控参数
                String extraInfo = extraMap.get("extraInfo");
                if (StringUtils.isNotBlank(extraInfo)) {
                    JSONObject extJson = JSON.parseObject(extraInfo);
                    String eveSource = extJson.getString("eveSource");
                    if (StringUtils.isNotBlank(eveSource)) {
                        params.put("eveSource", eveSource);
                    }
                    String eveToken = extJson.getString("eveToken");
                    if (StringUtils.isNotBlank(eveToken)) {
                        params.put("eveToken", eveToken);
                    }
                    String eveSid = extJson.getString("eveSid");
                    if (StringUtils.isNotBlank(eveSid)) {
                        params.put("eveSid", eveSid);
                    }
                    String dvToken = extJson.getString("dvToken");
                    if (StringUtils.isNotBlank(dvToken)) {
                        params.put("dvToken", dvToken);
                    }
                }
            }

            LOGGER.info("[浦惠到家]虚拟商品headers={}, 兑换参数 = [{}]", JSON.toJSONString(headers), JSON.toJSONString(params));
            String paramStr = transferRequestParam(params);
            StringEntity stringEntity = new StringEntity(paramStr, DEFAULT_CHARSET);
            stringEntity.setContentEncoding(DEFAULT_CHARSET);
            stringEntity.setContentType("application/json");
            httpPost.setEntity(stringEntity);

            httpPost.setConfig(config);
            Map<String, String> authParams = JSON.parseObject(paramStr, Map.class);
            request.setAuthParams(authParams);
            request.setHttpUrl(url);
            return httpPost;
        } catch (Exception e) {
            LOGGER.error("[浦惠到家]虚拟商品兑换请求生成失败 orderId=" + request.getOrderId(), e);
            throw new IllegalStateException(e);
        }
    }

    /**
     * 解析虚拟商品response
     *
     * @param request 请求
     * @param body    请求体
     * @return
     */
    @Override
    public String getVirtualResponse(SupplierRequest request, String body) {
        JSONObject result = new JSONObject();
        try {
            JSONObject responseJson = JSON.parseObject(body);
            String value = responseJson.getString("value");
            if (StringUtils.isBlank(value)) {
                throw new IllegalStateException("开发者返回value属性为空：" + body);
            }
            //解密
            String afterDecrypt = AESEncodeUtil.aesDecrypt(value, puhuiConfig.getAesSecret());
            if (Objects.equals(afterDecrypt, "404")) {
                throw new IllegalStateException("开发者接口404");
            }
            String decoded = URLDecoder.decode(afterDecrypt, DEFAULT_CHARSET);
            LOGGER.info("[浦惠到家]解密后responseBody=[{}]", decoded);
            JSONObject jsonObject = JSON.parseObject(decoded);
            Long responseCode = Optional.ofNullable(jsonObject.getLong("code")).orElse(-1L);
            if (responseCode == SUCCESS_CODE) {
                result.put("status", "success");
            } else {
                throw new IllegalStateException(jsonObject.getString("message"));
            }
        } catch (Exception e) {
            result.put("status", "fail");
            result.put("errorMessage", e.getMessage());
            LOGGER.error("[浦惠到家]解析虚拟商品兑换请求结果异常 orderId=" + request.getOrderId(), e);
        }
        return result.toJSONString();
    }


    /**
     * 生成header
     *
     * @param serialNoEnum 流水号枚举
     * @return
     */
    public Header[] createHeaders(PhAppSerialNoEnum serialNoEnum) {
        Map<String, String> strMap = Maps.newHashMap();
        //渠道id
        strMap.put("channelId", puhuiConfig.getChannelId());
        //时间戳
        strMap.put("timestamp", DateUtils.getSecondStr(new Date()));
        //流水号
        strMap.put("serialNo", createSerialNo(serialNoEnum));
        //签名
        String sign = PuhuiUtils.signToRequest(strMap, puhuiConfig.getMd5Secret());
        strMap.put("sign", sign);
        return PuhuiUtils.getHeaders(strMap);
    }

    /**
     * 生成流水号
     *
     * @param serialNoEnum {@link PhAppSerialNoEnum}
     * @return 流水号
     */
    public String createSerialNo(PhAppSerialNoEnum serialNoEnum) {
        /**
         * 对接浦惠app接口，流水号需要唯一，因为有多个接口，
         * 每个接口每天流水分配1000万，不足时采用备用流水号，已增加并发访问
         */
        String newKey = serialNoEnum.getKey() + DateUtils.getDayStr(new Date());
        //当天唯一的流水号,12位,前4位为channelId,如100108000001
        Long startSerialNo = PhAppSerialNoEnum.SERIAL_NO__MAP.get(serialNoEnum.getKey()).getStartSerialNo();
        long value = startSerialNo
                - redisAtomicClient.incrBy(newKey, 1, DateUtils.getToTomorrowSeconds(), TimeUnit.SECONDS);
        //采用备用流水号
        if (value + TEN_MILLION < startSerialNo || SpringEnvironmentUtils.isDevEnv() || SpringEnvironmentUtils.isTestEnv() || SpringEnvironmentUtils.isPreEnv()) {
            startSerialNo = PhAppSerialNoEnum.BACKUPS.getStartSerialNo();
            newKey = PhAppSerialNoEnum.BACKUPS.getKey() + DateUtils.getDayStr(new Date());
            value = startSerialNo
                    - redisAtomicClient.incrBy(newKey, 1, DateUtils.getToTomorrowSeconds(), TimeUnit.SECONDS);
        }
        return puhuiConfig.getChannelId() + value;
    }

    /**
     * 参数加密
     *
     * @param params 参数
     * @return
     */
    public String transferRequestParam(JSONObject params) {
        String requestJsonStr = JSON.toJSONString(params);
        LOGGER.info("[浦惠到家]请求入参加密前:{}", requestJsonStr);
        String decode = "";
        try {
            decode = URLEncoder.encode(requestJsonStr, DEFAULT_CHARSET);
        } catch (UnsupportedEncodingException e) {
            LOGGER.error("[浦惠到家]参数加密失败", e);
            return decode;
        }
        String value = AESEncodeUtil.aesEncrypt(decode, puhuiConfig.getAesSecret());
        String resp = JSONObject.toJSONString(new JSONObject().fluentPut("value", value));
        LOGGER.info("[浦惠到家]请求入参加密后:{}", resp);
        return resp;
    }
}
