package cn.com.duiba.biz.credits.impl;

import cn.com.duiba.biz.credits.PinganCardApi;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.PinganCardConfig;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.JsonTool;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @Author fengjun
 * @Date 2019-10-30
 * @Email: fengjun@duiba.com.cn
 * @Description:
 */
@Service
public class PinganCardApiImpl implements PinganCardApi {

    private static final Logger logger = LoggerFactory.getLogger(PinganCardApiImpl.class);

    // 数字签名，密钥算法
    private static final String RSA_KEY_ALGORITHM = "RSA";

    // 数字签名签名/验证算法
    private static final String SIGNATURE_ALGORITHM = "MD5withRSA";


    /**
     * 分隔符
     */
    private static final String DELIMITER = "?";

    //和对方沟通的添加到后面防止从前端看见真正原因的分隔符
    private static final  String WORD_FIXED_FLAG = "—————————————————————————————" +
            "—————————————————————————————";


    @Autowired
    private PinganCardConfig pinganCardConfig;
    /**
     * 是否平安卡中心APP
     *
     * @param appId
     * @return
     */
    @Override
    public Boolean isPingan(Long appId) {
        return pinganCardConfig.getAppIds().contains(appId);
    }

    /**
     * 平安卡中心发金币-封装虚拟商品接口参数
     *
     * @param request
     * @return
     */
    @Override
    public SupplierRequest getVirturalRequest(SupplierRequest request) {
        String url = request.getHttpUrl();
        //  解析URL，获取网关地址，请求参数
        List<String> analysisList = analysisUrl(url);
        if (CollectionUtils.isEmpty(analysisList)) {
            return request;
        }
        String host = analysisList.get(0);
        Map<String, String> params = AssembleTool.getUrlParams(analysisList.get(1));
        String paramString =buildGiveGoldenCoinsParams(
                params.get("orderNum")
                ,params.get("uid")
                ,params.get("params"),params.get("ip"));
        //Map<String, String> authParams =AssembleTool.getUrlParams(paramString);
        //request.setAuthParams(authParams);
        request.setHttpUrl(host+"?"+paramString);
        return request;
    }

    /**
     * 解析虚拟商品接口开发者请求响应
     *
     * @param body
     * @return
     */
    @Override
    public String getVirturalResponse(String body) {
        Map<String, String> duibaDoc = new HashMap<>();
        try {
            String data = checkResponse(body);
            duibaDoc.put("status", "success");
            duibaDoc.put("data", data);
        } catch (Exception e) {
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", e.getMessage());
            logger.info("[PinganCardApiImpl-getVirturalResponse]平安卡中心-虚拟商品请求响应解析异常:", e);
        }
        return JsonTool.objectToJson(duibaDoc);
    }

    /**
     * 解析response
     *
     * @param result
     * @return
     * @throws BizException
     */
    private String checkResponse(String result) throws BizException {
        JSONObject resultJson = JSONObject.parseObject(result);
        if (null == resultJson) {
            throw new BizException("平安卡中心-发金币解析返回体失败");
        }
        if ("950008".equals(resultJson.getString("responseCode"))){
            logger.info("平安卡中心-发金币返回失败，{}", resultJson.toJSONString());
            throw new BizException("【您已达到每日/月领取上限，无法继续领取】"+WORD_FIXED_FLAG+resultJson);

        }
        else if ("000010".equals(resultJson.getString("responseCode"))){
            logger.info("平安卡中心-发金币返回失败，{}", resultJson.toJSONString());
            throw new BizException("【当前系统繁忙，奖励未发放，请稍后再试】"+WORD_FIXED_FLAG+resultJson);
        }
        else if (!"000000".equals(resultJson.getString("responseCode"))) {
            logger.warn("平安卡中心-发金币返回失败，{}", resultJson.toJSONString());
            throw new BizException(WORD_FIXED_FLAG+"金币发放异常："+resultJson);
        }else {
            return "发金币成功";
        }


    }



    private String buildGiveGoldenCoinsParams(String outsourceId, String openId, String thirdActiviId,String ip) {
        //MD5( timeStamp+openId+thirdPart + ActivityCode + outsourceId+私钥)
        if (StringUtils.isBlank(ip)){
            logger.info("平安卡中心-发金币未获取到ip，请检查运营trade-access-web配置项，订单号：{}",outsourceId);
            ip="null";
        }
        StringBuilder stringBuilder = new StringBuilder();
        Long now = System.currentTimeMillis();

        String thirdPart=pinganCardConfig.getThirdPart();
        if (thirdActiviId.contains(",")){
            String[] ths= thirdActiviId.split(",");
            if (ths.length!=2){
                logger.warn("平安卡中心配置错误请运营检查虚拟商品配置:thirdActiviId：{}",thirdActiviId);
            }else {
                thirdPart = ths[1];
                thirdActiviId = ths[0];
            }
        }else {
            logger.info("平安卡中心-发金币虚拟商品标志符{}不带新活动thirdPart，请检查商品配置,订单号{},如果是老平安的活动则不用处理，商品",thirdActiviId,outsourceId);
        }
        String planCode="";
        if (thirdActiviId.contains(":")){
            String[] pcs=thirdActiviId.split(":");
            if (pcs.length!=2){
                logger.warn("平安卡中心配置错误请运营检查虚拟商品配置:thirdActiviId-pcs：{}",thirdActiviId);
            }else {
                planCode=thirdActiviId.split(":")[1];
                thirdActiviId=thirdActiviId.split(":")[0];
            }

        }
        String originalValue= now + openId + thirdPart
                    + thirdActiviId + outsourceId + pinganCardConfig.getInterfaceKey()+ip+planCode;


        String sign = DigestUtils.md5Hex(originalValue);

//        不断重试获取token
        String accessToken=null;
        int tokenFlag=0;

        while (StringUtils.isBlank(accessToken)){
            tokenFlag++;
            accessToken=getAccessToken();

            if (tokenFlag>=4){
                logger.warn("平安卡中心-发金币连续3次失败！");
                break;
            }
        }

        stringBuilder//.append(pinganCardConfig.getGiveCoinsUrl())
                .append("channel=").append(pinganCardConfig.getChannel())
                .append("&accessToken=").append(accessToken)
                .append("&thirdPart=").append(thirdPart)
                .append("&sign=").append(sign)
                .append("&timeStamp=").append(now)
                .append("&outsourceId=").append(outsourceId)
                .append("&openId=").append(openId)
                .append("&ip=").append(ip)
                .append("&activityCode=").append(thirdActiviId);
        if (StringUtils.isNotBlank(planCode)){
            stringBuilder.append("&planCode=").append(planCode);
        }



        return stringBuilder.toString();
    }


    private String getAccessToken() {
        StringBuilder stringBuilder = new StringBuilder();
        Double random = Math.random() * 10000;
        Long timestamp = System.currentTimeMillis();
        String data = pinganCardConfig.getChannel() + ";" + random.longValue() + ";" + timestamp;
        try {
            //PrivateKey privateKey=getPrivateKey(PRIVATE_KEY);
            String sign = sign(data.getBytes(), pinganCardConfig.getPrivateKey());
            stringBuilder.append(pinganCardConfig.getAccessTokenUrl())
                    .append("channel=").append(pinganCardConfig.getChannel())
                    .append("&random=").append(random.longValue())
                    .append("&timestamp=").append(timestamp)
                    .append("&sign=").append(sign);
            JSONObject response = JSONObject.parseObject(doGet(stringBuilder.toString()));
            return response.getString("accessToken");
        } catch (Exception e) {
            logger.warn("平安卡中心-发金币获取token失败 生成公钥失败，{}", data, e);
            return null;
        }


    }

    /**
     * 数字签名
     *
     * @param data    待签名数据
     * @param pri_key 私钥
     * @return 签名
     * @throws Exception 抛出异常
     */
    public static String sign(byte[] data, String pri_key) throws Exception {
        // 取得私钥
        byte[] pri_key_bytes = Base64.decodeBase64(pri_key);
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(pri_key_bytes);
        KeyFactory keyFactory = KeyFactory.getInstance(RSA_KEY_ALGORITHM);
        // 生成私钥
        PrivateKey priKey = keyFactory.generatePrivate(pkcs8KeySpec);
        // 实例化Signature
        Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
        // 初始化Signature
        signature.initSign(priKey);
        // 更新
        signature.update(data);

        return Base64.encodeBase64String(signature.sign());
    }

    private static RequestConfig config = RequestConfig.custom()
            .setConnectionRequestTimeout(2000)
            .setConnectTimeout(2000)
            .setSocketTimeout(2000)
            .build();

    public static String doGet(String url) throws Exception {
        CloseableHttpClient httpclient = HttpClients.custom().setDefaultRequestConfig(config).build();
        CloseableHttpResponse response=null;
        try {
            HttpGet httpget = new HttpGet(url);
            response = httpclient.execute(httpget);
            if (response.getStatusLine().getStatusCode() == 200) {
                return EntityUtils.toString(response.getEntity(), "UTF-8");
            }
        } catch (Exception e) {
            //logger.error("httpclient.doGet:{}" + e.getMessage(), url);
            throw e;
        } finally {
            try {
                if (null!=response){
                    response.close();
                }
            } catch (IOException e) {
               // logger.error("httpclient.close:" + e.getMessage());
            }
        }
        return null;
    }


    /**
     * 解析URL，获取网关地址和参数串
     *
     * @param url
     * @return java.lang.String
     * @throw
     */
    private static List<String> analysisUrl(String url) {
        List<String> analysis = Lists.newArrayList();
        if (StringUtils.isNotBlank(url)) {
            int index = url.indexOf(DELIMITER);
            if (index != -1) {
                analysis.add(url.substring(0, index));
                analysis.add(url.substring(index + 1, url.length()));
            } else {
                analysis.add(url);
                analysis.add("");
            }
        }
        return analysis;
    }
}
