package cn.com.duiba.remoteimpl.mng;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.mng.NewMngConfig;
import cn.com.duiba.thirdparty.api.mng.RemoteNewMngService;
import cn.com.duiba.wolf.redis.RedisAtomicClient;
import cn.com.duiba.wolf.redis.RedisLock;
import cn.com.duibaboot.ext.autoconfigure.httpclient.ssre.CanAccessInsideNetwork;
import cn.hutool.crypto.digest.DigestUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.util.Base64Utils;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author: pengyi
 * @description: 新分销商平台接口
 * @date: 2022/1/24 上午10:23
 */
@RestController
public class RemoteNewMngServiceImpl implements RemoteNewMngService {
    private static final Logger LOGGER = LoggerFactory.getLogger(RemoteNewMngServiceImpl.class);

    private static final String NEW_MNG_ACCESS_TOKEN_GET = "NEW_MNG_ACCESS_TOKEN_GET";
    private static final String NEW_MNG_ACCESS_TOKEN_REFRESH = "NEW_MNG_ACCESS_TOKEN_REFRESH";
    private static final String NEW_MNG_ACCESS_TOKEN_REFRESHING = "NEW_MNG_ACCESS_TOKEN_REFRESHING";

    @CanAccessInsideNetwork
    @Resource(name = "httpClient")
    private CloseableHttpClient httpClient;
    @CanAccessInsideNetwork
    @Resource(name = "stringRedisTemplate")
    private StringRedisTemplate redisTemplate;
    @Resource(name = "redisTemplate")
    private RedisAtomicClient redisAtomicClient;
    @Autowired
    private NewMngConfig newMngConfig;

    /**
     * 获取accessToken
     * http://gitlab2.dui88.com/business_platform/duiba-openapi-web/blob/feature/20211220-openapi/apidoc/%E5%BC%80%E5%8F%91%E6%96%87%E6%A1%A3/%E6%8E%88%E6%9D%83%E8%8E%B7%E5%8F%96accesstoken.md
     * @return
     */
    @Override
    public String getAccessToken() {
        if (redisTemplate.hasKey(NEW_MNG_ACCESS_TOKEN_GET)) {
            return redisTemplate.opsForValue().get(NEW_MNG_ACCESS_TOKEN_GET);
        }
        if (redisTemplate.hasKey(NEW_MNG_ACCESS_TOKEN_REFRESH)) {
            return refreshAccessToken();
        } else {
            return getAccessToken(Maps.newHashMap(), newMngConfig.getGetAccessTokenUrl());
        }
    }

    /**
     * 获取并缓存accessToken
     * @param url
     * @return
     */
    private String getAccessToken(Map<String,Object> params, String url) {
        HttpPost post = new HttpPost(url);
        long curTimestamp = System.currentTimeMillis();
        String sourceValue = newMngConfig.getAppKey() + ":" + newMngConfig.getAppSecret() + ":" + curTimestamp;
        String sign = DigestUtil.md5Hex(sourceValue).toUpperCase();
        params.put("appKey",newMngConfig.getAppKey());
        params.put("sign",sign);
        params.put("timestamp",curTimestamp);
        post.setEntity(new StringEntity(JSON.toJSONString(params),"UTF-8"));
        try(CloseableHttpResponse response = httpClient.execute(post)) {
            String responseStr = EntityUtils.toString(response.getEntity());
            LOGGER.info("新供应商获取/刷新accessToken响应，response:{}", responseStr);
            JSONObject resp = JSON.parseObject(responseStr);
            Boolean success = resp.getBoolean("success");
            if (!success) {
                throw new BizException("新供应商响应异常");
            }
            JSONObject result = resp.getJSONObject("result");
            String accessToken = result.getString("access_token");
            String refreshToken = result.getString("refresh_token");
            if (StringUtils.isBlank(accessToken) || StringUtils.isBlank(refreshToken)) {
                throw new BizException("新供应商响应token为空");
            }
            Map<String, String> tokenCache = Maps.newHashMap();
            tokenCache.put("accessToken", accessToken);
            tokenCache.put("refreshToken", refreshToken);
            redisTemplate.opsForValue().set(NEW_MNG_ACCESS_TOKEN_GET, accessToken, 90L, TimeUnit.MINUTES);
            redisTemplate.opsForValue().set(NEW_MNG_ACCESS_TOKEN_REFRESH, JSON.toJSONString(tokenCache), 2L, TimeUnit.HOURS);
            return accessToken;
        } catch (Exception e) {
            LOGGER.warn("新供应商获取/刷新accessToken异常，params：{}", JSON.toJSONString(params),e);
            return null;
        }
    }

    @Override
    public String refreshAccessToken() {
        if (redisTemplate.hasKey(NEW_MNG_ACCESS_TOKEN_GET)) {
            return redisTemplate.opsForValue().get(NEW_MNG_ACCESS_TOKEN_GET);
        }
        String tokenCache = redisTemplate.opsForValue().get(NEW_MNG_ACCESS_TOKEN_REFRESH);
        JSONObject jsonObject = JSON.parseObject(tokenCache);
        String accessToken = jsonObject.getString("accessToken");
        String refreshToken = jsonObject.getString("refreshToken");
        try (RedisLock redisLock = redisAtomicClient.getLock(NEW_MNG_ACCESS_TOKEN_REFRESHING, 5)) {
            if (redisLock == null) {
                LOGGER.info("新供应商刷新accessToken中，返回老token，params：{}", accessToken);
                return accessToken;
            }
            Map<String,Object> params = Maps.newHashMap();
            params.put("refreshToken", refreshToken);
            return getAccessToken(params, newMngConfig.getRefreshAccessTokenUrl());
        } catch (Exception e) {
            LOGGER.warn("新供应商刷新accessToken异常，refreshToken：{}, accessToken:{}", refreshToken, accessToken, e);
            return accessToken;
        }
    }
}
