package cn.com.duiba.service.impl;

import cn.com.duiba.domain.RequestTypeEnum;
import cn.com.duiba.domain.SupplierResponse;
import cn.com.duiba.mq.RocketMQMsgProducer;
import cn.com.duiba.mq.RocketMqTagConstant;
import cn.com.duiba.mq.RocketMqTopicManager;
import cn.com.duiba.service.DuibaVirtualSupplier;
import cn.com.duiba.service.HttpAsyncClientPool;
import cn.com.duiba.thirdparty.dto.SupplierRequestDto;
import cn.com.duiba.thirdparty.dto.SupplierResponseDto;
import cn.com.duiba.tool.CaiNiaoTool;
import cn.com.duiba.tool.HttpRequestLog;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.entity.GzipDecompressingEntity;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

public abstract class AbstractDuibaVirtualSupplier implements DuibaVirtualSupplier {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractDuibaVirtualSupplier.class);

    private static final String OK = "ok";
    private static final String FAIL = "fail";
    private static final String PROCESS = "process";
    private static final String STATUS = "status";

    @Autowired
    protected RocketMqTopicManager rocketMqTopicManager;
    @Autowired
    private RocketMQMsgProducer rocketProducer;
    @Autowired
    private HttpAsyncClientPool httpAsyncClientPool;

    public void sendResponse(SupplierResponseDto supplierResponseDto) {
        LOGGER.info("发送供应商回调信息,message:{},topic:{}", JSONObject.toJSONString(supplierResponseDto),
                rocketMqTopicManager.getDuibaVirtualNotifyTopic());
        rocketProducer.sendMsg(rocketMqTopicManager.getDuibaVirtualNotifyTopic(),
                RocketMqTagConstant.DUIBA_VIRTUAL_TAG,
                null,
                JSONObject.toJSONString(supplierResponseDto),
                false,
                null);
    }

    public static SupplierResponseDto getSupplierResponse(SupplierRequestDto request) {
        SupplierResponseDto response = new SupplierResponseDto();
        response.setCallbackType(SupplierResponseDto.CALLBACK_TYPE_COMPLETED);
        response.setSupplierName(RocketMqTagConstant.DUIBA_VIRTUAL_TAG);
        response.setOrderId(request.getOrderId());
        response.setAppId(request.getAppId());
        response.setConsumerId(request.getConsumerId());
        response.setSupplierOrderId(request.getSupplierOrderId());
        return response;
    }

    public static void setResponse(SupplierResponseDto response, boolean success, String desc) {
        Map<String, String> map = new HashMap<>();
        map.put("status", success ? OK : FAIL);
        if (StringUtils.isNotBlank(desc)) {
            response.setErrorMessage(desc);
        }
        response.setBody(JSONObject.toJSONString(map));
    }

    @Override
    public void submit(SupplierRequestDto request) {
        HttpRequestBase requestBase = null;
        try {
            requestBase = getVirtualRequest(request);
        } catch (Exception ex) {
            SupplierResponseDto resp = BeanUtils.copy(request, SupplierResponseDto.class);
            resp.setCallbackType(SupplierResponse.CALLBACK_TYPE_FAILED);
            resp.setError4Consumer(getDefaultError4ConsumerMessage());
            resp.setErrorMessage(ex.getClass().getName() + ":" + ex.getMessage());
            finallySendMessage(request, resp);
            return;
        }
        httpAsyncClientPool.submit(request.getAppId(), requestBase, new FutureCallback<HttpResponse>() {
            @Override
            public void completed(HttpResponse response) {
                SupplierResponseDto resp = new SupplierResponseDto();
                try {
                    resp.setOrderId(request.getOrderId());
                    resp.setSupplierOrderId(request.getSupplierOrderId());
                    resp.setSupplierName(request.getSupplierName());
                    resp.setUrl(request.getHttpUrl());
                    resp.setCallbackType(SupplierResponseDto.CALLBACK_TYPE_COMPLETED);
                    resp.setAppId(request.getAppId());
                    resp.setConsumerId(request.getConsumerId());

                    //解析返回结果
                    String result = parseResponseResult(response);
                    //根据供应商返回结果组装标准状态返回
                    String body = getVirtualResponse(request, result);
                    //解析状态,如失败状态则添加错误提示语
                    parseBody(body, resp);
                    //截取body,并回传
                    resp.setBody(HttpRequestLog.subBody(body));
                } catch (Exception e) {
                    LOGGER.error(getClass().getName() + ".completed exception", e);
                } finally {
                    finallySendMessage(request, resp);
                }
            }

            @Override
            public void failed(Exception ex) {
                LOGGER.error(getClass().getName() + ".failed,orderId:" + request.getOrderId(), ex);
                SupplierResponseDto resp = new SupplierResponseDto();
                try {
                    resp.setOrderId(request.getOrderId());
                    resp.setSupplierOrderId(request.getSupplierOrderId());
                    resp.setCallbackType(SupplierResponse.CALLBACK_TYPE_FAILED);
                    resp.setAppId(request.getAppId());
                    resp.setConsumerId(request.getConsumerId());
                    resp.setError4Consumer(getDefaultError4ConsumerMessage());
                    resp.setErrorMessage(ex.getClass().getName() + ":" + ex.getMessage());
                } catch (Exception e) {
                    LOGGER.error(getClass().getName() + ".failed", e);
                } finally {
                    finallySendMessage(request, resp);
                }
            }

            @Override
            public void cancelled() {
                LOGGER.warn(getClass().getName() + ".cancelled,orderId:{}", request.getOrderId());
                SupplierResponseDto resp = new SupplierResponseDto();
                try {
                    resp.setOrderId(request.getOrderId());
                    resp.setSupplierOrderId(request.getSupplierOrderId());
                    resp.setCallbackType(SupplierResponse.CALLBACK_TYPE_CANCELLED);
                    resp.setAppId(request.getAppId());
                    resp.setConsumerId(request.getConsumerId());
                } catch (Exception e) {
                    LOGGER.error(getClass().getName() + ".cancelled", e);
                } finally {
                    finallySendMessage(request, resp);
                }
            }

        }, RequestTypeEnum.VIRTUAL);
    }

    private void finallySendMessage(SupplierRequestDto supplierRequest, SupplierResponseDto supplierResponse) {
        sendResponse(supplierResponse);
    }

    private String parseResponseResult(HttpResponse response) throws IOException {
        Header header = response.getEntity().getContentEncoding();
        String result;
        if (header != null && header.toString().contains(CaiNiaoTool.CONTENT_ENCODING_GZIP)) {
            result = EntityUtils.toString(new GzipDecompressingEntity(response.getEntity()));
        } else {
            result = EntityUtils.toString(response.getEntity());
        }
        return result;
    }

    private void parseBody(String body, SupplierResponseDto resp) {
        if (StringUtils.isBlank(body)) {
            return;
        }
        try {
            JSONObject jsonObject = JSONObject.parseObject(body);
            String statusStr = jsonObject.getString(STATUS);
            if (FAIL.equalsIgnoreCase(statusStr)) {
                resp.setError4Consumer(getDefaultError4ConsumerMessage());
            }
        } catch (Exception e) {
            LOGGER.error("json exception:{}", body);
        }
    }

}
