package cn.com.duiba.tool.suning;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

/**
 * 加验签工具类
 */
public class SuningSignUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(SuningSignUtils.class);

    private static final int INDEX_NOT_FOUND = -1;
    private static final String EQ = "=";
    private static final String AND = "&";
    private static final String SIGN_TYPE = "signType";
    private static final String SIGN_KEY_INDEX = "signkeyIndex";
    private static final String SIGN = "sign";
    private static final String GS_SIGN = "gsSign";
    public static final String PARAMS = "params";
    private static final String ENCRYPT = "encrypt";
    private static final String KEY = "key";
    private static final String CONTENT_TYPE = "Content-Type";
    private static final String APPLICATION_JSON_UTF_8 = "application/json;charset=UTF-8";
    private static final String UTF8_CHARSET = "UTF-8";


    /**
     * 使用AES密钥 对params进行加密
     * 对AES密钥进行RSA加密
     */
    public static void encrypt(String aesKey, String publicKey, Map<String, Object> params) {
        try {
            String str1 = getString(params, PARAMS);
            String str2 = EncryptUtil.encryptBase64DecorateAES(str1, aesKey);
            params.put(PARAMS, str2);
            String encrypt = EncryptUtil.encryptByPublicKey(aesKey, publicKey);
            params.put(KEY, encrypt);
        } catch (Exception e) {
            LOGGER.error("AES加密失败", e);
        }
    }

    /**
     * 取出params加密字符串
     * 使用AES密钥 对params进行解密
     */
    public static Map<String, Object> decrypt(String aesKey, Map<String, Object> params) {
        String str = getString(params, PARAMS);
        try {
            if (params.containsKey(ENCRYPT) && (Boolean) params.get(ENCRYPT)) {
                str = EncryptUtil.decryptBase64DecorateAES(str, aesKey);
            }
            return JSON.parseObject(str, new TypeReference<Map<String, Object>>() {
            });
        } catch (Exception e) {
            LOGGER.error("AES解密失败", e);
            return null;
        }
    }

    private static String getString(Map map, Object key) {
        if (map != null) {
            Object answer = map.get(key);
            if (answer != null) {
                return answer.toString();
            }
        }
        return null;
    }

    /**
     * 验签
     *
     * @param resultMap 结果
     * @param publicKey 公钥
     */
    public static boolean verifySign(Map<String, Object> resultMap, String publicKey) {
        try {
            // 获取签名值sign
            String resultSign = (String) resultMap.get("sign");
            if (isBlank(resultSign)) {
                return false;
            }
            PublicKey pk = RSAUtil.getPublicKey(publicKey);
            // 排除不参与加签的字段。生成摘要明文（验签失败打印明文）
            String digestData = mapToString(resultMap, "sign", "sign_type","signType");
            // 对拼接明文进行MD5摘要（验签失败打印MD5）
            digestData = MD5.digest(digestData, "UTF-8");
            // 传入摘要、公钥和sign，进行验签
            return RSAUtil.vertiy(digestData, resultSign, pk);
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 加签
     *
     * @param privateKey 私钥
     * @param params     参数
     */
    public static void getSign(Map<String, Object> params, String privateKey) {
        try {
            // 排除不参与加签的字段。生成摘要明文
            String digestData = mapToString(params, "sign_type", "signkey_index","signType","signkeyIndex");
            // 对拼接明文进行MD5摘要
            digestData = MD5.digest(digestData, "UTF-8");
            // 传入摘要和私钥，进行加签
            PrivateKey pk = RSAUtil.getPrivateKey(privateKey);
            String sign = RSAUtil.sign(digestData, pk);
            // 将加签后的sign放入请求参数中
            params.put("sign", sign);
        } catch (Exception e) {
            LOGGER.error("生成签名发生异常,param={},privateKey={},error={}",params.toString(),privateKey,e.getMessage());
        }
    }

    private static String mapToString(Map<String, Object> map, String... exclude) {
        Map<String, String> treeMap = new TreeMap<String, String>();
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            String key = entry.getKey();
            if (key == null || contains(exclude, key)) {
                continue;
            }
            if (entry.getValue() instanceof String) {
                treeMap.put(entry.getKey(), entry.getValue().toString().trim());
            } else {
                treeMap.put(entry.getKey(), JSON.toJSONString(entry.getValue()));
            }
        }
        return append(treeMap);
    }

    private static String append(Map<String, String> map) {
        StringBuilder builder = new StringBuilder();
        for (Map.Entry<String, String> entry : map.entrySet()) {
            builder.append(entry.getKey());
            builder.append("=");
            builder.append(entry.getValue());
            builder.append("&");
        }
        if (builder.length() > 0) {
            builder.deleteCharAt(builder.length() - 1);
        }
        return builder.toString().trim();
    }


    private static boolean contains(Object[] array, Object objectToFind) {
        return indexOf(array, objectToFind) != -1;
    }

    private static int indexOf(Object[] array, Object objectToFind) {
        if (array == null) {
            return INDEX_NOT_FOUND;
        }
        if (objectToFind == null) {
            for (int i = 0; i < array.length; i++) {
                if (array[i] == null) {
                    return i;
                }
            }
        } else if (array.getClass().getComponentType().isInstance(objectToFind)) {
            for (int i = 0; i < array.length; i++) {
                if (objectToFind.equals(array[i])) {
                    return i;
                }
            }
        }
        return INDEX_NOT_FOUND;
    }


    private static boolean isBlank(String str) {
        int strLen;
        if (str == null || (strLen = str.length()) == 0) {
            return true;
        }
        for (int i = 0; i < strLen; i++) {
            if (!Character.isWhitespace(str.charAt(i))) {
                return false;
            }
        }
        return true;
    }


    public static Map<String, String> toStringMap(Map<String, Object> map) {
        Map<String, String> stringMap = new HashMap<>();
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            if (entry.getValue() instanceof String) {
                stringMap.put(entry.getKey(), entry.getValue().toString().trim());
            } else {
                stringMap.put(entry.getKey(), JSON.toJSONString(entry.getValue()));
            }
        }
        return stringMap;
    }
}
