package cn.com.duiba.constant.hsbc;

import cn.com.duiba.biz.credits.strategy.ApiStrategyRouter;
import cn.com.duiba.biz.credits.strategy.Impl.HsbcApiStrategy;
import com.alibaba.fastjson.JSON;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.Base64Utils;

import javax.annotation.Resource;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Map;
import java.util.Set;

/**
 * 汇丰汇选配置
 *
 * @author fja
 * @since 2021-12-06
 */
@Configuration
@ConfigurationProperties(prefix = "hsbc")
public class HsbcConfig implements InitializingBean {

    @Resource
    private HsbcApiStrategy hsbcApiStrategy;

    private final Logger log = LoggerFactory.getLogger(this.getClass());

    private Set<Long> appIds = Sets.newHashSet(-1L);

    private final static String ALGORITHM_RSA = "RSA";


    /**
     * 兑吧私钥配置
     */
    private String appId2RsaDuibaPrivateKeyConf;
    /**
     * {汇丰应用id: rsaPrivateKey配置}
     */
    private Map<String, String> appId2RsaDuibaPrivateKeyConfMap = Maps.newHashMap();
    private final Map<Long, PrivateKey> appId2RsaDuibaPrivateKeyMap = Maps.newHashMap();


    /**
     * 汇丰业务rsa公钥 apollo配置集合
     */
    private String appId2HsbcKeyConf;
    /**
     * 汇丰jwt rsa私钥 apollo配置集合
     */
    private String appId2JwtKeyConf;
    /**
     * 免登用户信息查询接口配置集合
     */
    private String appId2UserInfoUrlConf;

    /**
     * {汇丰应用id: rsaPublicKey配置}
     */
    private Map<String, String> appId2HsbcKeyConfMap = Maps.newHashMap();

    /**
     * {汇丰应用id: jwtPrivateKey配置}
     */
    private Map<String, String> appId2JwtKeyConfMap = Maps.newHashMap();

    /**
     * {汇丰应用id: url配置}
     */
    private Map<String, String> appId2UserInfoUrlMap = Maps.newHashMap();

    /**
     * {汇丰应用id: key配置}
     */
    private final Map<Long, PublicKey> appId2HsbcPublicKeyMap = Maps.newHashMap();


    @Override
    public void afterPropertiesSet() {
        ApiStrategyRouter.register(appIds, hsbcApiStrategy);

        try {
            //配置 字符串转换成Map
            this.appId2HsbcKeyConfMap = JSON.parseObject(this.appId2HsbcKeyConf, Map.class);
            this.appId2JwtKeyConfMap = JSON.parseObject(this.appId2JwtKeyConf, Map.class);
            this.appId2UserInfoUrlMap = JSON.parseObject(this.appId2UserInfoUrlConf, Map.class);
            this.appId2RsaDuibaPrivateKeyConfMap = JSON.parseObject(this.appId2RsaDuibaPrivateKeyConf, Map.class);

            //批量生成公钥
            batchGeneratePublicKey();

            //批量生成私钥
            batchGeneratePrivateKey();
        } catch (Exception e) {
            log.error("[hsbc] 秘钥初始化失败", e);
        }
    }


    /**
     * 生成私钥
     *
     * @param privateKeyConf 私钥配置
     * @return 私钥
     * @throws Exception -
     */
    private PrivateKey generatePrivateKey(String privateKeyConf) throws Exception {
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(Base64Utils.decodeFromString(privateKeyConf));
        KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM_RSA);
        return keyFactory.generatePrivate(keySpec);
    }


    /**
     * 生成公钥
     *
     * @param publicKeyConf 公钥配置
     * @return 公钥
     * @throws Exception -
     */
    private PublicKey generatePublicKey(String publicKeyConf) throws Exception {
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64Utils.decodeFromString(publicKeyConf));
        KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM_RSA);
        return keyFactory.generatePublic(keySpec);
    }

    /**
     * 是否汇丰汇选的app
     *
     * @param appId 应用id
     * @return true: 汇丰app
     */
    public boolean isHsbcApp(Long appId) {
        return appIds.contains(appId);
    }


    /**
     * 批量生成汇丰公钥 {appId: key}
     *
     * @throws Exception -
     */
    public void batchGeneratePublicKey() throws Exception {
        for (Map.Entry<String, String> entry : this.appId2HsbcKeyConfMap.entrySet()) {
            if (StringUtils.isBlank(entry.getValue())) {
                log.error(String.format("[hsbc] appId = %s公钥配置为空", entry.getKey()));
                continue;
            }
            this.appId2HsbcPublicKeyMap.put(Long.parseLong(entry.getKey()), generatePublicKey(entry.getValue()));
        }
    }


    /**
     * 批量生成兑吧私钥 {appId: key}
     *
     * @throws Exception -
     */
    public void batchGeneratePrivateKey() throws Exception {
        for (Map.Entry<String, String> entry : this.appId2RsaDuibaPrivateKeyConfMap.entrySet()) {
            if (StringUtils.isBlank(entry.getValue())) {
                log.error(String.format("[hsbc] appId = %s私钥配置为空", entry.getKey()));
                continue;
            }
            this.appId2RsaDuibaPrivateKeyMap.put(Long.parseLong(entry.getKey()), generatePrivateKey(entry.getValue()));
        }
    }


    public Set<Long> getAppIds() {
        return appIds;
    }

    public void setAppIds(Set<Long> appIds) {
        this.appIds = appIds;
    }

    public String getAppId2RsaDuibaPrivateKeyConf() {
        return appId2RsaDuibaPrivateKeyConf;
    }

    public void setAppId2RsaDuibaPrivateKeyConf(String appId2RsaDuibaPrivateKeyConf) {
        this.appId2RsaDuibaPrivateKeyConf = appId2RsaDuibaPrivateKeyConf;
    }

    public Map<String, String> getAppId2RsaDuibaPrivateKeyConfMap() {
        return appId2RsaDuibaPrivateKeyConfMap;
    }

    public void setAppId2RsaDuibaPrivateKeyConfMap(Map<String, String> appId2RsaDuibaPrivateKeyConfMap) {
        this.appId2RsaDuibaPrivateKeyConfMap = appId2RsaDuibaPrivateKeyConfMap;
    }

    public Map<Long, PrivateKey> getAppId2RsaDuibaPrivateKeyMap() {
        return appId2RsaDuibaPrivateKeyMap;
    }

    public String getAppId2HsbcKeyConf() {
        return appId2HsbcKeyConf;
    }

    public void setAppId2HsbcKeyConf(String appId2HsbcKeyConf) {
        this.appId2HsbcKeyConf = appId2HsbcKeyConf;
    }

    public String getAppId2JwtKeyConf() {
        return appId2JwtKeyConf;
    }

    public void setAppId2JwtKeyConf(String appId2JwtKeyConf) {
        this.appId2JwtKeyConf = appId2JwtKeyConf;
    }

    public String getAppId2UserInfoUrlConf() {
        return appId2UserInfoUrlConf;
    }

    public void setAppId2UserInfoUrlConf(String appId2UserInfoUrlConf) {
        this.appId2UserInfoUrlConf = appId2UserInfoUrlConf;
    }

    public Map<String, String> getAppId2HsbcKeyConfMap() {
        return appId2HsbcKeyConfMap;
    }

    public void setAppId2HsbcKeyConfMap(Map<String, String> appId2HsbcKeyConfMap) {
        this.appId2HsbcKeyConfMap = appId2HsbcKeyConfMap;
    }

    public Map<String, String> getAppId2JwtKeyConfMap() {
        return appId2JwtKeyConfMap;
    }

    public void setAppId2JwtKeyConfMap(Map<String, String> appId2JwtKeyConfMap) {
        this.appId2JwtKeyConfMap = appId2JwtKeyConfMap;
    }

    public Map<String, String> getAppId2UserInfoUrlMap() {
        return appId2UserInfoUrlMap;
    }

    public void setAppId2UserInfoUrlMap(Map<String, String> appId2UserInfoUrlMap) {
        this.appId2UserInfoUrlMap = appId2UserInfoUrlMap;
    }

    public Map<Long, PublicKey> getAppId2HsbcPublicKeyMap() {
        return appId2HsbcPublicKeyMap;
    }
}
