/**
 * Project Name:tuia-core-api<br>
 * File Name:AdvertMaterialClient.java<br>
 * Package Name:cn.com.duiba.tuia.core.api.client<br>
 * Date:2016年12月23日下午5:42:22<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.core.api.client;

import cn.com.duiba.tuia.core.api.constant.RedisKeys;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import cn.com.duiba.tuia.core.api.dto.AdvertMaterialAppRelationDto;
import cn.com.duiba.tuia.core.api.dto.AdvertMaterialCache;
import cn.com.duiba.tuia.core.api.dto.rsp.RspMaterialList;
import cn.com.duiba.tuia.core.api.remoteservice.RemoteAdvertMaterialService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.tuia.advert.cache.CacheKeyTool;

import com.alibaba.fastjson.JSONObject;

/**
 * ClassName: AdvertMaterialClient <br/>
 * Function: 广告素材Client. <br/>
 * date: 2016年12月23日 下午5:42:22 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
public class AdvertMaterialClient extends BaseClient {

    @Autowired
    private RemoteAdvertMaterialService remoteAdvertMaterialService;

    /**
     * 素材是否是新素材.<br>
     * <ol>
     * <li>返回true：新素材</li>
     * <li>返回false：老素材</li>
     * <li>返回true：调用接口发生异常</li>
     * </ol>
     *
     * @param advertId the advert id
     * @param materialId the material id
     * @param appId the app id
     * @return true, if checks if is new of material
     */
    public Boolean isNewOfMaterial(Long advertId, Long materialId, Long appId) {
        String key = CacheKeyTool.getCacheKey(RedisKeys.K03,materialId,appId);
        String value = stringRedisTemplate.opsForValue().get(key);
        int status;
        if (StringUtils.isBlank(value)) {
            DubboResult<Integer> dubboResult = remoteAdvertMaterialService.getNewOldStatus(advertId, materialId, appId);
            if (!dubboResult.isSuccess()) {
                // 调用失败，返回null
                return true;
            }
            status = dubboResult.getResult();
        } else {
            try {
                status = Integer.parseInt(value);
            } catch (NumberFormatException e) {
                stringRedisTemplate.delete(key);
                return true;
            }
        }

        return AdvertMaterialAppRelationDto.IS_NEW == status ? true : false;
    }

    /**
     * 查询素材列表(针对app维度).<br>
     * <ol>
     * <li>返回null：调用接口发生异常</li>
     * </ol>
     *
     * @param advertId the advert id
     * @param appId the app id
     * @return the material list
     */
    public RspMaterialList getMaterialList(Long advertId, Long appId) {
        String value = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisKeys.K04,advertId,appId));
        if (StringUtils.isBlank(value)) {
            DubboResult<RspMaterialList> dubboResult = remoteAdvertMaterialService.updateMaterialList(advertId, appId);
            if (!dubboResult.isSuccess()) {
                // 接口调用失败
                return null;
            }

            return dubboResult.getResult();
        } else {
            return JSONObject.parseObject(value, RspMaterialList.class);
        }
    }

    /**
     * 查询素材详情.<br>
     * <ol>
     * <li>返回null：调用接口发生异常</li>
     * </ol>
     *
     * @param materialId the material id
     * @return the material detail
     */
    public AdvertMaterialCache getMaterialCache(Long materialId) {
        String value = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisKeys.K06,materialId));
        if (StringUtils.isBlank(value)) {
            DubboResult<AdvertMaterialCache> dubboResult = remoteAdvertMaterialService.updateMaterialCache(materialId);
            if (!dubboResult.isSuccess()) {
                // 接口调用失败
                return null;
            }

            return dubboResult.getResult();
        } else {
            return JSONObject.parseObject(value, AdvertMaterialCache.class);
        }
    }
}
