package cn.com.duiba.tuia.ssp.center.api.tool;

import org.apache.commons.lang3.ObjectUtils;

import javax.swing.plaf.synth.SynthScrollBarUI;
import java.math.BigDecimal;
import java.util.Objects;

/**
 * 数据计算公式
 */
public class DataMathUtil {

    private static class DataMathUtilHolder {
        private static final DataMathUtil INSTANCE = new DataMathUtil();
    }

    /**
     * 获取DataMathUtil 对象
     *
     * @return
     */
    public static final DataMathUtil getInstance() {
        return DataMathUtilHolder.INSTANCE;
    }

    /**
     * 精确到2位小数
     *
     * @param v1 被除数
     * @param v2 除数
     * @param s  保留小数的位数 默认保留两位
     * @return 两个参数的商(Long)
     */
    public Long calculateDivide(Long v1, Long v2, Integer s) {
        if (Objects.isNull(s)) {
            s = SCALE.SCALE_NUM2.getNum();
        }
        BigDecimal bigDecimal = divide(v1, v2, s, null);
        return Objects.isNull(bigDecimal) ? 0L : bigDecimal.longValue();
    }

    /**
     * 乘法计算
     *
     * @param v1
     * @param number
     * @return
     */
    public Long calculateMultiply(Long v1,Float number) {
        if (number == null) {
            return null;
        }
        return BigDecimal.valueOf(v1).multiply(BigDecimal.valueOf(number)).setScale(0, BigDecimal.ROUND_HALF_UP).longValue();
    }

       /**
     * 返回浮点类型小数
     *
     * @param v1 被除数
     * @param v2 除数
     * @param s  保留小数的位数 默认保留两位
     * @return 返回浮点类型小数
     */
    public Float calculateDivideToFloat(Long v1, Long v2, Integer s) {
        if (Objects.isNull(s)) {
            s = SCALE.SCALE_NUM2.getNum();
        }
        BigDecimal bigDecimal = divide(v1, v2, s, null);
        return Objects.isNull(bigDecimal) ? 0F : bigDecimal.floatValue();
    }

    /**
     * 例如计算每千次展示收益可以采用此方法，该方法精确保留两位小数
     *
     * @param v1 被除数
     * @param v2 除数
     * @param s  保留小数的位数 默认保留两位
     * @param v3 乘数
     * @return
     */
    public Long calculateDivide(Long v1, Long v2, Integer s, Integer v3) {
        Objects.requireNonNull(v3, "乘数不能为null");
        if (Objects.isNull(s)) {
            s = SCALE.SCALE_NUM2.getNum();
        }
        BigDecimal bigDecimal = divide(v1, v2, s, v3);
        return Objects.isNull(bigDecimal) ? 0L : bigDecimal.longValue();
    }


    /**
     * 精确除法运算 . 当发生除不尽情况时 , 由scale参数指 定精度 , 以后数字四舍五入
     *
     * @param v1    被除数
     * @param v2    除数
     * @param scale 表示表示需要精确到小数点以后几位
     * @return 两个参数的商(BigDecimal)
     */
    private static BigDecimal divide(Long v1, Long v2, Integer scale, Integer v3) {

        if (checkParam(v1, v2, scale) == false) {
            return null;
        }

        BigDecimal divisor;
        if (ObjectUtils.notEqual(v3, null)) {
            divisor = new BigDecimal(v1 * v3);
        } else {
            divisor = new BigDecimal(v1);
        }
        BigDecimal Dividend = new BigDecimal(v2);
        return divisor.divide(Dividend, scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * 检查参数
     *
     * @param v1    被除数
     * @param v2    除数
     * @param scale 表示表示需要精确到小数点以后几位
     */
    private static boolean checkParam(Long v1, Long v2, Integer scale) {
        if (Objects.isNull(v1) || Objects.isNull(v2) || Objects.isNull(scale)) {
            return false;
        }
        if (v2 == 0) {
            return false;
        }
        return scale >= 0;
    }

    /**
     * 两个数相减
     *
     * @param minuend:被减数
     * @param meiosis:减数
     * @return
     * @since JDK 1.6
     */
    public static Long calculateProfitData(Long minuend, Long meiosis) {
        Long result = 0L;
        if (minuend != null && meiosis != null) {
            BigDecimal bigMinuend = new BigDecimal(minuend);
            BigDecimal bigMeiosis = new BigDecimal(meiosis);
            return bigMinuend.subtract(bigMeiosis).longValue();
        }
        return result;
    }

    /**
     * 小数点枚举
     */
    public enum SCALE {
        SCALE_NUM0(0), //保留0位

        SCALE_NUM1(1), //保留1位

        SCALE_NUM2(2), //保留2位

        SCALE_NUM3(3); //保留3位

        private Integer num;

        SCALE(Integer num) {
            this.num = num;
        }

        public Integer getNum() {
            return num;
        }

    }
}
