package cn.com.duiba.user.service.api.remoteservice.liveuser;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.user.service.api.dto.liveuser.BizUserIdDto;
import cn.com.duiba.user.service.api.dto.liveuser.LiveUserDto;
import cn.com.duiba.user.service.api.dto.liveuser.LiveUserNickNameAvatarDto;
import cn.com.duiba.user.service.api.dto.liveuser.UserInfo4LoginApiDto;
import cn.com.duiba.user.service.api.dto.user.LiveUserAuthPhoneNicknameDto;
import cn.com.duiba.user.service.api.param.liveuser.LiveUserEsSearchParam;
import cn.com.duiba.user.service.api.param.liveuser.LiveUserSearchParam;
import cn.com.duiba.user.service.api.param.liveuser.SyncUserApiParam;
import org.apache.commons.lang3.tuple.Pair;

import java.util.List;
import java.util.Map;

/**
 * 直播用户服务
 * @author 周舟
 * @date 2020-3-28 10:22:31
 * @version 1.0
 */
@AdvancedFeignClient
public interface RemoteLiveUserApiService {

    /**
     * 列表查询
     * @param queryParam 查询参数
     * @return 直播用户集合
     */
    List<LiveUserDto> findList(LiveUserSearchParam queryParam);

    /**
     * 保存更新直播用户
     * @param dto 直播用户
     * @return 主键，为0，说明新增失败
     */
    Long saveAndUpdate(LiveUserDto dto);

    /**
     * 根据id查询
     * @param id 主键id
     * @return 直播用户
     */
    LiveUserDto findById(Long id);

    /**
     * 根据第三方用户ID和类型查询用户ID
     * @param bizUserId 业务方用户ID
     * @param bizType 业务类型
     * @see cn.com.duiba.user.service.api.enums.liveuser.BizTypeApiEnum
     * @return 主键id
     */
    Long findIdByBizUserIdAndType(String bizUserId, Integer bizType);

    /**
     * 查询授权电话
     * @param bizUserId 业务方用户ID
     * @param bizType 业务类型
     * @see cn.com.duiba.user.service.api.enums.liveuser.BizTypeApiEnum
     * @return 授权电话
     */
    String findAuthPhoneByBizUserIdAndType(String bizUserId, Integer bizType);

    /**
     * 批量查询直播系统用户ID
     * @param bizUserIdList 业务方用户ID集合
     * @param bizType 业务类型
     * @see cn.com.duiba.user.service.api.enums.liveuser.BizTypeApiEnum
     * @return 业务方ID和直播用户ID
     */
    List<BizUserIdDto> findIdsByBizUserIdsAndType(List<String> bizUserIdList, Integer bizType);

    /**
     * 根据直播用户ID集合，批量查询业务方用户ID和授权手机号
     * @param ids 直播用户ID集合
     * @return 业务方ID和授权手机号
     */
    List<BizUserIdDto> findBizUserIdAndAuthPhoneByIds(List<Long> ids);

    /**
     * 批量查询授权手机号
     * @param ids 主键ID集合
     * @return key：主键，value：授权手机号
     */
    Map<Long, String> findAuthPhoneByIds(List<Long> ids);

    /**
     * 批量查询主键
     * @param authPhones 授权手机号集合
     * @return key：授权手机号，value：主键
     */
    Map<String, Long> findIdByAuthPhones(List<String> authPhones);


    /**
     * 根据id查询
     * @param ids 主键id列表
     * @return 直播用户
     */
    List<LiveUserDto> findByIds(List<Long> ids);

    /**
     * 批量查询昵称和头像
     * @param ids 主键集合
     * @return 昵称头像集合
     */
    List<LiveUserNickNameAvatarDto> findNicknameAvatarByIds(List<Long> ids);

    /**
     * 更新授权手机号
     * @param id 主键
     * @param authPhone 授权手机号
     * @param authPhoneCountryCode 授权手机号的区号
     * @return 执行结果
     */
    int updateAuthPhone(Long id, String authPhone, String authPhoneCountryCode);

    /**
     * 更新授权手机号
     * @param bizUserId 业务方用户ID
     * @param bizType 业务类型
     * @see cn.com.duiba.user.service.api.enums.liveuser.BizTypeApiEnum
     * @param authPhone 授权手机号
     * @param authPhoneCountryCode 授权手机号的区号
     * @return 执行结果
     */
    @Deprecated
    int updateAuthPhoneByBiz(String bizUserId, Integer bizType, String authPhone, String authPhoneCountryCode);


    /**
     * 更新授权手机号
     * @param bizUserId 业务方用户ID
     * @param bizType 业务类型
     * @see cn.com.duiba.user.service.api.enums.liveuser.BizTypeApiEnum
     * @param authPhone 授权手机号
     * @param authPhoneCountryCode 授权手机号的区号
     * @param liveId 直播间id
     * @return 执行结果
     */
    int updateAuthPhoneByBizAndLiveId(String bizUserId, Integer bizType, String authPhone, String authPhoneCountryCode,Long liveId);



    /**
     * 根据id删除
     * @param id 主键id
     * @return 是否成功 1 成功 其他失败
     */
    Long deleteById(Long id);


    /**
     * 根据授权手机号查询
     * @param authPhone 授权手机号
     * @return 用户信息
     */
    List<LiveUserDto> findByAuthPhone(String authPhone);


    /**
     * 同步第三方用户信息，并返回直播系统用户信息
     * @param param 第三方用户信息
     * @return 本系统用户信息
     */
    UserInfo4LoginApiDto getAndSyncUserV2(SyncUserApiParam param);

    /**
     * 根据三方业务id 业务类型获取 直播系统用户id
     * @param bizUserId 三方业务用户id
     * @param bizType 业务类型
     * @see cn.com.duiba.user.service.api.enums.liveuser.BizTypeApiEnum
     * @return  加密的客集集直播系统用户id
     */
    String getLiveUserIdByBizUserIdAndType(String bizUserId,Integer bizType);


    /**
     * 批量同步用户
     * @param params
     * @return
     */
    List<UserInfo4LoginApiDto> batchGetAndSyncUserV2(List<SyncUserApiParam> params);

    /**
     * 根据昵称搜索用户信息
     *
     * @param nickname 昵称
     * @return 用户信息集合
     */
    List<LiveUserNickNameAvatarDto> searchByNickname(String nickname);

    /**
     * 从es中查询 直播用户信息
     * @param searchParam 查询条件
     * @return key:总数 val:列表
     */
    Pair<Long,List<LiveUserNickNameAvatarDto>> findByConditionFromEs(LiveUserEsSearchParam searchParam);

    /**
     * 根据授权手机号批量查询
     *
     * @param authPhones 授权手机号集合
     * @return 用户id集合
     */
    List<LiveUserDto> findByAuthPhones(List<String> authPhones);


    /**
     * 批量查询授权手机号及昵称
     * @param ids 用户ID集合
     * @return 授权手机号
     */
    List<LiveUserAuthPhoneNicknameDto> findAuthPhoneNicknameByIds(List<Long> ids);

    /**
     * 根据第三方userID 和 bizType查询用户信息
     * @param bizUserId 第三方userID
     * @param bizType 第三方类型
     * @return 用户信息
     */
    LiveUserDto findByBizUserIdAndBizType(String bizUserId, Integer bizType);

    /**
     * 更新头像及昵称
     * @param id 主键
     * @param nickname 昵称
     * @param avatar 头像
     * @return 执行结果
     */
    int updateNicknameAvatar(Long id, String nickname, String avatar);

    /**
     * 根据id查询缓存
     * @param id 直播用户id
     * @return 用户id
     */
    Long findKjjUserIdByIdCache(Long id);

    /**
     * 根据userId查询缓存
     * @param userId 用户id
     * @return 直播用户id
     */
    Long findIdByKjjUserIdCache(Long userId);

}
