package cn.com.duiba.wolf.cache;

import cn.com.duiba.wolf.utils.NumberUtils;
import net.rubyeye.xmemcached.*;
import net.rubyeye.xmemcached.auth.AuthInfo;
import net.rubyeye.xmemcached.command.BinaryCommandFactory;
import net.rubyeye.xmemcached.command.KestrelCommandFactory;
import net.rubyeye.xmemcached.command.TextCommandFactory;
import net.rubyeye.xmemcached.transcoders.SerializingTranscoder;
import net.rubyeye.xmemcached.utils.AddrUtil;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * @author huangwq
 */
public class XMemcacheClient implements InitializingBean, CacheClient {

    private static Logger           logger                     = LoggerFactory.getLogger(XMemcacheClient.class);

    /** 缓存类的名称 */
    private final String            CACHE_NAME                 = XMemcacheClient.class.getName();

    /** 表示缓存永不失效 */
    private static final int        CACHE_NO_EXPIRY            = 0;

    /** 计数器的默认初始值 */
    private static final int        COUNTER_DEFAULT_INIT_VALUE = 0;

    private int                     connectionPoolSize;
    private int                     connectionTimeout          = MemcachedClient.DEFAULT_CONNECT_TIMEOUT;
    private int                     operationTimeout           = 1000;//(int) MemcachedClient.DEFAULT_OP_TIMEOUT;

    // 默认保存2天
    private static final int        DEFAULT_EXP                = 3600 * 24 * 2;

    private MemcachedSessionLocator sessionLocator;
    private MemcachedClient         memcachedClient;

    private String                  servers;
    private String                  authInfos;
    private boolean enableHeartBeat;
    //TEXT/BINARY
    private String protocol;

    @Override
    public <T> T get(String key) {
        key = getStringNoBlank(key);
        try {
            return memcachedClient.get(key);
        } catch (IllegalArgumentException e) {
            logger.error(e.getMessage() + ",key is:" + key);
        } catch (Exception e) {
            logger.error("memcachedClient get(String) error,key:" + key, e);
        }
        return null;
    }

    @Override
    public long getLong(String key) {
        key = getStringNoBlank(key);
        try {
            Object obj = memcachedClient.get(key);
            if (obj == null) {
                return 0;
            } else if (obj instanceof String) {
                return NumberUtils.parseLong((String) obj, 0);
            } else if (obj instanceof Integer) {
                return (Integer) obj;
            } else if (obj instanceof Long) {
                return (Long) obj;
            }
        } catch (IllegalArgumentException e) {
            logger.error(e.getMessage() + ",key is:" + key);
        } catch (Exception e) {
            logger.error("memcachedClient get(String) error,key:" + key, e);
        }
        return 0;
    }

    public Map<String, Object> get(Collection<String> keys) {
        Map<String, Object> value = null;
        try {
            value = memcachedClient.get(keys);
        } catch (Exception e) {
            logger.error("memcachedClient get(Collection<String>) error", e);
        }
        return value;
    }

//    @Override
//    public void set(String key, Object value) {
//        set(key, value, DEFAULT_EXP, TimeUnit.SECONDS);
//    }

    @Override
    public void set(String key, Object value, int expSeconds) {
        set(key, value, expSeconds, TimeUnit.SECONDS);
    }

    @Override
    public void set(String key, Object value, int expiry, TimeUnit unit) {
        if (value == null) {
            return;
        }
        key = getStringNoBlank(key);

        int exp = getValidExpiryTime(expiry, unit);
        try {
            memcachedClient.set(key, exp, value);// boolean success =
        } catch (Exception e) {
            logger.error("memcachedClient set() error,key:" + key, e);
        }
    }

    @Override
    public void incr(String key, long delta) {
        incr(key, delta, delta, DEFAULT_EXP, TimeUnit.MILLISECONDS);
    }

    @Override
    public void incr(String key, long delta, long expiry, TimeUnit unit) {
        incr(key, delta, delta, expiry, unit);
    }

    @Override
    public void incr(String key, long delta, long initValue, long expiry, TimeUnit unit) {
        // long count = COUNTER_DEFAULT_INIT_VALUE;

        int exp = getValidExpiryTime(expiry, unit);
        try {
            // count =
            long count = memcachedClient.incr(key, delta, initValue, MemcachedClient.DEFAULT_OP_TIMEOUT, exp);
            logger.debug("key:{},afterInce:{}", key, count);
        } catch (Exception e) {
            logger.error("memcachedClient incr() error,key:" + key + this.get(key), e);
        }
    }

    @Override
    public void decr(String key, long delta) {
        decr(key, delta, COUNTER_DEFAULT_INIT_VALUE, DEFAULT_EXP, TimeUnit.MILLISECONDS);
    }

    @Override
    public void decr(String key, long delta, long expiry, TimeUnit unit) {
        decr(key, delta, COUNTER_DEFAULT_INIT_VALUE, expiry, unit);
    }

    @Override
    public void decr(String key, long delta, long initValue, long expiry, TimeUnit unit) {
        // long count = COUNTER_DEFAULT_INIT_VALUE;

        int exp = getValidExpiryTime(expiry, unit);
        try {
            // count =
            memcachedClient.decr(key, delta, initValue, MemcachedClient.DEFAULT_OP_TIMEOUT, exp);
        } catch (Exception e) {
            logger.error("memcachedClient decr() error,key:" + key, e);
        }
    }

    @Override
    public void remove(String key) {
        key = getStringNoBlank(key);
        try {
            memcachedClient.delete(key);
        } catch (Exception e) {
            logger.error("memcachedClient delete() error,key:" + key, e);
        }
    }

    public void setSessionLocator(MemcachedSessionLocator sessionLocator) {
        this.sessionLocator = sessionLocator;
    }

    public int getConnectionPoolSize() {
        return connectionPoolSize;
    }

    public void setConnectionPoolSize(int connectionPoolSize) {
        this.connectionPoolSize = connectionPoolSize;
    }

    public int getConnectionTimeout() {
        return connectionTimeout;
    }

    public void setConnectionTimeout(int connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
        if (memcachedClient != null) {
            memcachedClient.setConnectTimeout(connectionTimeout);
        }
    }

    public int getOperationTimeout() {
        return operationTimeout;
    }

    public void setOperationTimeout(int operationTimeout) {
        this.operationTimeout = operationTimeout;
        if (memcachedClient != null) {
            memcachedClient.setOpTimeout(operationTimeout);
        }
    }

    public String getServers() {
        return servers;
    }

    public void setServers(String servers) {
        this.servers = servers;
    }

    public String getAuthInfos() {
        return authInfos;
    }

    public void setAuthInfos(String authInfos) {
        this.authInfos = authInfos;
    }



//    private Integer[] getWeights(int serverCount) {
//        Integer[] weights = new Integer[serverCount];
//        for (int i = 0; i < serverCount; i++) {
//            weights[i] = 1;
//        }
//        return weights;
//    }

    /**
     * @Title: getStringNoBlank
     * @Description: 清空字符串内空格
     * @param @param str
     * @param @return 设定文件
     * @author yulc
     * @return String 返回类型
     * @throws
     */
    public static String getStringNoBlank(String str) {
        if (str != null && !"".equals(str)) {
            return str.replaceAll("\\s*|\t|\r|\n", "").replaceAll("　", "").replaceAll("&nbsp;", "");
        } else {
            return str;
        }
    }

    private void initClient() {
        if (memcachedClient != null) {
            logger.info(CACHE_NAME + " already initialized");
            return;
        }

        String[] servers = StringUtils.split(this.getServers(), ",");
        String[] authInfos = StringUtils.split(this.getAuthInfos(), ",");
        List<InetSocketAddress> addressList = new ArrayList<InetSocketAddress>();

        Map<InetSocketAddress, AuthInfo> authInfoMap = new HashMap<InetSocketAddress, AuthInfo>();
        int i = 0;
        for (String server : servers) {
            InetSocketAddress address = AddrUtil.getOneAddress(server);
            addressList.add(address);
            if (authInfos != null && authInfos.length > i) {
                String authInfo = authInfos[i];
                String[] temp = authInfo.split(":");
                if (temp.length >= 1) {
                    authInfoMap.put(address, AuthInfo.plain(temp[0], temp[1]));
                    logger.info("auth " + server + " with username:" + temp[0] + ",password:" + temp[1]);
                }
            }
            i++;
        }

        MemcachedClientBuilder builder = new XMemcachedClientBuilder(addressList);// weights
        builder.setAuthInfoMap(authInfoMap);

        if (sessionLocator != null) {
            builder.setSessionLocator(sessionLocator);
        }

        SerializingTranscoder transcoder = new SerializingTranscoder();
        transcoder.setCompressionThreshold(1024);
        builder.setTranscoder(transcoder);

        //TEXT/BINARY 默认BINARY
        if (StringUtils.isBlank(protocol) || protocol.equals("BINARY")) {
            builder.setCommandFactory(new BinaryCommandFactory());
        }else if(protocol.equals("TEXT")){
            builder.setCommandFactory(new TextCommandFactory());
        }else if(protocol.equals("KESTREL")){
            builder.setCommandFactory(new KestrelCommandFactory());
        }
        builder.setKeyProvider(new KeyProvider() {

            @Override
            public String process(String key) {
                if (key == null) {
                    return null;
                }
                return key.trim();// "prefix_" +
            }
        });
        builder.setHealSessionInterval(2000);//连接断开两秒后尝试重连
        // builder.setSessionLocator(new KetamaMemcachedSessionLocator());// 一致性哈希（consistent hash)
        // builder.setSessionLocator(new ElectionMemcachedSessionLocator());
        // builder.setTranscoder(new SerializingTranscoder());
        //builder.getConfiguration().setCheckSessionTimeoutInterval();
        //builder.getConfiguration().setSessionIdleTimeout();

        // builder.getTranscoder().setPackZeros(true);
        // builder.getTranscoder().setPrimitiveAsString(false);

        // memcachedClient.setOptimizeGet(true);
        // memcachedClient.setOptimizeMergeBuffer(true);
        // memcachedClient.setPrimitiveAsString(false);
        // memcachedClient.setSanitizeKeys(false);

        if (connectionPoolSize > 0) {
            builder.setConnectionPoolSize(connectionPoolSize);// 设置多个连接会有bug。1个足够了
        }
        // builder.setSocketOption(StandardSocketOption.SO_RCVBUF, 32 * 1024); // 设置接收缓存区为32K，默认16K
        // builder.setSocketOption(StandardSocketOption.SO_SNDBUF, 16 * 1024); // 设置发送缓冲区为16K，默认为8K

        try {
            memcachedClient = builder.build();
            memcachedClient.setConnectTimeout(connectionTimeout);
            memcachedClient.setOpTimeout(operationTimeout);
            memcachedClient.setEnableHeartBeat(enableHeartBeat);
            logger.info(addressList + " initialized{}");
        } catch (IOException e) {
            logger.error("Initialize " + getClass() + " error", e);
        }
    }

    /**
     * 添加 Memcached 服务器.
     *
     * @param servers Memcached 服务器地址, 每个服务器的格式为: [host1]:[port1] [host2]:[port2]. 例如, 192.168.0.1:11211.
     * @throws IOException 如果在添加过程中失败, 会抛出此异常
     */
    public void addServer(String... servers) throws IOException {
        if (servers == null || servers.length == 0) {
            throw new IllegalArgumentException("servers can't be null");
        }

        StringBuilder hostList = new StringBuilder();
        for (String server : servers) {
            hostList.append(server).append(" ");
        }
        hostList.deleteCharAt(hostList.length() - 1);

        memcachedClient.addServer(hostList.toString());
        logger.info("Memcached server(s) added: [" + hostList + "]");
    }

    /**
     * 添加Memcached服务器.
     *
     * @param servers Memcached服务器地址, 每个服务器的格式为: [host1]:[port1] [host2]:[port2]. 例如, 192.168.0.1:11211.
     * @param weights 服务器的权重, 例如：{1, 1, 2}
     * @throws IOException 如果在添加过程中失败, 会抛出此异常
     */
    public void addServers(String[] servers, int[] weights) throws IOException {
        if (servers == null || servers.length == 0) {
            throw new IllegalArgumentException("servers can't be null");
        }

        if (weights != null && weights.length < servers.length) {
            throw new IllegalArgumentException("weights can't be less than servers");
        }

        for (int i = 0; i < servers.length; i++) {
            InetSocketAddress address = AddrUtil.getOneAddress(servers[i]);
            memcachedClient.addServer(address, weights[i]);
        }

    }

    /**
     * 删除 Memcached 服务器, server 的格式为: [ip]:[port]. 例如, 192.168.0.1:11211.
     *
     * @param servers Memcached 服务器地址
     * @throws IOException 如果在删除过程中失败, 会抛出此异常
     */
    public void removeServer(String... servers) {
        if (servers == null || servers.length == 0) {
            throw new IllegalArgumentException("servers can't be null");
        }

        StringBuilder hostList = new StringBuilder();
        for (String server : servers) {
            hostList.append(server).append(" ");
        }
        hostList.deleteCharAt(hostList.length() - 1);

        memcachedClient.removeServer(hostList.toString());
        logger.info("Memcached server(s) removed: [" + hostList + "]");
    }

    private int getValidExpiryTime(long expiry, TimeUnit unit) {
        int exp = (int) unit.toSeconds(expiry);
        return (exp > CACHE_NO_EXPIRY) ? exp : DEFAULT_EXP;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        this.initClient();
    }

    @Override
    public void putAttribute(String category, String key, Object value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void putAttribute(String category, String key, Object value, int exp) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Map<String, Object> getAttribute(String category) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Object getAttribute(String category, String key) {
        throw new UnsupportedOperationException();
    }

    public static void main(String[] args) throws Exception {
        XMemcacheClient client = new XMemcacheClient();
        // client.setServers("115.29.179.116:11311,115.29.179.116:11312,115.29.179.116:11313");//
        client.setServers("42.121.192.49:11311");//
        // client.setAuthInfos("5200e98ceb8711e3:695a_3a73");
        client.afterPropertiesSet();

        // Object o = client.get("routerConfInfo_5cd0c7b2d04811e3a7a800163e0017df");
        Object o = client.get("cacheKeyRouterSeq_X12014031201000595");
        if (o != null) {
            System.out.println(o.getClass());
        } else {
            System.out.println("o is null");
        }

        System.out.println(client.get("1test_3"));
        System.out.println(client.get("1test_4"));
        System.out.println(client.get("1test_5"));
        System.out.println(client.get("1test_6"));

        String mapKey = "map12123";
        // Map<Integer, String> map = new HashMap<Integer, String>();
        // map.set(1, "fuck");
        // client.set(mapKey, map, 60, TimeUnit.MINUTES);
        System.out.println(client.get(mapKey));
        // System.out.println(client.get(mapKey).getClass());

        System.out.println("-------------------");
        String key = "appleRepeatRequest_123211114abcda12345121abcdfe122f";
        long count = client.getLong(key);
        System.out.println(count);
        client.incr(key, 1, 30, TimeUnit.SECONDS);
        Thread.sleep(2000);
        client.incr(key, 1, 30, TimeUnit.SECONDS);
        Object re = client.get(key);
        System.out.println("counter:" + re + "," + re.getClass());
        // for (int i = 0; i < 10000; i++) {
        // client.incr(key, 1, 30, TimeUnit.MINUTES);
        // System.out.println(client.get(key));
        // }
        System.exit(0);
    }

    @Override
    public void flushAll() {
        try {
            memcachedClient.flushAll();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
    }

    public void setProtocol(String protocol) {
        this.protocol = protocol;
    }

    public void setEnableHeartBeat(boolean enableHeartBeat) {
        this.enableHeartBeat = enableHeartBeat;
    }
}
