package cn.com.duiba.wolf.perf.timeprofile;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import java.util.*;

public class DBTimeProfile {

    private static final Logger                               log       = LoggerFactory.getLogger(DBTimeProfile.class);
    private static final ThreadLocal<Stack<TimeProfileStack>> stack     = new ThreadLocal<Stack<TimeProfileStack>>();
    private static final ThreadLocal<List<TimeProfileStack>>  logs      = new ThreadLocal<List<TimeProfileStack>>();
    private static final ThreadLocal<HttpServletRequest>              request   = new ThreadLocal<HttpServletRequest>();
    private static int                                  threshold = 300;                                         // 超时时间，单位毫秒

    public static void enter() {
        enter("");
    }

    public static void enter(String tag) {
        if (stack.get() == null) {
            return;
        }
        TimeProfileStack t = new TimeProfileStack();
        t.setEnterTime(new Date());
        t.setTag(tag);
        t.setDeep(stack.get().size());

        t.index = logs.get().size() + stack.get().size();
        stack.get().push(t);
    }

    public static void release() {
        if (stack.get() == null) {
            return;
        }
        TimeProfileStack t = stack.get().pop();
        t.setReleaseTime(new Date());
        logs.get().add(t);
    }

    public static void start() {
        stack.set(new Stack<TimeProfileStack>());
        logs.set(new ArrayList<TimeProfileStack>());
        enter();
    }

    public static void end() {
        Stack<TimeProfileStack> s1 = stack.get();
        if(s1 == null){
            return;
        }
        TimeProfileStack t = s1.pop();
        t.setReleaseTime(new Date());
        logs.get().add(t);

        Long timeconsume = t.getReleaseTime().getTime() - t.getEnterTime().getTime();
        if (timeconsume > threshold) {
            // 输出日志
            StringBuffer sb = new StringBuffer();
            if (request.get() != null) {
                String url = RequestTool.getRequestUrl(request.get());
                sb.append("DBTimeProfile timeout " + timeconsume + "ms >" + threshold + "ms, url=" + url);
            } else {
                sb.append("DBTimeProfile timeout " + timeconsume + "ms ");
            }
            List<TimeProfileStack> list = new ArrayList<TimeProfileStack>(logs.get());
            Collections.sort(list, new Comparator<TimeProfileStack>() {

                @Override
                public int compare(TimeProfileStack o1, TimeProfileStack o2) {
                    if (o1.index > o2.index) {
                        return 1;
                    } else if (o1.index < o2.index) {
                        return -1;
                    }
                    return 0;
                }
            });
            for (TimeProfileStack s : list) {
                sb.append("\r\n\t");
                for (int i = 0; i < s.getDeep(); i++) {
                    sb.append("-");
                }
                Long consume = s.getReleaseTime().getTime() - s.getEnterTime().getTime();
                sb.append(consume * 100 / timeconsume + "%");
                sb.append("  " + consume + "ms");
                if (s.getTag() != null) {
                    sb.append("  " + s.getTag());
                }

                if (s.getStackTraceElement() != null) {
                    StackTraceElement ste = s.getStackTraceElement();
                    sb.append("  " + ste.getClassName() + "." + ste.getMethodName() + " line:" + ste.getLineNumber());
                }
            }
            log.error(sb.toString());
        }
        request.set(null);
        stack.set(null);
        logs.set(null);
    }

    public static void setThreshold(int threshold) {
        DBTimeProfile.threshold = threshold;
    }

    /**
     * dump堆栈信息
     * 
     * @param ms 毫秒 ，超过此毫秒时才会dump
     */
    public static String dump(long ms) {
        if (logs.get() == null) {
            return null;
        }
        TimeProfileStack t = stack.get().pop();
        t.setReleaseTime(new Date());
        logs.get().add(t);

        Long timeconsume = 0L;
        for (TimeProfileStack s : logs.get()) {
            if (s.deep == 0) {
                timeconsume += s.getReleaseTime().getTime() - s.getEnterTime().getTime();
            }
        }
        if (timeconsume < ms) {
            return null;
        }
        // 输出日志
        StringBuffer sb = new StringBuffer();
        sb.append("DBTimeProfile timeout " + timeconsume + "ms ");
        List<TimeProfileStack> list = new ArrayList<TimeProfileStack>(logs.get());
        Collections.sort(list, new Comparator<TimeProfileStack>() {

            @Override
            public int compare(TimeProfileStack o1, TimeProfileStack o2) {
                if (o1.index > o2.index) {
                    return 1;
                } else if (o1.index < o2.index) {
                    return -1;
                }
                return 0;
            }
        });
        for (TimeProfileStack s : list) {
            Long consume = s.getReleaseTime().getTime() - s.getEnterTime().getTime();
            if (consume == 0) {
                continue;
            }
            sb.append("\r\n\t");
            for (int i = 0; i < s.getDeep(); i++) {
                sb.append("  ");
            }
            sb.append("-");
            sb.append(consume * 100 / timeconsume + "%");
            sb.append("  " + consume + "ms");
            if (s.getTag() != null) {
                sb.append("  " + s.getTag());
            }

            if (s.getStackTraceElement() != null) {
                StackTraceElement ste = s.getStackTraceElement();
                sb.append("  " + ste.getClassName() + "." + ste.getMethodName() + " line:" + ste.getLineNumber());
            }
        }
        request.set(null);
        stack.set(null);
        logs.set(null);

        return sb.toString();
    }

    public static void main(String[] args) throws Exception {
        DBTimeProfile.start();
        try {
            Thread.sleep(1000);
            DBTimeProfile.enter("in");
            Thread.sleep(1000);
            DBTimeProfile.release();
        } finally {
            DBTimeProfile.release();
        }

        System.out.println(DBTimeProfile.dump(0));
    }

    public static class TimeProfileStack {

        private Date              enterTime;
        private Date              releaseTime;
        private int               deep;
        private String            tag;
        int                       index;
        private StackTraceElement stackTraceElement;

        public Date getEnterTime() {
            return enterTime;
        }

        public void setEnterTime(Date enterTime) {
            this.enterTime = enterTime;
        }

        public Date getReleaseTime() {
            return releaseTime;
        }

        public void setReleaseTime(Date releaseTime) {
            this.releaseTime = releaseTime;
        }

        public int getDeep() {
            return deep;
        }

        public void setDeep(int deep) {
            this.deep = deep;
        }

        public String getTag() {
            return tag;
        }

        public void setTag(String tag) {
            this.tag = tag;
        }

        public StackTraceElement getStackTraceElement() {
            return stackTraceElement;
        }

        public void setStackTraceElement(StackTraceElement stackTraceElement) {
            this.stackTraceElement = stackTraceElement;
        }
    }
}
