package cn.com.duiba.zhongyan.activity.service.api.utils.question;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.biz.tool.utils.Conditions;
import cn.com.duiba.zhongyan.activity.service.api.enums.CommonStateEnum;
import cn.com.duiba.zhongyan.activity.service.api.enums.QuestionTypeEnum;
import cn.com.duiba.zhongyan.activity.service.api.param.AnswerParam;
import cn.com.duiba.zhongyan.activity.service.api.param.OptionParam;
import cn.com.duiba.zhongyan.activity.service.api.param.QuestionAnswerParam;
import cn.hutool.extra.emoji.EmojiUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 多选题校验处理
 *
 * @author jiangyesheng
 * @version 1.0
 * @date 2022/5/9
 */
public class MultipleChoiceHandler extends AbstractQuestionAnswerCheckHandler {

    @Override
    public QuestionTypeEnum getType() {
        return QuestionTypeEnum.MULTIPLE_CHOICE;
    }

    @Override
    void specificQuestionCheck(QuestionAnswerParam param) throws BizException {
        commonChoiceQuestionCheck(param, 10);
        checkMinAndMaxSelectNumConfig(param);
    }

    @Override
    String specificAnswerCheck(QuestionAnswerParam param, AnswerCheckContext context) throws BizException {
        Integer questionId = param.getQuestionId();
        checkOptionParam(param.getOption(), questionId,10);
        checkMinAndMaxSelectNumAnswer(param, context);
        Map<String, OptionParam> optionMap = param.getOption().stream().collect(Collectors.toMap(OptionParam::getOptionId, Function.identity()));

        StringBuilder userInput = new StringBuilder();
        for (AnswerParam answer : param.getAnswerList()) {
            Conditions.expectTrue(StringUtils.isNotBlank(answer.getAnswer()), "题目["+ questionId +"]:答案错误-答案不能为空");
            OptionParam option = optionMap.get(answer.getAnswer());
            Conditions.expectNotNull(option, "题目["+ questionId +"]:答案错误-答案不在选项中");
            // 校验必填框
            if (CommonStateEnum.STATE_ZERO.getState().equals(option.getAllowWrite())
                    && CommonStateEnum.STATE_ZERO.getState().equals(option.getTipStatus())) {
                Conditions.expectTrue(StringUtils.isNotBlank(answer.getRequiredAnswer()), "题目["+ questionId +"]:答案错误-填空框不能为空");
            }
            if (StringUtils.isNotBlank(answer.getRequiredAnswer())) {
                Conditions.expectTrue(EmojiUtil.removeAllEmojis(answer.getRequiredAnswer()).length() <= 50, "题目["+ questionId +"]:答案错误-填空框不能超过50个字符");
                userInput.append(answer.getRequiredAnswer());
            }
        }
        return userInput.toString();
    }

    /**
     * 检查最少最多选择个数限制
     * @param param
     */
    private void checkMinAndMaxSelectNumConfig(QuestionAnswerParam param) throws BizException {
       if (Objects.nonNull(param.getMinSelectNum())){
           Conditions.expectTrue(param.getMinSelectNum() > 0, "题目["+ param.getQuestionId() +"]:配置错误-最少选择个数不能小于0");
       }
       if (Objects.nonNull(param.getMaxSelectNum())){
           Conditions.expectTrue(param.getMaxSelectNum() <= param.getOption().size(), "题目["+ param.getQuestionId() +"]:配置错误-最多选择个数不能大于选项个数");
       }
       if (Objects.nonNull(param.getMinSelectNum()) && Objects.nonNull(param.getMaxSelectNum())){
           Conditions.expectTrue(param.getMinSelectNum() <= param.getMaxSelectNum(), "题目["+ param.getQuestionId() +"]:配置错误-最少选择个数不能大于最多选择个数");
       }
    }


    private void checkMinAndMaxSelectNumAnswer(QuestionAnswerParam param, AnswerCheckContext context) throws BizException {
        int answerSize = CollectionUtils.isEmpty(param.getAnswerList()) ? 0 : param.getAnswerList().size();
        if (Objects.nonNull(param.getMinSelectNum())){
            Conditions.expectTrue(answerSize >= param.getMinSelectNum(), String.format("第%s题必选%s到%s个选项！", context.getQuestionNo(),param.getMinSelectNum(),param.getMaxSelectNum()));
        }
        if (Objects.nonNull(param.getMaxSelectNum())){
            Conditions.expectTrue(answerSize <= param.getMaxSelectNum(),  String.format("第%s题必选%s到%s个选项！", context.getQuestionNo(),param.getMinSelectNum(),param.getMaxSelectNum()));
        }
    }

}
