package cn.com.duiba.user.server.api.request;


import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.user.server.api.dto.consumer.ConsumerCookieDTO;
import cn.com.duiba.user.server.api.dto.consumer.ConsumerDTO;
import cn.com.duiba.user.server.api.dto.consumer.TimeBasedRollingKeyDTO;
import cn.com.duiba.wolf.perf.timeprofile.RequestTool;
import cn.com.duiba.wolf.utils.NumberUtils;
import cn.com.duiba.wolf.utils.SecurityUtils;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.nio.charset.Charset;


/**
 * 兑吧用户客户端，用于从request cookie获取用户信息或者注入用户信息到cookie中
 *
 * @author zouweixiang
 * @date 2021/11/24
 */
@Slf4j
public class DuibaConsumerCookieClient {

	protected static final String CONSUMER_WDATA4_COOKIE = "zy_wdata4";
	protected static final String LOGIN_TIME_COOKIE = "zy_w_ts";

	private final KmsClient kmsClient;

	public DuibaConsumerCookieClient(KmsClient kmsClient) {
		this.kmsClient = kmsClient;
	}

	/**
	 * 从request cookie中获取用户信息，未登录或超过24小时则返回null
	 *
	 * @param request
	 * @return
	 */
	public ConsumerCookieDTO getConsumerCookieDto(HttpServletRequest request) throws BizException {
		String wdata4 = RequestTool.getCookie(request, CONSUMER_WDATA4_COOKIE);
		long ts = NumberUtils.parseLong(RequestTool.getCookie(request, LOGIN_TIME_COOKIE), -1L);
		return getConsumerCookieDto(wdata4,ts);
	}

	public ConsumerCookieDTO getConsumerCookieDto(String wdata4,long ts) throws BizException{
		if (wdata4 == null || wdata4.isEmpty()) {
			return null;
		}
//		long ts = NumberUtils.parseLong(RequestTool.getCookie(request, LOGIN_TIME_COOKIE), -1L);
		if (ts == -1L || !isValidTime(ts)) {
			return null;
		}
		TimeBasedRollingKeyDTO tbrk = getTimeBasedRollingKey(ts);
		log.info("登陆拦截器:tbrk:{},ts:{}",JSON.toJSONString(tbrk),ts);
		String userJson;//如果ts是伪造的，则解密时可能会抛出异常，需要处理
		try {
			userJson = this.decrypt(wdata4, tbrk);
		}catch (Exception e){
			log.warn("登录拦截，cookie解析异常, wdata4:{}, ts:{}", wdata4, ts, e);
			throw new BizException("登录异常，请重新登录");
		}
		ConsumerCookieDTO c = JSON.parseObject(userJson, ConsumerCookieDTO.class);
		if (!isValidTime(c.getTime())) {
			return null;
		}
		return c;
	}

	/**
	 * 登录请求，用于把用户信息放入cookie(包括wdata4、w_ts两个cookie)
	 * @param tbConsumerDto
	 * @param response
	 * @return 设置到cookie中的用户信息
	 */
	public ConsumerCookieDTO injectConsumerInfoIntoCookie(ConsumerDTO tbConsumerDto, HttpServletResponse response) {
		long loginTime = System.currentTimeMillis();
		ConsumerCookieDTO cookieDto = makeConsumerCookieDto(tbConsumerDto, loginTime);

		String userJson = JSON.toJSONString(cookieDto);

		String userCookieVal;

		TimeBasedRollingKeyDTO tbrk = getTimeBasedRollingKey(loginTime);
		userCookieVal = this.encrypt(userJson, tbrk);
		log.info("登陆拦截器:tbrk:{},ts:{}",JSON.toJSONString(tbrk),loginTime);
		Cookie userCookie = new Cookie(CONSUMER_WDATA4_COOKIE, userCookieVal);
		//防止被js获得
		userCookie.setHttpOnly(true);
		userCookie.setPath("/");
		response.addCookie(userCookie);

		String loginTimeStr = String.valueOf(loginTime);
		Cookie tsCookie = new Cookie(LOGIN_TIME_COOKIE, loginTimeStr);
		//防止被js获得
		tsCookie.setHttpOnly(true);
		tsCookie.setPath("/");
		response.addCookie(tsCookie);

		return cookieDto;
	}

	/**
	 * 登录时间在24小时内为有效
	 *
	 * @param loginTime
	 * @return
	 */
	private boolean isValidTime(long loginTime) {
		long now = System.currentTimeMillis();
		return loginTime > now - 86400000 && loginTime < now + 300000;
	}

	private TimeBasedRollingKeyDTO getTimeBasedRollingKey(long ts) {
		return kmsClient.getCachedTimeBasedRollingKey(ts);
	}

	/**
	 * 加密数据
	 *
	 * @param srcData 原始数据
	 * @param keyDto  密钥对象
	 * @return
	 */
	public String encrypt(String srcData, TimeBasedRollingKeyDTO keyDto) {
		if (keyDto == null) {
			throw new NullPointerException("keyDto must not be null");
		}
		return SecurityUtils.encode2StringByBase64(SecurityUtils.encodeByAes(srcData, keyDto.getSecretKey()));
	}

	/**
	 * 解密数据
	 *
	 * @param ciphertext 密文
	 * @param keyDto     密钥对象
	 * @return
	 */
	public String decrypt(String ciphertext, TimeBasedRollingKeyDTO keyDto) {
		if (keyDto == null) {
			throw new NullPointerException("keyDto must not be null");
		}
		return new String(SecurityUtils.decodeByAes(SecurityUtils.decodeBase64(ciphertext), keyDto.getSecretKey()), Charset.forName("UTF-8"));
	}

	private ConsumerCookieDTO makeConsumerCookieDto(ConsumerDTO tbConsumerDto, long loginTime) {
		if (tbConsumerDto.getId() == null) {
			throw new IllegalArgumentException("consumer's id must not be null");
		}
		if (tbConsumerDto.getPartnerUserId() == null) {
			throw new IllegalArgumentException("consumer's partnerUserId must not be null");
		}
		ConsumerCookieDTO cookieDto = new ConsumerCookieDTO();
		cookieDto.setCid(tbConsumerDto.getId());
		cookieDto.setPartnerUserId(tbConsumerDto.getPartnerUserId());
		cookieDto.setTime(loginTime);
		cookieDto.setUnionId(tbConsumerDto.getUnionId());
		cookieDto.setType(tbConsumerDto.getUserType());
		return cookieDto;
	}
}
