package cn.com.wawa.proxy.api.client;

import cn.com.wawa.proxy.api.bean.ConnectionConfig;
import cn.com.wawa.proxy.api.constant.Constants;
import cn.com.wawa.proxy.api.enums.ProtocolActionEnum;
import cn.com.wawa.proxy.api.event.impl.ClientPushEvent;
import cn.com.wawa.proxy.api.protocol.KeepAliveProtocol;
import com.alibaba.fastjson.JSONObject;
import org.apache.mina.core.future.ConnectFuture;
import org.apache.mina.core.service.IoHandlerAdapter;
import org.apache.mina.core.session.IoSession;
import org.apache.mina.filter.codec.ProtocolCodecFilter;
import org.apache.mina.filter.codec.serialization.ObjectSerializationCodecFactory;
import org.apache.mina.filter.logging.LoggingFilter;
import org.apache.mina.transport.socket.nio.NioSocketConnector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetSocketAddress;

/**
 * Created by danke on 2017/11/15.
 */
public abstract class ConnectionClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionClient.class);

    private static final String OBJECT_FILTER = "objectFilter";
    private static final String HEART_BEAT = "heartbeat";
    private static final String LOGGERSTR = "logger";

    //链接配置
    private ConnectionConfig connectionConfig;
    //链接会话
    private IoSession ioSession;
    //客户端链接
    private NioSocketConnector connector;
    //改客户端标识
    private String useId;

    public void connect(String ip,String useId){//初始化
        this.connectionConfig = new ConnectionConfig.Builder(Constants.MainConstants.port).setIp(ip).build();
        this.useId = useId;
        connect();
    }

    private void connect() {
        connector = new NioSocketConnector();
        connector.setHandler(new IoHandlerAdapter() {
            @Override
            public void messageReceived(IoSession session, Object message) throws Exception {
                super.messageReceived(session, message);
                KeepAliveProtocol protocol = JSONObject.parseObject(JSONObject.toJSONString(message), KeepAliveProtocol.class);
                consumersReceived(session,protocol);
            }

            /**
             * 客户端正常关闭向服务端发送关闭消息
             * @param session
             * @throws Exception
             */
            @Override
            public void sessionClosed(IoSession session) throws Exception {
                //向服务端发送关闭指定session的消息
                KeepAliveProtocol protocol = new KeepAliveProtocol();
                protocol.setAction(ProtocolActionEnum.CLOSE_SESSION.getCode());
                protocol.setSessionKey(useId);
                ioSession.write(protocol);
            }

            /**
             * 异常关闭向服务器发送关闭事件
             * @param session
             * @param cause
             * @throws Exception
             */
            @Override
            public void exceptionCaught(IoSession session, Throwable cause){
                if (LOGGER.isWarnEnabled()) {
                    LOGGER.warn("EXCEPTION, please implement " + getClass().getName()
                            + ".exceptionCaught() for proper handling:", cause);
                }
                try{
                    throw new Exception(cause);
                }catch (Exception e){
                    LOGGER.error("服务端强制关闭了链接");
                }
                //向服务端发送关闭session消息
                KeepAliveProtocol protocol = new KeepAliveProtocol();
                protocol.setAction(ProtocolActionEnum.CLOSE_SESSION.getCode());
                protocol.setSessionKey(useId);
                ioSession.write(protocol);
            }
        });
        connector.getFilterChain().addLast(LOGGERSTR, new LoggingFilter());
        connector.getFilterChain().addLast(OBJECT_FILTER, new ProtocolCodecFilter(new ObjectSerializationCodecFactory()));
        ConnectFuture future = null;
        try {
            future = connector.connect(new InetSocketAddress(connectionConfig.getIp(), connectionConfig.getPort()));
            future.awaitUninterruptibly();
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (future != null) {
            //第一次链接时自报家门,现在是伪实现,之后应该在推送中标注
            this.ioSession = future.getSession();
            //TODO 这边应该将客户端详细信息推送给服务端,以便服务端将连接详细信息分发
            KeepAliveProtocol protocol = new KeepAliveProtocol();
            protocol.setAction(ProtocolActionEnum.CREATE_SESSION.getCode());
            protocol.setSessionKey(useId);
            ioSession.write(protocol);
        }
    }


    public void close(){
        if (null == connector)
            return;
        if (ioSession != null){
            ioSession.getCloseFuture().setClosed();
            ioSession.getCloseFuture().awaitUninterruptibly();
        }
        connector.dispose();
    }




    /**
     * 改方法接收服务端推送信息,具体业务需自己实现
     * 参数已经约定序列化完成
     * @param session
     * @param protocol
     */
    public abstract void consumersReceived(IoSession session, KeepAliveProtocol protocol);

    public ConnectionConfig getConnectionConfig() {
        return connectionConfig;
    }

    public void setConnectionConfig(ConnectionConfig connectionConfig) {
        this.connectionConfig = connectionConfig;
    }

    public IoSession getIoSession() {
        return ioSession;
    }

    public void setIoSession(IoSession ioSession) {
        this.ioSession = ioSession;
    }

    public String getUseId() {
        return useId;
    }

    public void setUseId(String useId) {
        this.useId = useId;
    }

    public NioSocketConnector getConnector() {
        return connector;
    }

}
