package cn.com.wawa.proxy.api.client;

import cn.com.wawa.proxy.api.bean.ConnectionConfig;
import cn.com.wawa.proxy.api.constant.Constants;
import cn.com.wawa.proxy.api.enums.ProtocolActionEnum;
import cn.com.wawa.proxy.api.event.impl.ClientPushEvent;
import cn.com.wawa.proxy.api.protocol.KeepAliveProtocol;
import cn.com.wawa.proxy.api.util.KeyUtil;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Preconditions;
import org.apache.mina.core.buffer.IoBuffer;
import org.apache.mina.core.future.ConnectFuture;
import org.apache.mina.core.service.IoHandlerAdapter;
import org.apache.mina.core.session.IoSession;
import org.apache.mina.filter.codec.ProtocolCodecFilter;
import org.apache.mina.filter.codec.serialization.ObjectSerializationCodecFactory;
import org.apache.mina.filter.logging.LoggingFilter;
import org.apache.mina.transport.socket.nio.NioSocketConnector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.UnsupportedEncodingException;
import java.net.InetSocketAddress;
import java.util.List;

/**
 * Created by danke on 2017/11/15.
 */
public abstract class ConnectionClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionClient.class);

    private static final String OBJECT_FILTER = "objectFilter";
    private static final String HEART_BEAT = "heartbeat";
    private static final String LOGGERSTR = "logger";

    //链接配置
    private ConnectionConfig connectionConfig;
    //链接会话
    private IoSession ioSession;
    //客户端链接
    private NioSocketConnector connector;
    //该客户端标识
    private String clientKey;
    //该客户端能够被业务服务器识别的id
    private Long clientId;

    public void connect(String ip,Long clientId){//初始化
        this.connectionConfig = new ConnectionConfig.Builder(Constants.MainConstants.port).setIp(ip).setIdleTime(30).build();
        this.clientKey = KeyUtil.createClientKey(clientId);
        this.clientId = clientId;
        connect();
    }

    private void connect() {
        connector = new NioSocketConnector();
        connector.setHandler(new IoHandlerAdapter() {
            @Override
            public void messageReceived(IoSession session, Object message) throws Exception {
                super.messageReceived(session, message);
                KeepAliveProtocol protocol = JSONObject.parseObject(JSONObject.toJSONString(message), KeepAliveProtocol.class);
                consumersReceived(session,protocol);
            }

            /**
             * 客户端正常关闭向服务端发送关闭消息
             * @param session
             * @throws Exception
             */
            @Override
            public void sessionClosed(IoSession session) throws Exception {
                LOGGER.error("服务端正常关闭了链接");
            }

            /**
             * 异常关闭向服务器发送关闭事件
             * @param session
             * @param cause
             * @throws Exception
             */
            @Override
            public void exceptionCaught(IoSession session, Throwable cause){

            }
        });
        connector.getFilterChain().addLast(LOGGERSTR, new LoggingFilter());
        connector.getFilterChain().addLast(OBJECT_FILTER, new ProtocolCodecFilter(new ObjectSerializationCodecFactory()));
        ConnectFuture future = null;
        try {
            future = connector.connect(new InetSocketAddress(connectionConfig.getIp(), connectionConfig.getPort()));
            future.awaitUninterruptibly();
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (future != null) {
            //第一次链接时自报家门
            this.ioSession = future.getSession();
            KeepAliveProtocol protocol = new KeepAliveProtocol();
            protocol.setAction(ProtocolActionEnum.CREATE_SESSION.getCode());
            protocol.setClientId(clientId);
            protocol.setClientKey(clientKey);
            String jsonStrr = JSONObject.toJSONString(protocol);
            byte[] eventByte = null;
            try {
                eventByte = jsonStrr.getBytes("UTF-8");
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
            IoBuffer ioBuffer = IoBuffer.wrap(eventByte);
            ioSession.write(ioBuffer);
        }

    }

    /**
     * 会话推送消息
     * @param protocol
     */
    public void write(KeepAliveProtocol protocol){
        Preconditions.checkNotNull(ioSession,"服务器已经关闭会话连接");
        synchronized (ioSession){
            String jsonStrr = JSONObject.toJSONString(protocol);
            byte[] eventByte = null;
            try {
                eventByte = jsonStrr.getBytes("UTF-8");
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
            IoBuffer ioBuffer = IoBuffer.wrap(eventByte);
            ioSession.write(ioBuffer);
        }
    }

    /**
     * 用于创建会话协议或者关闭会话
     * @param protocolActionEnum
     * @return
     */
    public KeepAliveProtocol createProtocol(ProtocolActionEnum protocolActionEnum){
        return createProtocol(protocolActionEnum,null,null);
    }

    /**
     * 组建协议
     * 协议类型:客户端发送的消息只会是请求类型
     * 动作:暂时只有事件推送,可使用默认值
     * 推送的clientKey
     * @return
     */
    public KeepAliveProtocol createProtocol(ProtocolActionEnum protocolActionEnum,List<String> pushClientKeys,ClientPushEvent clientPushEvent){
        KeepAliveProtocol keepAliveProtocol = new KeepAliveProtocol();
        keepAliveProtocol.setClientKey(this.clientKey);
        keepAliveProtocol.setId(System.currentTimeMillis());
        keepAliveProtocol.setClientPushEvent(clientPushEvent);
        keepAliveProtocol.setAction(protocolActionEnum.getCode());
        keepAliveProtocol.setPushClientKeys(pushClientKeys);
        return keepAliveProtocol;
    }


    public void close(){
        if (null == connector)
            return;
        if (ioSession != null){
            //创建关闭连接消息
            KeepAliveProtocol closeProtocol = createProtocol(ProtocolActionEnum.CLOSE_SESSION);
            ioSession.write(closeProtocol);
            ioSession.getCloseFuture().setClosed();
            ioSession.getCloseFuture().awaitUninterruptibly();
        }
        connector.dispose();
    }

    /**
     * 该方法接收服务端推送信息,具体业务需自己实现
     * 参数已经约定序列化完成
     * @param session
     * @param protocol
     */
    public abstract void consumersReceived(IoSession session, KeepAliveProtocol protocol);

    public ConnectionConfig getConnectionConfig() {
        return connectionConfig;
    }

    public String getClientKey() {
        return clientKey;
    }

    public NioSocketConnector getConnector() {
        return connector;
    }

}
