package cn.com.wawa.proxy.api.client;

import cn.com.wawa.proxy.api.bean.ConnectionConfig;
import cn.com.wawa.proxy.api.constant.Constants;
import cn.com.wawa.proxy.api.enums.RequestCodeEnums;
import cn.com.wawa.proxy.api.event.impl.CreateSessionBody;
import cn.com.wawa.proxy.api.protocol.KeepAliveProtocol;
import cn.com.wawa.proxy.api.util.SecureTool;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Preconditions;
import org.apache.mina.core.future.ConnectFuture;
import org.apache.mina.core.service.IoHandlerAdapter;
import org.apache.mina.core.session.IoSession;
import org.apache.mina.filter.codec.ProtocolCodecFilter;
import org.apache.mina.filter.codec.serialization.ObjectSerializationCodecFactory;
import org.apache.mina.filter.logging.LoggingFilter;
import org.apache.mina.transport.socket.nio.NioSocketConnector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetSocketAddress;

/**
 * Created by danke on 2017/11/15.
 */
public abstract class ConnectionClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionClient.class);

    private static final String OBJECT_FILTER = "objectFilter";
    private static final String HEART_BEAT = "heartbeat";
    private static final String LOGGERSTR = "logger";

    //链接配置
    private ConnectionConfig connectionConfig;
    //链接会话
    private IoSession ioSession;
    //客户端链接
    private NioSocketConnector connector;
    //该客户端能够被业务服务器识别的id
    private Long clientId;

    public void connect(String ip,Long clientId,String token){//初始化
        this.connectionConfig = new ConnectionConfig.Builder(Constants.MainConstants.port).setIp(ip).setIdleTime(30).build();
        this.clientId = clientId;
        connect(token);
    }

    private void connect(String token) {
        connector = new NioSocketConnector();
        connector.setHandler(new IoHandlerAdapter() {
            @Override
            public void messageReceived(IoSession session, Object message) throws Exception {
                super.messageReceived(session, message);
                KeepAliveProtocol protocol = JSONObject.parseObject(JSONObject.toJSONString(message), KeepAliveProtocol.class);
                consumersReceived(session,protocol);
            }

            /**
             * 客户端正常关闭向服务端发送关闭消息
             * @param session
             * @throws Exception
             */
            @Override
            public void sessionClosed(IoSession session) throws Exception {
                LOGGER.error("服务端正常关闭了链接");
/*                Thread.sleep(5000);
                tryConnection();*/
            }

            /**
             * 异常关闭向服务器发送关闭事件
             * @param session
             * @param cause
             * @throws Exception
             */
            @Override
            public void exceptionCaught(IoSession session, Throwable cause){

            }
        });

        connector.getFilterChain().addLast(LOGGERSTR, new LoggingFilter());
        connector.getFilterChain().addLast(OBJECT_FILTER, new ProtocolCodecFilter(new ObjectSerializationCodecFactory()));
        tryConnection(token);

    }

    public void tryConnection(String token){
        ConnectFuture future = null;
        try {
            future = connector.connect(new InetSocketAddress(connectionConfig.getIp(), connectionConfig.getPort()));
            future.awaitUninterruptibly();
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (future != null) {
            //第一次链接时自报家门
            this.ioSession = future.getSession();
            KeepAliveProtocol protocol = new KeepAliveProtocol();
            protocol.setRequestCode(RequestCodeEnums.CREATE_SESSION.getCode());
            CreateSessionBody createSessionBody = new CreateSessionBody();
            createSessionBody.setToken(token);
            protocol.setBody(createSessionBody);
            ioSession.write(protocol);
        }
    }

    /**
     * 会话推送消息
     * @param protocol
     */
    public void write(KeepAliveProtocol protocol){
        Preconditions.checkNotNull(ioSession,"服务器已经关闭会话连接");
        synchronized (ioSession){
            ioSession.write(protocol);
/*            String jsonStrr = JSONObject.toJSONString(protocol);
            byte[] eventByte = null;
            try {
                eventByte = jsonStrr.getBytes("UTF-8");
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
            IoBuffer ioBuffer = IoBuffer.wrap(eventByte);
            ioSession.write(ioBuffer);*/
        }
    }

    /**
     * 组建协议
     * 协议类型:客户端发送的消息只会是请求类型
     * 动作:暂时只有事件推送,可使用默认值
     * 推送的clientKey
     * @return
     */
    public KeepAliveProtocol createProtocol(){
        KeepAliveProtocol keepAliveProtocol = new KeepAliveProtocol();
        return keepAliveProtocol;
    }


    public void close(){
        if (null == connector)
            return;
        if (ioSession != null){
            //创建关闭连接消息
        }
        connector.dispose();
    }

    /**
     * 该方法接收服务端推送信息,具体业务需自己实现
     * 参数已经约定序列化完成
     * @param session
     * @param protocol
     */
    public abstract void consumersReceived(IoSession session, KeepAliveProtocol protocol);

    public ConnectionConfig getConnectionConfig() {
        return connectionConfig;
    }

    public NioSocketConnector getConnector() {
        return connector;
    }

}
