package cn.com.wawa.proxy.api.client;

import cn.com.wawa.proxy.api.bean.ConnectionConfig;
import cn.com.wawa.proxy.api.code.HCodeFactory;
import cn.com.wawa.proxy.api.constant.Constants;
import cn.com.wawa.proxy.api.enums.RequestCodeEnums;
import cn.com.wawa.proxy.api.protocol.KeepAliveProtocolHead;
import cn.com.wawa.proxy.api.protocol.KeyPair;
import com.alibaba.fastjson.JSONObject;
import org.apache.mina.core.future.ConnectFuture;
import org.apache.mina.core.service.IoHandlerAdapter;
import org.apache.mina.core.service.IoService;
import org.apache.mina.core.service.IoServiceListener;
import org.apache.mina.core.session.IdleStatus;
import org.apache.mina.core.session.IoSession;
import org.apache.mina.filter.codec.ProtocolCodecFilter;
import org.apache.mina.filter.logging.LoggingFilter;
import org.apache.mina.transport.socket.nio.NioSocketConnector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetSocketAddress;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Created by danke on 2017/11/15.
 * 长连接客户端
 * PS:需要自己实现接受逻辑
 * 需要实现获取ip和token逻辑,以便于断线重连
 * 可以重写关闭逻辑doSessionClosed用于自己实现关闭连接时的业务逻辑
 */
public abstract class ConnectionClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionClient.class);

    private static final String LOGGERSTR = "logger";

    private static final int DEFAULT_TIMER_LATE = 10*1000;//单位ms
    private static final int DEFAULT_TIMER_TIMING = 15*1000;//单位ms

    //链接配置
    private ConnectionConfig connectionConfig;
    //链接会话
    private IoSession ioSession;
    //客户端链接
    private NioSocketConnector connector;
    //该客户端能够被业务服务器识别的id
    private String userId;
    //存一份act,用于断线重连
    private RequestCodeEnums act;
    //心跳发送延时间隔
    private int timerLate;
    //心跳发送间隔
    private int timerTimming;
    //心跳包
    private Timer timer;
    //标记是否需要断线重连,主动关闭时请设置为false
    private AtomicBoolean flag = new AtomicBoolean(true);

    public void connect(String userId,RequestCodeEnums act){
        connect(userId,act,DEFAULT_TIMER_LATE,DEFAULT_TIMER_TIMING);
    }

    /**
     *
     * @param userId 用户id
     * @param act 注册类型    CREATE_USER(1,"创建用户端连接"),CREATE_MACHINE(2,"创建机器端连接"),CREATE_SERVICE(3,"创建业务服务器连接"),
     */
    public void connect(String userId,RequestCodeEnums act,Integer timerLate,Integer timerTimming){//初始化
        KeyPair<String,String> ipAndToken = getIpAndToken();
        this.connectionConfig = new ConnectionConfig.Builder(Constants.MainConstants.port).setIp(ipAndToken.getFirstOne()).setIdleTime(30).build();
        this.timerLate = null == timerLate ? DEFAULT_TIMER_LATE : timerLate;
        this.timerTimming = null == timerTimming ? DEFAULT_TIMER_TIMING : timerTimming;
        this.userId = userId;
        this.act = act;
        JSONObject protocol = new JSONObject();
        KeepAliveProtocolHead protocolHead = new KeepAliveProtocolHead();
        protocolHead.setAct(act.getCode());
        protocolHead.setFrom(userId);
        protocolHead.setSign(ipAndToken.getSecondOne());
        protocol.put("head",protocolHead);
        connect(protocol.toJSONString());
        timer = new Timer();
        timer.scheduleAtFixedRate(new TimerTask() {
            @Override
            public void run() {
                JSONObject protocol = new JSONObject();
                JSONObject protocolHead = new JSONObject();
                protocolHead.put("act",0);
                protocol.put("head",protocolHead);
                write(protocol.toJSONString());
                if (LOGGER.isDebugEnabled())
                    LOGGER.debug("heart beat send msg={}",protocol);
            }
        },timerLate,timerTimming);
    }

    private void connect(String createProtocol) {
        connector = new NioSocketConnector();
        connector.setHandler(new IoHandlerAdapter() {
            @Override
            public void messageReceived(IoSession session, Object message) throws Exception {
                if (null == message)
                    return;
                super.messageReceived(session, message);
                String messageStr = String.valueOf(message);
                consumersReceived(session,messageStr);
            }

            /**
             * 客户端正常关闭向服务端发送关闭消息
             * @param session
             * @throws Exception
             */
            @Override
            public void sessionClosed(IoSession session) throws Exception {
                LOGGER.error("服务端正常关闭了链接");
                doSessionClosed(session);
            }
        });
        connector.addListener(new IoListener() {//添加断线监听器
            @Override
            public void sessionDestroyed(IoSession arg0) throws Exception {
                if(!flag.get())
                    return;
                String token;
                for (;;) {
                    try {
                        Thread.sleep(3000);
                        //获取ip和token
                        KeyPair<String,String> ipAndToken = getIpAndToken();
                        token = ipAndToken.getSecondOne();
                        ConnectFuture future = connector.connect(new InetSocketAddress(ipAndToken.getFirstOne(),Constants.MainConstants.port));
                        if (future != null){
                            future.awaitUninterruptibly();// 等待连接创建成功
                            ioSession = future.getSession();// 获取会话
                            break;
                        }
                    } catch (Exception e) {
                        LOGGER.info("重连服务器登录失败,3秒再连接一次:" + e.getMessage());
                    }
                }
                //发送注册信息;
                JSONObject protocol = new JSONObject();
                KeepAliveProtocolHead protocolHead = new KeepAliveProtocolHead();
                protocolHead.setAct(act.getCode());
                protocolHead.setFrom(userId);
                protocolHead.setSign(token);
                protocol.put("head",protocolHead);
                ioSession.write(protocol.toJSONString());
            }
        });
        connector.getFilterChain().addLast("codec", new ProtocolCodecFilter(new HCodeFactory()));
        tryConnection(createProtocol);

    }

    public void tryConnection(String createProtocol){
        ConnectFuture future = null;
        try {
            future = connector.connect(new InetSocketAddress(connectionConfig.getIp(), connectionConfig.getPort()));
            future.awaitUninterruptibly();
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (future != null) {
            //第一次链接时自报家门
            this.ioSession = future.getSession();
            ioSession.write(createProtocol);
        }
    }

    /**
     * 会话推送消息
     * @param protocol
     */
    public void write(String protocol){
        synchronized (ioSession){
            ioSession.write(protocol);
        }
    }

    /**
     * 平滑关闭客户端
     * @param needReTry 是否需要断线重连,true-需要,false-不需要
     */
    public void close(Boolean needReTry){
        if (null == connector)
            return;
        if (ioSession != null){
            ioSession.getCloseFuture().setClosed();
            //创建关闭连接消息
        }
        connector.dispose();
        flag.set(needReTry);
    }

    /**
     * 该方法接收服务端推送信息,具体业务需自己实现
     * 参数已经约定序列化完成
     * @param session
     * @param message
     */
    public abstract void consumersReceived(IoSession session, String message);

    /**
     * 关闭会话连接
     * 客户端要自己实现逻辑,请重写
     * @return
     */
    public void doSessionClosed(IoSession session){
        // Empty handler
    }

    /**
     * 获取ip和token的方法,必须客户端实现
     * 这边需要注意,返回值自己校验,不允许为空,为空直接抛出异常,客户端不做参数校验
     * @return
     */
    public abstract KeyPair<String,String> getIpAndToken();

    public ConnectionConfig getConnectionConfig() {
        return connectionConfig;
    }

    public NioSocketConnector getConnector() {
        return connector;
    }

    public String getUserId() {
        return userId;
    }

    public int getTimerLate() {
        return timerLate;
    }

    public int getTimerTimming() {
        return timerTimming;
    }

    private static class IoListener implements IoServiceListener {
        @Override
        public void serviceActivated(IoService arg0) throws Exception {
            // TODO Auto-generated method stub
        }
        @Override
        public void serviceDeactivated(IoService arg0) throws Exception {
            // TODO Auto-generated method stub
        }
        @Override
        public void serviceIdle(IoService arg0, IdleStatus arg1) throws Exception {
            // TODO Auto-generated method stub
        }
        @Override
        public void sessionCreated(IoSession arg0) throws Exception {
            // TODO Auto-generated method stub
        }

        @Override
        public void sessionDestroyed(IoSession arg0) throws Exception {
            // TODO Auto-generated method stub
        }

    }
}
