package cn.com.wawa.proxy.api.client;

import cn.com.wawa.proxy.api.bean.ProxyDto;
import cn.com.wawa.proxy.api.code.HCodeFactory;
import cn.com.wawa.proxy.api.constant.Constants;
import cn.com.wawa.proxy.api.enums.RequestCodeEnums;
import cn.com.wawa.proxy.api.protocol.KeepAliveProtocolHead;
import com.alibaba.fastjson.JSONObject;
import org.apache.mina.core.future.ConnectFuture;
import org.apache.mina.core.service.IoHandlerAdapter;
import org.apache.mina.core.service.IoService;
import org.apache.mina.core.service.IoServiceListener;
import org.apache.mina.core.session.IdleStatus;
import org.apache.mina.core.session.IoSession;
import org.apache.mina.filter.codec.ProtocolCodecFilter;
import org.apache.mina.transport.socket.nio.NioSocketConnector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 长连接客户端
 */
public abstract class MinaClient {

    private static final Logger logger = LoggerFactory.getLogger(MinaClient.class);

    private NioSocketConnector connector;

    private IoSession ioSession;

    private String clientId;

    private RequestCodeEnums act;

    private static ScheduledThreadPoolExecutor scheduled = new ScheduledThreadPoolExecutor(1);

    public void connect(String clientId, RequestCodeEnums act) {
        this.clientId = clientId;
        this.act = act;
        connector = new NioSocketConnector();
        connector.setConnectTimeoutMillis(5000);
        connector.getSessionConfig().setIdleTime(IdleStatus.READER_IDLE, 60);
        connector.getFilterChain().addLast("codec", new ProtocolCodecFilter(new HCodeFactory()));
        connector.setHandler(new ClientIoHandlerAdapter());
        connector.addListener(new ClientIoListener() {
            @Override
            public void sessionDestroyed(IoSession session) throws Exception {
                retryConnect();
            }
        });
        startKeepAlive();
    }

    public void write(String message) {
        if (ioSession == null || !ioSession.isConnected() || ioSession.isClosing()) {
            throw new RuntimeException("ioSession is close");
        }
        ioSession.write(message);
    }

    private void startKeepAlive() {
        scheduled.scheduleWithFixedDelay(new Runnable() {
            @Override
            public void run() {
                try {
                    Thread.currentThread().setName("keepAlive");
                    if (ioSession == null || !ioSession.isConnected() || ioSession.isClosing()) {
                        retryConnect();
                    }
                    JSONObject protocol = new JSONObject();
                    JSONObject protocolHead = new JSONObject();
                    protocolHead.put("act", RequestCodeEnums.HEART_CHECK.getCode());
                    protocol.put("head", protocolHead);
                    write(protocol.toJSONString());
                } catch (Exception e) {
                    logger.error("scheduleWithFixedDelay", e);
                }
            }
        }, 0, 30, TimeUnit.SECONDS);
    }

    private synchronized void retryConnect() {
        while (true) {
            try {
                // 如果连接可用不进行重连
                if (ioSession != null && ioSession.isConnected()) {
                    return;
                }
                // 获取代理地址信息
                ProxyDto proxy = getIpAndToken();
                // 连接到代理服务器
                logger.info("tryConnect server ip:" + proxy.getHost() + " port:" + proxy.getPort() + " token:" + proxy.getToken());

                ConnectFuture future = connector.connect(new InetSocketAddress(proxy.getHost(), proxy.getPort()));
                future.awaitUninterruptibly();
                ioSession = future.getSession();
                if (ioSession.isConnected()) {
                    JSONObject protocol = new JSONObject();
                    KeepAliveProtocolHead protocolHead = new KeepAliveProtocolHead();
                    protocolHead.setAct(act.getCode());
                    protocolHead.setFrom(clientId);
                    protocolHead.setSign(proxy.getToken());
                    protocol.put("head", protocolHead);
                    ioSession.write(protocol.toJSONString());
                    logger.info("connect success");
                    return;
                }
            } catch (Exception e) {
                logger.error("connect error:" + e.getMessage());
            }
            try {
                Thread.sleep(3000);
            } catch (InterruptedException e) {
                logger.error("Thread.sleep", e);
            }
        }
    }

    protected abstract void msgReceived(String message);


    protected abstract ProxyDto getIpAndToken();


    private class ClientIoHandlerAdapter extends IoHandlerAdapter {
        @Override
        public void sessionClosed(IoSession session) throws Exception {
            ioSession.close(true);
        }

        @Override
        public void sessionIdle(IoSession session, IdleStatus status) throws Exception {
            if (session == IdleStatus.READER_IDLE) {
                ioSession.close(true);
            }
        }

        @Override
        public void exceptionCaught(IoSession session, Throwable cause) throws Exception {
            if (cause instanceof IOException) {
                ioSession.close(true);
            }
        }

        @Override
        public void messageReceived(IoSession session, Object message) throws Exception {
            String msg = message.toString();
            try {
                JSONObject jsonObject = JSONObject.parseObject(msg);
                String headStr = jsonObject.getString(Constants.ProtocolConstant.PROTOCPL_HEAD_KEY);
                KeepAliveProtocolHead head = JSONObject.parseObject(headStr, KeepAliveProtocolHead.class);
                //忽略心跳消息
                if (head.getAct() == RequestCodeEnums.HEART_CHECK.getCode()) {
                    return;
                }
            } catch (Exception e) {
                logger.error("messageReceived", e);
            }
            msgReceived(msg);
        }
    }

    private class ClientIoListener implements IoServiceListener {
        @Override
        public void serviceActivated(IoService service) throws Exception {

        }

        @Override
        public void serviceIdle(IoService service, IdleStatus idleStatus) throws Exception {

        }

        @Override
        public void serviceDeactivated(IoService service) throws Exception {

        }

        @Override
        public void sessionCreated(IoSession session) throws Exception {

        }

        @Override
        public void sessionDestroyed(IoSession session) throws Exception {

        }
    }
}
