package cn.lili.common.exception;

import cn.hutool.core.text.CharSequenceUtil;
import cn.lili.common.enums.ResultCode;
import cn.lili.common.enums.ResultUtil;
import cn.lili.common.utils.StringUtils;
import cn.lili.common.vo.ResultMessage;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 全局异常异常处理
 *
 * @author Chopper
 */
@RestControllerAdvice
@Slf4j
public class GlobalControllerExceptionHandler {

    /**
     * 如果超过长度，则前后段交互体验不佳，使用默认错误消息
     */
    static Integer MAX_LENGTH = 200;

    /**
     * 自定义异常
     *
     * @param e
     * @return
     */
    @ExceptionHandler(ServiceException.class)
    @ResponseStatus(value = HttpStatus.BAD_REQUEST)
    public ResultMessage<Object> handleServiceException(HttpServletRequest request, final Exception e, HttpServletResponse response) {


        //如果是自定义异常，则获取异常，返回自定义错误消息
        if (e instanceof ServiceException) {
            ServiceException serviceException = ((ServiceException) e);
            ResultCode resultCode = serviceException.getResultCode();

            Integer code = null;
            String message = null;

            if (resultCode != null) {
                code = resultCode.code();
                message = resultCode.message();
            }
            //如果有扩展消息，则输出异常中，跟随补充异常
            if (message != null && !serviceException.getMsg().equals(ServiceException.DEFAULT_MESSAGE)) {
                message = appendErrorMessage(message, serviceException.getMsg());
            }
            // 对一些特殊异常处理，不再打印error级别的日志
            assert serviceException.getResultCode() != null;
            if (serviceException.getResultCode().equals(ResultCode.DEMO_SITE_EXCEPTION)) {
                log.debug("[DEMO_SITE_EXCEPTION]:{}", serviceException.getResultCode().message(), e);
                return ResultUtil.error(code, message);
            }
            if (serviceException.getResultCode().equals(ResultCode.USER_AUTH_EXPIRED)) {
                log.debug("403 :{}", serviceException.getResultCode().message(), e);
                return ResultUtil.error(code, message);
            }
            log.error("全局异常[ServiceException]:errorCode:{}, errorMessage:{}", serviceException.getResultCode().code(), serviceException.getResultCode().message(), e);
            return ResultUtil.error(code, message);

        } else {
            log.error("全局异常[Exception]:", e);
        }
        //默认错误消息
        String errorMsg = "服务器异常，请稍后重试";
        if (e != null && e.getMessage() != null && e.getMessage().length() < MAX_LENGTH) {
            errorMsg = e.getMessage();
        }
        return ResultUtil.error(ResultCode.ERROR.code(), errorMsg);
    }

    @ExceptionHandler(RuntimeException.class)
    @ResponseStatus(value = HttpStatus.BAD_REQUEST)
    public ResultMessage<Object> runtimeExceptionHandler(HttpServletRequest request, final Exception e, HttpServletResponse response) {
        log.error("全局异常[RuntimeException]:", e);
        return ResultUtil.error(ResultCode.ERROR);
    }

    @ExceptionHandler(value = MethodArgumentNotValidException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResultMessage<Object> handleValidException(MethodArgumentNotValidException exception) {
        BindingResult result = exception.getBindingResult();
        String errorMessage= StringUtils.EMPTY;
        if (result.hasErrors()) {
            errorMessage= result.getFieldErrors().stream().map(FieldError::getDefaultMessage).collect(Collectors.joining(","));
            log.error("handleValidException errorMessage:{}:",errorMessage);
        }
        return ResultUtil.error(ResultCode.PARAMS_ERROR.code(), errorMessage);
    }


    @ExceptionHandler(BindException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResultMessage<Object> handleBindException(BindException e) {
        StringBuilder message = new StringBuilder();
        List<FieldError> fieldErrors = e.getBindingResult().getFieldErrors();
        for (FieldError error : fieldErrors) {
            message.append(error.getField()).append(error.getDefaultMessage()).append(",");
        }
        message = new StringBuilder(message.substring(0, message.length() - 1));
        log.error("handleBindException errorMessage:{}", message.toString());
        return ResultUtil.error(ResultCode.PARAMS_ERROR.code(), message.toString());
    }

    /**
     * 拼接错误消息
     *
     * @param message       原始消息
     * @param appendMessage 需要拼接的消息
     * @return 拼接后的消息
     */
    private String appendErrorMessage(String message, String appendMessage) {

        //这里的代码看起来有点乱，简单解释一下
        //场景1：服务A，服务B=》
        // 服务A调用服务B=》
        // 服务B抛出异常{扩展消息}，拼接后成为{默认消息}：{扩展消息}
        // 异常被服务A捕获=》
        // 最终消息拼接过程中，当前方法体参数message是{默认消息}，参数appendMessage是服务A给的{默认消息}+{扩展消息}，最终会形成{默认消息}+{默认消息}+{扩展消息}
        //场景2：只有服务A=》
        // 服务A抛出异常{扩展消息}=》
        // 当前方法体拼接{默认消息}：{扩展消息} 并输出返回。
        //
        //总的来说，由于消息拼接是一个流式传递，由服务间传递，所以这里的消息可能存在A包含B，也可能出现B包含A，
        // 所以这里需要双重判定，A包含B=》返回A，B包含A=》返回B，否则返回拼接后的消息

        if (message.contains(appendMessage)) {
            return message;
        }
        if (appendMessage.contains(message)) {
            return appendMessage;
        }
        return CharSequenceUtil.format("{}:{}", message, appendMessage);
    }
}
