package cn.lili.common.utils;

import org.springframework.context.ApplicationContext;
import org.springframework.core.env.*;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * @Author yuquanfeng
 * @Date 2023/5/26
 * @Description
 * @Param
 * @return
 **/
public class SpringEnvironmentUtils {
    /**
     * 开发环境标识
     */
    private static final String DEV = "dev";
    /**
     * 测试环境
     */
    private static final String TEST = "test";
    /**
     * 生产环境
     */
    private static final String RELEASE = "release";


    private SpringEnvironmentUtils(){}


    /**
     * 从spring的Environment获得扁平化的所有key value对
     * @param environment
     * @return
     */
    public static LinkedHashMap<String, Object> getFlatEnvironments(Environment environment) {
        final LinkedHashMap<String, Object> properties = new LinkedHashMap<>();
        for (Iterator it = ((AbstractEnvironment) environment).getPropertySources().iterator(); it.hasNext(); ) {
            PropertySource propertySource = (PropertySource) it.next();
            handlePropertySource(propertySource, properties);
        }
        return properties;
    }

    private static void handlePropertySource(PropertySource propertySource, LinkedHashMap<String, Object> properties){
        if (propertySource instanceof EnumerablePropertySource) {
            for (String key : ((EnumerablePropertySource) propertySource).getPropertyNames()) {
                properties.putIfAbsent(key, propertySource.getProperty(key));
            }
        }
        if (propertySource instanceof PropertiesPropertySource) {
            for (Map.Entry<String, Object> e : ((MapPropertySource) propertySource).getSource().entrySet()) {
                properties.putIfAbsent(e.getKey(), e.getValue());
            }
        }
        if (propertySource instanceof CompositePropertySource) {
            for (Map.Entry<String, Object> e : getPropertiesInCompositePropertySource((CompositePropertySource) propertySource).entrySet()) {
                properties.putIfAbsent(e.getKey(), e.getValue());
            }
        }
    }

    private static LinkedHashMap<String, Object> getPropertiesInCompositePropertySource(CompositePropertySource compositePropertySource) {
        final LinkedHashMap<String, Object> properties = new LinkedHashMap<>();
        compositePropertySource.getPropertySources().forEach(propertySource -> {
            if (propertySource instanceof EnumerablePropertySource) {
                for (String key : ((EnumerablePropertySource) propertySource).getPropertyNames()) {
                    properties.putIfAbsent(key, propertySource.getProperty(key));
                }
            }
            if (propertySource instanceof MapPropertySource) {
                for (Map.Entry<String, Object> e : ((MapPropertySource) propertySource).getSource().entrySet()) {
                    properties.putIfAbsent(e.getKey(), e.getValue());
                }
            }
            if (propertySource instanceof CompositePropertySource) {
                for (Map.Entry<String, Object> e : getPropertiesInCompositePropertySource((CompositePropertySource) propertySource).entrySet()) {
                    properties.putIfAbsent(e.getKey(), e.getValue());
                }
            }
        });
        return properties;
    }


    private static String[] getActiveProfiles(){
        if(MainApplicationContextHolder.getApplicationContext() == null){
            throw new IllegalStateException("应用还没有启动，你过早地调用了判断环境的方法");
        }
        return MainApplicationContextHolder.getApplicationContext().getEnvironment().getActiveProfiles();
    }



    /**
     * 是否在开发环境
     * @return
     */
    public static boolean isDevEnv(){
        for(String p : getActiveProfiles()){
            if(DEV.equals(p)){
                return true;
            }
        }
        return false;
    }


    /**
     * 是否在测试环境
     * @return
     */
    public static boolean isTestEnv(){
        for(String p : getActiveProfiles()){
            if(TEST.equals(p)){
                return true;
            }
        }
        return false;
    }


    /**
     * 是否在线上环境
     * @return
     */
    public static boolean isReleaseEnv(){
        String[] activeProfiles = getActiveProfiles();
        for(String p : activeProfiles){
            if(p != null && p.startsWith(RELEASE)){
                return true;
            }
        }
        return false;
    }

    /**
     * 获取当前所在的环境
     * @return
     */
    public static String getCurrentEnv() {
        if(SpringEnvironmentUtils.isDevEnv()) {
            return SpringEnvironmentUtils.DEV;
        }

        if(SpringEnvironmentUtils.isTestEnv()) {
            return SpringEnvironmentUtils.TEST;
        }

        if(SpringEnvironmentUtils.isReleaseEnv()) {
            return SpringEnvironmentUtils.RELEASE;
        }

        throw new IllegalStateException("没有获取系统环境");
    }
}
