package cn.tuia.mango.core.util;

import lombok.AllArgsConstructor;
import lombok.Getter;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * @author lijicong
 * @since 2020-04-11
 */
public class DateFormatKit {

    // 时间格式
    public static final String DATE_PATTERN = "yyyy-MM-dd";
    public static final String DATETIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public static final String DATETIME_MS_PATTERN = "yyyy-MM-dd HH:mm:ss:SSS";
    public static final String TIME_PATTERN = "HH:mm:ss";

    // 时间序号格式
    public static final String DATE_NO_PATTERN = "yyyyMMdd";
    public static final String DATETIME_NO_PATTERN = "yyyyMMddHHmmss";
    public static final String DATETIME_MS_NO_PATTERN = "yyyyMMddHHmmssSSS";
    public static final String TIME_NO_PATTERN = "HHmmss";
    public static final String TIME_MS_NO_PATTERN = "HHmmssSSS";

    // 年月日
    public static final String YEAR_PATTERN = "yyyy";
    public static final String MONTH_PATTERN = "MM";
    public static final String DAY_PATTERN = "dd";
    public static final String YEAR_MONTH_PATTERN = "yyyyMM";

    @AllArgsConstructor
    @Getter
    public enum DateFormat {
        // 时间格式
        DATE(DATE_PATTERN),
        DATETIME(DATETIME_PATTERN),
        DATETIME_MS(DATETIME_MS_PATTERN),
        TIME(TIME_PATTERN),

        // 时间序号格式
        DATE_NO(DATE_NO_PATTERN),
        DATETIME_NO(DATETIME_NO_PATTERN),
        DATETIME_MS_NO(DATETIME_MS_NO_PATTERN),
        TIME_NO(TIME_NO_PATTERN),
        TIME_MS_NO(TIME_MS_NO_PATTERN),

        // 年月日
        YEAR(YEAR_PATTERN),
        MONTH(MONTH_PATTERN),
        DAY(DAY_PATTERN),
        YEAR_MONTH(YEAR_MONTH_PATTERN),
        ;

        private String pattern;

        public String format(Date date) {
            return map.get(getPattern()).get().format(date);
        }

        public Integer formatInt(Date date) {
            return Integer.valueOf(format(date));
        }

        public Date parse(String source) {
            try {
                return map.get(getPattern()).get().parse(source);
            } catch (ParseException e) {
            }
            return null;
        }

        public Date parseOrReturnNow(String source) {
            Date date = parse(source);
            return date == null ? new Date() : date;
        }

        public static final Map<String, ThreadLocal<SimpleDateFormat>> map = new HashMap<>(13);

        static {
            for (DateFormat format : DateFormat.values()) {
                ThreadLocal<SimpleDateFormat> threadLocal = ThreadLocal.withInitial(() -> new SimpleDateFormat(format.getPattern()));
                map.put(format.getPattern(), threadLocal);
            }
        }
    }

    public static String formatDate(Date date) {
        return DateFormat.DATE.format(date);
    }

    public static String formatDateTime(Date date) {
        return DateFormat.DATETIME.format(date);
    }

    public static String formatDateTimeMillisNo(Date date) {
        return DateFormat.DATETIME_MS_NO.format(date);
    }

}
