package cn.tuia.mango.core.util;

import cn.tuia.mango.core.enums.CodeDescriptionEnumerable;
import org.springframework.util.Assert;

import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * 枚举工具类
 * @author lijicong
 * @since 2020-04-06
 */
public abstract class EnumKit {

    /** 缓存枚举类型以及枚举的值 */
    private static Map<Class<?>, Map<?, ?>> enumConstantMap = new ConcurrentHashMap<>();

    /**
     * 用enumCode构造一个Enum
     * @param enumClass 枚举Class对象
     * @param enumCode 枚举code
     * @param <K> 枚举Key类型
     * @param <V> 枚举value类型
     * @param <E> 枚举类
     * @return
     */
    public static <K, V, E extends Enum<E> & CodeDescriptionEnumerable<K, V>> Optional<E> of(Class<E> enumClass, K enumCode) {
        Assert.notNull(enumClass, "enumClass must not be null");
        Map<?, ?> enumMap = enumConstantMap.get(enumClass);
        if (enumMap == null) {
            Map<K, E> map = Arrays.stream(enumClass.getEnumConstants()).collect(Collectors.toMap(e -> e.getCode(), e -> e));
            enumConstantMap.put(enumClass, map);
            enumMap = map;
        }
        return Optional.ofNullable((E) enumMap.get(enumCode));
    }

    /**
     * 用enumCode构造一个Enum，for循环遍历查找，效率较低
     * @param enumClass 枚举Class对象
     * @param enumCode 枚举code
     * @param <K> 枚举Key类型
     * @param <V> 枚举value类型
     * @param <E> 枚举类
     * @return
     */
    public static <K, V, E extends Enum<E> & CodeDescriptionEnumerable<K, V>> Optional<E> find(Class<E> enumClass, K enumCode) {
        Assert.notNull(enumClass, "enumClass must not be null");
        return Arrays.stream(enumClass.getEnumConstants()).filter(e -> e.getCode().equals(enumCode)).findAny();
    }

}
