/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.mapreduce;

import com.aliyun.odps.Column;
import com.aliyun.odps.TableSchema;
import com.aliyun.odps.tunnel.TableTunnel;
import com.aliyun.odps.tunnel.TunnelException;
import com.aliyun.odps.type.TypeInfo;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.MapWritable;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.util.ToolRunner;
import org.apache.phoenix.mapreduce.util.PhoenixMapReduceUtil;
import org.apache.phoenix.schema.types.*;
import org.apache.phoenix.util.ColumnInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.*;
import java.sql.SQLException;
import java.sql.Types;
import java.util.List;

import static org.apache.phoenix.mapreduce.ODPSMapper.ODPS_ERROR_DATA_PATH;

public class ODPSBulkLoadTool extends AbstractBulkLoadTool {
    protected static final Logger LOG = LoggerFactory.getLogger(ODPSBulkLoadTool.class);

    static final String ODSP_IGNORED_DATA_DIRECTORY = "/tmp/odps_ignored_data";
    static final Option ACCESS_ID_OPT = new Option("access_id", true, "your access id");
    static final Option ACCESS_KEY_OPT = new Option("access_key", true, "your access key");
    static final Option ODPS_URL_OPT = new Option("odps_url", true, "ODPS URL");
    static final Option TUNNEL_OPT = new Option("odps_tunnel_url", true, "tunnel URL");
    static final Option PROJECT_OPT = new Option("odps_project", true, "project name");
    static final Option TABLE_OPT = new Option("odps_table", true, "table name");
    static final Option PARTITION_SPEC_OPT = new Option("odps_partition_spec", true, "your partition specific(Optional)");
    static final Option PARTITION_NUMBER_OPT = new Option("odps_partition_number", true, "split number((Optional))");
    static final Option ODPS_COLUMNS_OPT = new Option("odps_column_names", true, "Comma-separated list of columns(Optional)");
    static final Option ERROR_DATA_PATH_OPT = new Option("error_data_path", true, "hdfs path(Optional), default path: hdfs:///tmp/odps_ignored_data");

    @Override
    protected void configureOptions(CommandLine cmdLine, List<ColumnInfo> importColumns,
            Configuration conf) throws SQLException {
        List<String> sourceColumnNames = null;
        if (cmdLine.hasOption(ODPS_COLUMNS_OPT.getOpt())) {
            sourceColumnNames = Lists.newArrayList(
                    Splitter.on(",").trimResults().split
                            (cmdLine.getOptionValue(ODPS_COLUMNS_OPT.getOpt())));
            conf.set(ODPSMapper.ODPS_COLUMN_INFO_CONFKEY, Joiner.on("|").useForNull("").join(sourceColumnNames));
            List<ColumnInfo> mappedColumns = importColumns.subList(0, sourceColumnNames.size());
            LOG.info("Columns " + mappedColumns.toString() + " of Phoenix table are mapped!");
            conf.set(ODPSMapper.PHOENIX_MAPPED_COLUMN_INFO_CONFKEY, Joiner.on("|").useForNull("").join(mappedColumns));
        }

        List<String> targetColumnNames = null;
        if (cmdLine.hasOption(IMPORT_COLUMNS_OPT.getOpt())) {
            targetColumnNames = Lists.newArrayList(
                    Splitter.on(",").trimResults().split
                            (cmdLine.getOptionValue(IMPORT_COLUMNS_OPT.getOpt())));
            if (sourceColumnNames != null && sourceColumnNames.size() !=  targetColumnNames.size()) {
                throw new IllegalStateException("ODPS table columns size is not equal Phoenix table columns size!");
            }
        }

        if (sourceColumnNames != null && sourceColumnNames.size() > targetColumnNames.size()) {
            throw new IllegalStateException("ODPS table columns size should not bigger than Phoenix table columns size!");
        }

        conf.set(ODPSMapper.ACCESS_KEY_ID_CONFKEY, cmdLine.getOptionValue(ACCESS_ID_OPT.getOpt()));
        conf.set(ODPSMapper.ACCESS_KEY_SECRET_CONFKEY, cmdLine.getOptionValue(ACCESS_KEY_OPT.getOpt()));
        conf.set(ODPSMapper.ODPS_URL_CONFKEY, cmdLine.getOptionValue(ODPS_URL_OPT.getOpt()));
        conf.set(ODPSMapper.ODPS_TUNNEL_URL_CONFKEY, cmdLine.getOptionValue(TUNNEL_OPT.getOpt()));
        conf.set(ODPSMapper.ODPS_PROJECT_CONFKEY, cmdLine.getOptionValue(PROJECT_OPT.getOpt()));
        conf.set(ODPSMapper.ODPS_TABLE_NAME_CONFKEY, cmdLine.getOptionValue(TABLE_OPT.getOpt()));

        String path;
        if (cmdLine.hasOption(ERROR_DATA_PATH_OPT.getOpt())) {
            path = cmdLine.getOptionValue(ERROR_DATA_PATH_OPT.getOpt());
        } else {
            path = ODSP_IGNORED_DATA_DIRECTORY;
        }
        conf.set(ODPS_ERROR_DATA_PATH, path);

        if (cmdLine.hasOption(PARTITION_SPEC_OPT.getOpt())) {
            conf.set(ODPSMapper.ODPS_TABLE_PARTITION_SPEC_CONFKEY, cmdLine.getOptionValue(PARTITION_SPEC_OPT.getOpt()));
        }
        if (cmdLine.hasOption(PARTITION_NUMBER_OPT.getOpt())) {
            conf.setInt(ODPSMapper.ODPS_PARTITION_NUMBER_CONFKEY, Integer.valueOf(cmdLine.getOptionValue(PARTITION_NUMBER_OPT.getOpt())));
        }

        // check columns of ODPS table
        try {
            TableTunnel.DownloadSession downloadSession = PhoenixMapReduceUtil.getDownloadSession(conf);
            TableSchema sourceSchema = downloadSession.getSchema();
            if (sourceColumnNames != null) {
                int i = 0;
                for (String sc : sourceColumnNames) {
                    try {
                        Column column = sourceSchema.getColumn(sc);
                        isCastableTo(column, importColumns.get(i));
                    } catch (IllegalArgumentException e) {
                        LOG.error("Column:" + sc + " does not exists in table:" + cmdLine.getOptionValue(TABLE_OPT.getOpt()));
                        throw e;
                    }
                    i++;
                }
            } else {
                if (sourceSchema.getColumns().size() > importColumns.size()) {
                    List<String> defaultSourceColumns = Lists.transform(sourceSchema.getColumns(),
                                    new Function<Column, String>() {
                                        @Nullable
                                        @Override
                                        public String apply(@Nullable Column input) {
                                            return input.getTypeInfo().getTypeName() + ":" + input.getName();
                                        }
                                    });
                    LOG.info("All columns of ODPS table: " + defaultSourceColumns);
                    LOG.info("All columns of Phoenix table: " + importColumns);
                    throw new IllegalStateException("ODPS table columns size should not bigger than Phoenix table columns size!");
                } else {
                    conf.set(ODPSMapper.ODPS_COLUMN_INFO_CONFKEY, Joiner.on("|").useForNull("")
                            .join(Lists.transform(sourceSchema.getColumns(),
                                    new Function<Column, String>() {
                                @Nullable
                                @Override
                                public String apply(@Nullable Column input) {
                                    return input.getName();
                                }
                            })));
                    List<ColumnInfo> mappedColumns = importColumns.subList(0, sourceSchema.getColumns().size());
                    LOG.info("Columns " + mappedColumns.toString() + " of Phoenix table are mapped!");
                    conf.set(ODPSMapper.PHOENIX_MAPPED_COLUMN_INFO_CONFKEY, Joiner.on("|").useForNull("").join(mappedColumns));
                }
            }
        } catch (TunnelException e) {
            throw new IllegalStateException(e.getCause());
        }

        conf.setClass(ODPSMapper.ODPS_INPUT_CLASS, MapWritable.class, Writable.class);

        try {
            FileSystem fs = FileSystem.get(conf);
            Path dirPath = new Path(conf.get(ODPS_ERROR_DATA_PATH));
            LOG.info("Ignored data directory :" + dirPath.toString() + " will be recreated!");
            fs.delete(dirPath, true);
            fs.mkdirs(dirPath);

        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void isCastableTo(Column sourceColumn, ColumnInfo targetColumn) {
        TypeInfo typeInfo = sourceColumn.getTypeInfo();
        if (targetColumn.getPDataType().getSqlType() == Types.VARCHAR ||
                targetColumn.getPDataType().getSqlType() == Types.CHAR) {
            return;
        }
        boolean isCastable;
        switch (typeInfo.getOdpsType()) {
            case BIGINT:
                isCastable = PLong.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case DOUBLE:
                isCastable = PDouble.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case BOOLEAN:
                isCastable = PBoolean.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case STRING:
                isCastable = PVarchar.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case DECIMAL:
                isCastable = PDecimal.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case TINYINT:
                isCastable = PTinyint.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case SMALLINT:
                isCastable = PSmallint.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case INT:
                isCastable = PInteger.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case FLOAT:
                isCastable = PFloat.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case CHAR:
                isCastable = PChar.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case VARCHAR:
                isCastable = PChar.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case DATETIME:
                isCastable = PTime.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case DATE:
                isCastable = PDate.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            case TIMESTAMP:
                isCastable = PTimestamp.INSTANCE.isCastableTo(targetColumn.getPDataType());
                break;
            default:
                throw new IllegalStateException("UnSupported data type: " + typeInfo.getTypeName() + " as source data type");
        }
        if (!isCastable) {
            throw new IllegalStateException("Unsupported data type: " + sourceColumn.getTypeInfo().getTypeName() + " of ODPS table  " +
                    " cast to data type:" + targetColumn.getPDataType().getSqlTypeName() + " of Phoenix table");
        }

    }

    @Override
    protected
    void setupJob(Job job) {
        job.setInputFormatClass(ODPSInputFormat.class);
        job.setMapperClass(ODPSMapper.class);
        if (job.getJar() == null) {
            job.setJarByClass(ODPSMapper.class);
        }
    }

    @Override
    protected Options getOptions() {
        Options options = super.getOptions();
        ACCESS_ID_OPT.setRequired(true);
        options.addOption(ACCESS_ID_OPT);
        ACCESS_KEY_OPT.setRequired(true);
        options.addOption(ACCESS_KEY_OPT);
        ODPS_URL_OPT.setRequired(true);
        options.addOption(ODPS_URL_OPT);
        TUNNEL_OPT.setRequired(true);
        options.addOption(TUNNEL_OPT);
        PROJECT_OPT.setRequired(true);
        options.addOption(PROJECT_OPT);
        TABLE_OPT.setRequired(true);
        options.addOption(TABLE_OPT);
        options.addOption(PARTITION_SPEC_OPT);
        options.addOption(PARTITION_NUMBER_OPT);
        options.addOption(ERROR_DATA_PATH_OPT);
        options.addOption(ODPS_COLUMNS_OPT);
        return options;
    }

    public static void main(String[] args) throws Exception {
        int exitStatus = ToolRunner.run(new ODPSBulkLoadTool(), args);
        System.exit(exitStatus);
    }
}
