/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.mapreduce;

import com.aliyun.odps.tunnel.TableTunnel;
import com.aliyun.odps.tunnel.TunnelException;
import com.aliyun.odps.tunnel.io.TunnelRecordReader;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.io.MapWritable;
import org.apache.hadoop.io.NullWritable;
import org.apache.hadoop.mapreduce.*;
import org.apache.phoenix.mapreduce.util.PhoenixMapReduceUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import static org.apache.phoenix.mapreduce.util.PhoenixMapReduceUtil.getRanges;

public class ODPSInputFormat <T extends MapWritable> extends InputFormat<NullWritable,T>  {
    protected static final Logger LOG = LoggerFactory.getLogger(ODPSInputFormat.class);

    @Override
    public List<InputSplit> getSplits(JobContext context) throws IOException, InterruptedException {
        Configuration configuration = context.getConfiguration();
        int splitNumber = configuration.getInt(ODPSMapper.ODPS_PARTITION_NUMBER_CONFKEY, 1);
        try {
            TableTunnel.DownloadSession downloadSession = PhoenixMapReduceUtil.getDownloadSession(configuration);
            long downloadCount = downloadSession.getRecordCount();
            List<Pair<Long, Long>> ranges = getRanges(downloadCount, splitNumber);
            LOG.info("Total " + downloadCount + " records in source table, split number is " + ranges.size());
            ArrayList<InputSplit> splits = new ArrayList<>(ranges.size());
            int i = 0;
            for (Pair<Long, Long> range : ranges) {
                TunnelRecordReader reader = downloadSession.openRecordReader(range.getFirst(),
                                range.getSecond() - range.getFirst() + 1);
                long totalSize = reader.getTotalBytes();
                splits.add(new ODPSInputSplit(i++, range.getFirst(), range.getSecond()- range.getFirst() + 1, totalSize));
                reader.close();
            }
            return splits;
        } catch (TunnelException e) {
            throw new IOException(e.getCause());
        }
    }

    @Override
    public RecordReader<NullWritable, T> createRecordReader(InputSplit split,
            TaskAttemptContext context) throws IOException, InterruptedException {
        final Configuration configuration = context.getConfiguration();
        @SuppressWarnings("unchecked")
        final Class<T> inputClass = (Class<T>)configuration.getClass(ODPSMapper.ODPS_INPUT_CLASS, NullWritable.class);
        return new ODPSRecordReader(inputClass, configuration);
    }
}
