/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.coprocessor;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.util.List;

import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CoprocessorEnvironment;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.coprocessor.BaseRegionObserver;
import org.apache.hadoop.hbase.coprocessor.ObserverContext;
import org.apache.hadoop.hbase.coprocessor.RegionCoprocessorEnvironment;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.io.WritableUtils;
import org.apache.phoenix.execute.TupleProjector;
import org.apache.phoenix.expression.Expression;
import org.apache.phoenix.expression.ExpressionType;
import org.apache.phoenix.schema.tuple.ResultTuple;
import org.apache.phoenix.schema.tuple.Tuple;

import static org.apache.phoenix.coprocessor.BaseScannerRegionObserver.FILTER_PB_BYTES;

public class GetRegionObserver extends BaseRegionObserver {
    private final ImmutableBytesWritable ptr = new ImmutableBytesWritable();

    public GetRegionObserver() {
        super();
    }

    @Override
    public void start(CoprocessorEnvironment e) throws IOException {
        super.start(e);
    }

    @Override
    public void stop(CoprocessorEnvironment e) throws IOException {
        super.stop(e);
    }

    @Override
    public void postGetOp(ObserverContext<RegionCoprocessorEnvironment> e, Get get,
            List<Cell> results) throws IOException {
        if (results.isEmpty()) {
            return;
        }

        ResultTuple rsTuple = new ResultTuple(Result.create(results));

        //filter rows by the where expression
        byte[] whereBytes = get.getAttribute(FILTER_PB_BYTES);
        if (whereBytes != null) {
            Expression expression = deserializeExpression(whereBytes);
            if (!expression.evaluate(rsTuple, ptr) ||
                    ptr.getLength() == 0 ||
                    !Boolean.TRUE.equals(expression.getDataType().toObject(ptr))) {
                results.clear();
                return;
            }
        }

        boolean useNewValueColumnQualifier =
                get.getAttribute(BaseScannerRegionObserver.USE_NEW_VALUE_COLUMN_QUALIFIER) == null ? false : true;
        TupleProjector projector = TupleProjector.deserializeProjectorFromGet(get);
        if (projector != null) {
            Tuple projectTuple = projector.projectResults(rsTuple, useNewValueColumnQualifier);
            results.clear();
            results.add(projectTuple.getValue(0));
        }
    }

    private Expression deserializeExpression(byte[] b) {
        ByteArrayInputStream stream = new ByteArrayInputStream(b);
        try {
            DataInputStream input = new DataInputStream(stream);
            ExpressionType type = ExpressionType.values()[WritableUtils.readVInt(input)];
            Expression whereExpression = type.newInstance();
            whereExpression.readFields(input);
            return whereExpression;
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            try {
                stream.close();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public String toString() {
        return this.getClass().getName();
    }
}
