/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.coprocessor;

import com.google.common.collect.Lists;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;

import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.HTableInterface;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.coprocessor.RegionCoprocessorEnvironment;
import org.apache.hadoop.hbase.filter.Filter;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.regionserver.RegionScanner;
import org.apache.hadoop.hbase.regionserver.ScannerContext;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.io.WritableUtils;
import org.apache.phoenix.coprocessor.generated.PhoenixFilterProtos;
import org.apache.phoenix.execute.TupleProjector;
import org.apache.phoenix.execute.TupleProjector.ProjectedValueTuple;
import org.apache.phoenix.expression.Expression;
import org.apache.phoenix.expression.ExpressionType;
import org.apache.phoenix.filter.FilterFactory;
import org.apache.phoenix.hbase.index.covered.update.ColumnReference;
import org.apache.phoenix.index.IndexMaintainer;
import org.apache.phoenix.schema.PTable;
import org.apache.phoenix.schema.tuple.EncodedColumnQualiferCellsList;
import org.apache.phoenix.schema.tuple.PositionBasedResultTuple;
import org.apache.phoenix.schema.tuple.ResultTuple;
import org.apache.phoenix.schema.tuple.Tuple;
import org.apache.phoenix.util.EncodedColumnsUtil;
import org.apache.phoenix.util.ServerUtil;

import static org.apache.phoenix.coprocessor.BaseScannerRegionObserver.FILTER_TYPE;
import static org.apache.phoenix.coprocessor.BaseScannerRegionObserver.OFFSET_ON_SERVER;
import static org.apache.phoenix.coprocessor.BaseScannerRegionObserver.FILTER_PB_BYTES;
import static org.apache.phoenix.execute.TupleProjector.SCAN_PROJECTOR;
import static org.apache.phoenix.query.QueryConstants.ENCODED_CQ_COUNTER_INITIAL_VALUE;
import static org.apache.phoenix.util.IndexUtil.deserializeDataTableColumns;

public class LookupJoinRegionScanner implements RegionScanner {
    private static final int MULTI_GET_BATCH_SIZE = 100;

    private final RegionScanner delegate;
    private final TupleProjector projector;
    private final RegionCoprocessorEnvironment env;
    private final int getBatchSize;
    private final boolean useQualifierAsListIndex;
    private final boolean useNewValueColumnQualifier;
    private final HTableInterface htable;
    private final ImmutableBytesWritable ptr;
    private long count;
    private long limit;
    private Queue<Cell> resultQueue;
    private IndexMaintainer indexMaintainer;
    private ColumnReference[] dataColumns;
    private Filter filter;
    private Pair<Integer, Integer> minMaxQualifiers;
    private PTable.QualifierEncodingScheme encodingScheme;
    private boolean useQualifierAsIndex;
    private boolean hasMore = false;

    @SuppressWarnings("unchecked")
    public LookupJoinRegionScanner(RegionScanner scanner, ColumnReference[] dataColumns, RegionCoprocessorEnvironment env,
            boolean useQualifierAsIndex, boolean useNewValueColumnQualifier, Scan scan, IndexMaintainer indexMaintainer) throws IOException {
        this.env = env;
        this.delegate = scanner;
        this.dataColumns = deserializeDataTableColumns(scan, BaseScannerRegionObserver.DATA_TABLE_COLUMNS_FOR_LOOKUP_JOIN);
        this.resultQueue = new LinkedList<>();
        this.count = 0;
        this.useQualifierAsListIndex = useQualifierAsIndex;
        this.useNewValueColumnQualifier = useNewValueColumnQualifier;
        this.getBatchSize = MULTI_GET_BATCH_SIZE;
        this.ptr = new ImmutableBytesWritable();
        this.indexMaintainer = indexMaintainer;
        this.htable = env.getTable(TableName.valueOf(scan.getAttribute(BaseScannerRegionObserver.SEMI_LEFT_TABLE_NAME)));
        this.projector = TupleProjector.deserializeProjectorFromScan(scan, BaseScannerRegionObserver.getSemiAttrName(SCAN_PROJECTOR));
        byte[] limitBytes = scan.getAttribute(OFFSET_ON_SERVER);
        if (limitBytes != null) {
            this.limit = Bytes.toInt(limitBytes);
        } else {
            this.limit = -1;
        }

        byte[] filterTypeBytes = scan.getAttribute(FILTER_TYPE);
        if (filterTypeBytes != null) {
            try {
                PhoenixFilterProtos.FilterType type = PhoenixFilterProtos.FilterType.valueOf(Bytes.toInt(filterTypeBytes));
                byte[] filterBytes = scan.getAttribute(FILTER_PB_BYTES);
                this.filter = FilterFactory.DeserFilter(type, filterBytes);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        this.minMaxQualifiers = getMinMaxQualifiersFromScan(scan);
        this.useQualifierAsIndex = EncodedColumnsUtil.useQualifierAsIndex(this.minMaxQualifiers);
        byte[] qualifierEncodingSchema = scan.getAttribute(BaseScannerRegionObserver.getSemiAttrName(BaseScannerRegionObserver.QUALIFIER_ENCODING_SCHEME));
        this.encodingScheme = qualifierEncodingSchema == null ?
                PTable.QualifierEncodingScheme.NON_ENCODED_QUALIFIERS : PTable.QualifierEncodingScheme.fromSerializedValue(qualifierEncodingSchema[0]);
    }

    private Pair<Integer, Integer> getMinMaxQualifiersFromScan(Scan scan) {
        Integer minQ = null, maxQ = null;
        byte[] minQualifier = scan.getAttribute(BaseScannerRegionObserver.getSemiAttrName(BaseScannerRegionObserver.MIN_QUALIFIER));
        if (minQualifier != null) {
            minQ = Bytes.toInt(minQualifier);
        }
        byte[] maxQualifier = scan.getAttribute(BaseScannerRegionObserver.getSemiAttrName(BaseScannerRegionObserver.MAX_QUALIFIER));
        if (maxQualifier != null) {
            maxQ = Bytes.toInt(maxQualifier);
        }
        if (minQualifier == null) {
            return null;
        }
        return new Pair<>(minQ, maxQ);
    }

    @Override
    public long getMvccReadPoint() {
        return delegate.getMvccReadPoint();
    }

    @Override
    public HRegionInfo getRegionInfo() {
        return delegate.getRegionInfo();
    }

    @Override
    public boolean isFilterDone() throws IOException {
        return delegate.isFilterDone() && resultQueue.isEmpty();
    }

    private byte[] getRowKey(List<Cell> result) {
        Tuple tuple = useQualifierAsListIndex ? new PositionBasedResultTuple(result) : new ResultTuple(Result.create(result));
        tuple.getKey(ptr);
        return indexMaintainer.buildDataRowKey(ptr, null);
    }

    @Override
    public boolean nextRaw(List<Cell> result) throws IOException {
        try {
            if (resultQueue.isEmpty()) {
                List<Get> gets = Lists.newArrayListWithExpectedSize(getBatchSize);
                int c = 0;
                do {
                    hasMore = delegate.nextRaw(result);
                    if (!result.isEmpty()) {
                        byte[] key = getRowKey(result);
                        Get get = new Get(key);

                        // for the width table.The dataColumns should contain all columns in the query.
                        if (dataColumns != null) {
                            PTable.ImmutableStorageScheme storageScheme = indexMaintainer.getIndexStorageScheme();
                            for (int i = 0; i < dataColumns.length; i++) {
                                if (storageScheme == PTable.ImmutableStorageScheme.SINGLE_CELL_ARRAY_WITH_OFFSETS) {
                                    get.addFamily(dataColumns[i].getFamily());
                                } else {
                                    get.addColumn(dataColumns[i].getFamily(), dataColumns[i].getQualifier());
                                }
                            }
                        }
                        if (filter != null) {
                            get.setFilter(filter);
                        }
                        gets.add(get);
                        result.clear();
                    }
                } while (hasMore && (++c % getBatchSize != 0));

                if (!gets.isEmpty()) {
                    for (Result r : htable.get(gets)) {
                        if (limit != -1 && count >= limit) {
                            hasMore = false;
                            break;
                        }
                        if (r.isEmpty()) {
                            continue;
                        }

                        List<Cell> rs = useQualifierAsIndex ?
                                new EncodedColumnQualiferCellsList(ENCODED_CQ_COUNTER_INITIAL_VALUE, minMaxQualifiers.getSecond(), encodingScheme) :  new ArrayList<Cell>();
                        rs.addAll(r.listCells());

                        Tuple resultTuple = useQualifierAsListIndex ? new PositionBasedResultTuple(rs) : new ResultTuple(Result.create(rs));
                        ProjectedValueTuple tuple = projector.projectResults(resultTuple);

                        resultQueue.offer(tuple.getValue(0));
                        count++;
                    }
                }
            }

            Cell cell = resultQueue.poll();
            result.clear();
            if (cell == null && hasMore == false) {
                resultQueue.clear();
                return false;
            }

            if (cell != null) {
                result.add(cell);
            }
            return true;
        } catch (Throwable t) {
            ServerUtil.throwIOException(env.getRegion().getRegionInfo().getRegionNameAsString(), t);
            return false; // impossible
        }
    }

    @Override
    public boolean nextRaw(List<Cell> result, ScannerContext scannerContext) throws IOException {
        throw new IOException("Next with scannerContext should not be called in Phoenix environment");
    }

    @Override
    public boolean reseek(byte[] row) throws IOException {
        return delegate.reseek(row);
    }

    @Override
    public void close() throws IOException {
        htable.close();
        delegate.close();
    }

    @Override
    public boolean next(List<Cell> result) throws IOException {
        throw new IOException("Next should not be used in SemiJoin scanner");
    }

    @Override
    public boolean next(List<Cell> result, ScannerContext scannerContext) throws IOException {
        throw new IOException("Next with scanner context should not be used in SemiJoin scanner");
    }

    @Override
    public long getMaxResultSize() {
        return delegate.getMaxResultSize();
    }

    @Override
    public int getBatch() {
        return delegate.getBatch();
    }

}

