/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.filter;

import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.filter.Filter;
import org.apache.hadoop.hbase.filter.FilterList;
import org.apache.phoenix.coprocessor.generated.PhoenixFilterProtos;

public class FilterFactory {

    public static Filter DeserFilter(PhoenixFilterProtos.FilterType type, byte[] pbBytes)
            throws DeserializationException {
       switch (type) {
           case SINGLE_CQ_KEY_VALUE_COMPARISON:
               return SingleCQKeyValueComparisonFilter.parseFrom(pbBytes);
           case SINGLE_CF_CQ_KEY_VALUE_COMPARISON:
               return SingleCFCQKeyValueComparisonFilter.parseFrom(pbBytes);
           case SINGLE_KEY_VALUE_COMPARISON:
               return SingleKeyValueComparisonFilter.parseFrom(pbBytes);
           case SKIP_SCAN:
               return SkipScanFilter.parseFrom(pbBytes);
           case ROWKEY_COMPARISON:
               return RowKeyComparisonFilter.parseFrom(pbBytes);
           case MULTI_KEY_VALUE_COMPARISON:
               return MultiKeyValueComparisonFilter.parseFrom(pbBytes);
           case MULTI_ENOCDE_CQ_KEY_VALUE_COMPARISON:
               return MultiEncodedCQKeyValueComparisonFilter.parseFrom(pbBytes);
           case MULTI_CF_CQ_KEY_VALUE_COMPARISON:
               return MultiCFCQKeyValueComparisonFilter.parseFrom(pbBytes);
           case MULTI_CQ_KEY_VALUE_COMPARISON:
               return MultiCQKeyValueComparisonFilter.parseFrom(pbBytes);
           case DISTINCT_PREFIX:
               return DistinctPrefixFilter.parseFrom(pbBytes);
           case COLUMN_PROJECT:
               return ColumnProjectionFilter.parseFrom(pbBytes);
           case ENCODE_QUALIFIER_COLUMN_PROJECT:
               return EncodedQualifiersColumnProjectionFilter.parseFrom(pbBytes);
           case FILTER_LIST:
                return FilterList.parseFrom(pbBytes);
           default:
                throw new IllegalArgumentException("Unsupported filter type");
       }
    }

    public static PhoenixFilterProtos.FilterType getFilterType(Filter filter) {
        if (filter instanceof SingleCQKeyValueComparisonFilter) {
            return PhoenixFilterProtos.FilterType.SINGLE_CQ_KEY_VALUE_COMPARISON;
        } else if (filter instanceof SingleCFCQKeyValueComparisonFilter) {
            return PhoenixFilterProtos.FilterType.SINGLE_CF_CQ_KEY_VALUE_COMPARISON;
        } else if (filter instanceof SingleKeyValueComparisonFilter) {
            return PhoenixFilterProtos.FilterType.SINGLE_KEY_VALUE_COMPARISON;
        } else if (filter instanceof SkipScanFilter) {
            return PhoenixFilterProtos.FilterType.SKIP_SCAN;
        } else if (filter instanceof RowKeyComparisonFilter) {
            return PhoenixFilterProtos.FilterType.ROWKEY_COMPARISON;
        } else if (filter instanceof MultiKeyValueComparisonFilter) {
            return PhoenixFilterProtos.FilterType.MULTI_KEY_VALUE_COMPARISON;
        } else if (filter instanceof MultiEncodedCQKeyValueComparisonFilter) {
            return PhoenixFilterProtos.FilterType.MULTI_ENOCDE_CQ_KEY_VALUE_COMPARISON;
        } else if (filter instanceof MultiCFCQKeyValueComparisonFilter) {
            return PhoenixFilterProtos.FilterType.MULTI_CF_CQ_KEY_VALUE_COMPARISON;
        } else if (filter instanceof MultiCQKeyValueComparisonFilter) {
            return PhoenixFilterProtos.FilterType.MULTI_CQ_KEY_VALUE_COMPARISON;
        } else if (filter instanceof DistinctPrefixFilter) {
            return PhoenixFilterProtos.FilterType.DISTINCT_PREFIX;
        } else if (filter instanceof ColumnProjectionFilter) {
            return PhoenixFilterProtos.FilterType.COLUMN_PROJECT;
        } else if (filter instanceof FilterList) {
            return PhoenixFilterProtos.FilterType.FILTER_LIST;
        } else {
            return PhoenixFilterProtos.FilterType.ENCODE_QUALIFIER_COLUMN_PROJECT;
        }
    }
}
