/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.hbase.index.table;

import java.io.IOException;
import java.util.concurrent.ExecutorService;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.CoprocessorEnvironment;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.ConnectionFactory;
import org.apache.hadoop.hbase.client.CoprocessorHConnection;
import org.apache.hadoop.hbase.client.HTableInterface;
import org.apache.hadoop.hbase.coprocessor.RegionCoprocessorEnvironment;
import org.apache.hadoop.hbase.ipc.RpcControllerFactory;
import org.apache.hadoop.hbase.ipc.controller.ServerRpcControllerFactory;
import org.apache.hadoop.hbase.regionserver.HRegionServer;
import org.apache.hadoop.hbase.regionserver.RegionServerServices;
import org.apache.phoenix.hbase.index.util.ImmutableBytesPtr;
import org.apache.phoenix.query.QueryServicesOptions;

public class CoprocessorHTableFactory implements HTableFactory {
    private Connection conn = null;

    /**
     * Create a short-circuit connection if we can with the specified RPC controller class.
     * @see CoprocessorHConnection#getConnectionForEnvironment(CoprocessorEnvironment)
     */
    public CoprocessorHTableFactory(CoprocessorEnvironment e) {
        try {
            Configuration conf = new Configuration(e.getConfiguration());

            // don't set retry times here, we'll retry in InfiniteRetryFailurePolicy
            if (e instanceof RegionCoprocessorEnvironment) {
                RegionCoprocessorEnvironment rce = (RegionCoprocessorEnvironment) e;
                RegionServerServices services = rce.getRegionServerServices();
                if (services instanceof HRegionServer) {
                    conn = new CoprocessorHConnection(conf, (HRegionServer) services);
                }
            }
            if (conn == null) {
                conn = ConnectionFactory.createConnection(conf);
            }
        } catch (IOException ioe) {
            throw new RuntimeException("Failed creating inter-RS connection", ioe);
        }
    }

    @Override
    public Connection getConnection() {
        return conn;
    }

    @Override
    public HTableInterface  getTable(ImmutableBytesPtr tableName) throws IOException {
        return (HTableInterface)conn.getTable(TableName.valueOf(tableName.copyBytesIfNecessary()));
    }
    
    @Override
    public HTableInterface getTable(ImmutableBytesPtr tableName,ExecutorService pool) throws IOException {
        return (HTableInterface)conn.getTable(TableName.valueOf(tableName.copyBytesIfNecessary()), pool);
    }

    @Override
    public void shutdown() {
        //
    }
}