/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.hbase.index.write;

import com.google.common.collect.Multimap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hadoop.hbase.client.*;
import org.apache.phoenix.hbase.index.table.HTableFactory;
import org.apache.hadoop.hbase.coprocessor.RegionCoprocessorEnvironment;
import org.apache.phoenix.hbase.index.table.HTableInterfaceReference;
import org.apache.phoenix.hbase.index.util.ImmutableBytesPtr;
import org.apache.phoenix.query.QueryServices;
import org.apache.phoenix.query.QueryServicesOptions;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class InfiniteRetryFailurePolicy implements IndexFailurePolicy {
    private static final Log LOG = LogFactory.getLog(InfiniteRetryFailurePolicy.class);
    public static long INDEX_RETRY_LOG_PAUSE = 5000;      // in ms

    private RegionCoprocessorEnvironment env;
    private HTableFactory factory;
    private long indexRetryPause;
    private Admin admin;
    private int maxRetryBeforeCheckTableExistence;

    public void setup(Stoppable parent, RegionCoprocessorEnvironment env) {
        this.env = env;
        Configuration conf = env.getConfiguration();
        indexRetryPause =
                conf.getLong(QueryServices.INDEX_UPDATE_RETRY_PAUSE,
                        QueryServicesOptions.DEFAULT_INDEX_UPDATE_RETRY_PAUSE);
        maxRetryBeforeCheckTableExistence =
                conf.getInt(QueryServices.INDEX_CHECK_TABLE_EXISTS_FREQUENCY,
                        QueryServicesOptions.DEFAULT_INDEX_CHECK_TABLE_EXISTS_FREQUENCY);
        factory = IndexWriterUtils.getDefaultDelegateHTableFactory(env);
        try {
            admin = factory.getConnection().getAdmin();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public void stop(String why) {
        try {
            admin.close();
        } catch (IOException e) {
            LOG.error("Closed admin failed!");
            e.printStackTrace();
        }
        LOG.debug(this.getClass().getName() + " is stopped!");
    }

    /**
     * @return True if {@link #stop(String)} has been closed.
     */
    public boolean isStopped() {
        return true;
    }

    /**
     * Handle the failure of the attempted index updates
     * retry the index mutation infinitely
     *
     * @param attempted map of index table -> mutations to apply
     * @param cause     reason why there was a failure
     * @throws IOException
     */
    public void handleFailure(Multimap<HTableInterfaceReference, Mutation> attempted,
            Exception cause) throws IOException {
        Set<Map.Entry<HTableInterfaceReference, Collection<Mutation>>>
                entries =
                attempted.asMap().entrySet();
        long start = System.currentTimeMillis();
        for (Map.Entry<HTableInterfaceReference, Collection<Mutation>> entry : entries) {
            List<Mutation> mutations = (List<Mutation>) entry.getValue();
            HTableInterfaceReference tableReference = entry.getKey();
            ImmutableBytesPtr tableName = tableReference.get();
            long retry = 0;

            // retry this index update forever until succeed
            while (true) {
                Table table = null;
                try {
                    table = factory.getTable(tableName);
                    table.batch(mutations);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Succeeded retry " + mutations.size() + " index updates to "
                                + tableReference.getTableName() + " after retry " + retry
                                + " times.");
                    }
                    break;
                } catch (TableNotFoundException e) {
                    // stop retrying index update when index table not found
                    LOG.info("Stop retrying index update when index table not found. tableName = "
                            + tableReference);
                    break;
                } catch (Throwable t) {
                    retry++;
                    // check table existence for every N retries, because various exceptions can be raised when table not exists.
                    if (retry % maxRetryBeforeCheckTableExistence == 0 && indexTableNotExists(
                            tableName)) {
                        LOG.info(
                                "Stop retrying index update when index table not found. tableName = "
                                        + tableReference);
                        break;
                    }
                    long now = System.currentTimeMillis();
                    if (now - start > INDEX_RETRY_LOG_PAUSE) {
                        LOG.error("Retry index updates to " + tableReference + " for " + retry
                                + " times.", t);
                        start = now;
                    }
                } finally {
                    if (table != null) {
                        try {
                            table.close();
                        } catch (IOException e) {
                            LOG.error("Failed closing table", e);
                        }
                    }
                }

                // sleep before next retry
                try {
                    Thread.sleep(indexRetryPause);
                } catch (Throwable t) {
                }
            } // end while dead loop
        } // end for loop
    }

    /**
     * @param tableName
     * @return if the specified table not exists, return true; otherwise, return true;
     * if error occurs, return true too
     */
    private boolean indexTableNotExists(ImmutableBytesPtr tableName) {
        boolean notExists = false;
        try {
            notExists = !admin.tableExists(TableName.valueOf(tableName.copyBytesIfNecessary()));
        } catch (Throwable t) {
            LOG.error("Failed check table existence " + tableName.toString(), t);
        } finally {
            if (admin != null) {
                try {
                    admin.close();
                } catch (Throwable t) {
                    LOG.error("Failed closing admin object", t);
                }
            }
        }
        return notExists;
    }
}
